local sq = squill._internal

-- I wanted to use squill to store its own data

-- Manually load in the schema table (CREATE TABLE etc depend on it and so they
-- can't be used)
sq.bootstrap_col_names = {}
sq.bootstrap_col_types = {}
local function import(table_name, t)
    sq.bootstrap_col_names[table_name] = t[1]
    sq.bootstrap_col_types[table_name] = t[2]

    if sq.storage:contains("squill/" .. table_name .. "/" .. t[1][1]) then
        -- Already initialised
        return
    end

    for i = 1, #t[1] do
        local col = {type = t[2][i], length = #t - 2}
        for row_idx = 3, #t do
            col[row_idx - 2] = t[row_idx][i]
        end
        sq.set_column("squill", table_name, t[1][i], col)
    end
end

import("schema", {
    -- luacheck: no max line length
    {"db_name",  "table",    "column",        "type",      "id",        "not_null",  "primary_key", "autoincrement", "check"},
    {sq.STRINGS, sq.STRINGS, sq.STRINGS,      sq.STRINGS,  sq.INTEGERS, sq.BOOLEANS, sq.BOOLEANS,   sq.INTEGERS,     sq.BLOBS},
    {"squill",   "schema",   "db_name",       sq.STRINGS,  1,           true,        false},
    {"squill",   "schema",   "table",         sq.STRINGS,  2,           true,        false},
    {"squill",   "schema",   "column",        sq.STRINGS,  3,           true,        false},
    {"squill",   "schema",   "type",          sq.STRINGS,  4,           true,        false},
    {"squill",   "schema",   "id",            sq.INTEGERS, 5,           true,        true},
    {"squill",   "schema",   "not_null",      sq.BOOLEANS, 6,           true,        false},
    {"squill",   "schema",   "primary_key",   sq.BOOLEANS, 7,           true,        false},
    {"squill",   "schema",   "autoincrement", sq.INTEGERS, 8,           false,       false},
    {"squill",   "schema",   "check",         sq.BLOBS,    9,           false,       false},

    {"squill",   "uniques",  "column_id",     sq.INTEGERS, 10,          true,        false},
    {"squill",   "uniques",  "unique_id",     sq.INTEGERS, 11,          true,        false},

    {"squill", "foreign_keys", "key_id",        sq.INTEGERS, 12, true, false},
    {"squill", "foreign_keys", "child_id",      sq.INTEGERS, 13, true, false},
    {"squill", "foreign_keys", "parent_table",  sq.STRINGS,  14, true, false},
    {"squill", "foreign_keys", "parent_column", sq.STRINGS,  15, true, false},
    {"squill", "foreign_keys", "sort_order",    sq.INTEGERS, 16, true, true, 2},

    -- Note: If adding new fields, make sure to figure out how to migrate
    -- existing databases
})

import("uniques", {
    -- unique_id is per-table
    {"column_id", "unique_id"},
    {sq.INTEGERS, sq.INTEGERS},

    -- schema: UNIQUE(id) (normally implicit as part of PRIMARY KEY)
    {5, 1},

    -- schema: UNIQUE(db_name, table, column)
    {1, 2}, {2, 2}, {3, 2},

    -- uniques: UNIQUE(column_id, unique_id)
    {10, 1}, {11, 1},

    -- foreign_keys: UNIQUE(key_id, child_id, parent_table, parent_column)
    {12, 1}, {13, 1}, {14, 1}, {15, 1},

    -- foreign_keys: UNIQUE(sort_order)
    {16, 2},
})

import("foreign_keys", {
    -- The parent table is referenced by name so that tables can be created out
    -- of order without errors
    {"key_id", "child_id", "parent_table", "parent_column", "sort_order"},
    {sq.INTEGERS, sq.INTEGERS, sq.STRINGS, sq.STRINGS, sq.INTEGERS},

    -- uniques: FOREIGN KEY (column_id) REFERENCES schema (id);
    {0, 10, "schema", "id", 0},

    -- foreign_keys: FOREIGN KEY (child_id) REFERENCES schema (id)
    {1, 13, "schema", "id", 1},
})

-- Because Squill depends on itself, it needs to be able to load itself before
-- compiling statements
local all_stmts = {}
local select_stmts = {}
function sq.bootstrap_statement(stmt, return_mode)
    local func
    local function prepare()
        func = squill.prepare_statement("squill", stmt, return_mode)
    end
    all_stmts[#all_stmts + 1] = prepare
    if stmt:find("^%s-SELECT") then
        select_stmts[#select_stmts + 1] = prepare
    end

    return function(...)
        assert(func, "Query called before bootstrap complete!")
        return func(...)
    end
end

function sq.bootstrap()
    -- Bootstrap the statements in a random order if mtt is enabled to make
    -- sure there aren't accidental interdependencies (which would make
    -- maintaining this mod harder)
    -- This is not done in production so that servers don't randomly crash
    -- on startup if one of these dependencies goes unnoticed
    if core.global_exists("mtt") and mtt.enabled then
        table.shuffle(select_stmts)
        table.shuffle(all_stmts)
    end

    -- SELECT statements are compiled first as they're used when compiling
    -- statements to get data out of the schema table etc.
    -- There are special cases when bootstrapping so that they can get compiled
    for _, func in ipairs(select_stmts) do
        func()
    end

    sq.bootstrap = nil
    sq.bootstrap_statement = nil
    sq.bootstrap_col_names = nil
    sq.bootstrap_col_types = nil

    -- Clear the cache to avoid using the current potentially suboptimal
    -- queries when recompiling
    squill.drop_query_cache()

    -- Compile all the built-in statements now that statements are able to get
    -- compiled. SELECTs get recompiled to possibly take advantage of
    -- optimisations that depend on being able to access the database.
    sq.allow_modifying_internal_db = true
    for _, func in ipairs(all_stmts) do
        func()
    end
    sq.allow_modifying_internal_db = nil

    -- Clear cache so that future attempts to alter the schema table error
    squill.drop_query_cache()
end
