-- LuaSQL compatibility layer
local find = string.find
local unpack = table.unpack or unpack

-- Needed by LuaSQL's unit tests
local mt_err = "LuaSQL: you're not allowed to get this metatable"

local Environment = {__metatable = mt_err}
Environment.__index = Environment

local Connection = {__metatable = mt_err}
Connection.__index = Connection

local Cursor = {__metatable = mt_err}
Cursor.__index = Cursor

function Environment:close()
    local ok = not self._closed
    self._closed = true
    return ok
end

function Environment:connect(db_name)
    assert(not self._closed)

    -- Catch invalid database name errors
    local ok, begin_stmt = pcall(squill.prepare_statement, db_name, "BEGIN")
    if not ok then return nil, begin_stmt end

    return setmetatable({
        _db_name = db_name,
        _begin = begin_stmt,
        _commit = squill.prepare_statement(db_name, "COMMIT"),
        _rollback = squill.prepare_statement(db_name, "ROLLBACK"),
        _autocommit = true,
    }, Connection)
end

function Connection:close()
    if self._closed then
        return false, "Connection already closed"
    end

    -- Force a rollback
    self:setautocommit(true)

    self._closed = true
    return true
end

function Connection:commit()
    local ok, err = pcall(self._commit)
    if not ok then return false, err end

    if not self._autocommit then
        ok, err = pcall(self._begin)
        if not ok then return false, err end
    end

    return true
end

function Connection:execute(stmt, ...)
    local all_rows, all_returned_cols = squill.exec(self._db_name, stmt, ...)
    if #all_returned_cols[1] == 0 then
        return all_rows[1].affected_rows or 0
    end

    return setmetatable({
        _rows = all_rows[1],
        _columns = all_returned_cols[1],
        _idx = 1,
    }, Cursor)
end

function Connection:rollback()
    local ok, err = pcall(self._rollback)
    if not ok then return false, err end

    if not self._autocommit then
        ok, err = pcall(self._begin)
        if not ok then return false, err end
    end

    return true
end

function Connection:setautocommit(autocommit)
    -- Per the LuaSQL documentation, this must be emulated by the driver if the
    -- underlying database doesn't support it
    assert(autocommit ~= nil)
    if autocommit then
        self._autocommit = true
        self:rollback()
    elseif self._autocommit then
        self._autocommit = false
        self._begin()
    end
    return true
end

function Connection:escape(str)
    return (string.gsub(str, "'", "''"))
end

function Cursor:close()
    if not self._idx then
        return false, "Cursor already closed"
    end
    self._idx = nil
    return true
end

local function get_values_list(self, row, dst_table)
    for i, col_name in ipairs(self._columns) do
        dst_table[i] = row[col_name]
    end
    return dst_table
end

function Cursor:fetch(dst_table, modestring)
    local row = self._rows[self._idx]
    if not row then
        self._idx = nil
        return nil
    end
    self._idx = self._idx + 1

    if not dst_table then
        return unpack(get_values_list(self, row, {}), 1, #self._columns)
    end

    if not modestring or find(modestring, "n", 1, true) then
        get_values_list(self, row, dst_table)
    end

    if modestring and find(modestring, "a", 1, true) then
        -- Iterate over _columns so null values are set to nil
        for _, col_name in ipairs(self._columns) do
            dst_table[col_name] = row[col_name]
        end
    end

    return dst_table
end

function Cursor:getcolnames()
    return self._columns
end

function Cursor:getcoltypes()
    -- No API exists for this in Squill
    error("getcoltypes() is not implemented")
end

function Cursor:numrows()
    return #self._rows
end

return {
    squill = function()
        return setmetatable({}, Environment)
    end
}
