# Squill

A subset of SQL for Luanti (formerly known as Minetest). This document assumes
that you are already familiar with SQL.

## Goals

 - No additional installation steps (mod security, external libraries, etc)
 - Reasonable performance for small-ish databases
 - The ability to use both Squill (for singleplayer and smaller servers) and
   SQLite (for large servers which go out of their way to set it up) without
   too much special code.
 - No hard limit on the amount of data stored (unlike `core.serialize()`),
   although storing a lot of data will be slow.
 - Faster startup time than running `core.parse_json()` on a giant file.

## Non-goals

 - Implement every SQL feature
 - Scalability - if you have very large tables, expect high memory usage and
   low performance.

## SQL support

Squill implements a small subset of SQL. I have tried to follow SQLite's syntax
where practical.

The following value types are supported:

 - `real`: Numbers
 - `integer`, `int`: Integers. Attempting to store floating-point values will
   cause an error.
 - `text`: Short strings. Not suitable for storing binary data.
 - `bool`, `boolean`: Booleans
 - `blob`: Blobs (strings) that can store binary data. Suitable for large
   amounts of data, but probably slow if used in a `WHERE` condition.

All tables are roughly the same as SQLite's strict tables - inserting the wrong
type into a column (if it can't be converted through type coercion) will result
in an error.

Notably, the following features are not implemented:

 - Recursive queries
 - Indexes
 - Foreign keys across multiple columns
 - Right and full joins
 - Nested joins/using brackets for joins
 - `ON DELETE`/`ON UPDATE`
 - `DEFAULT`
 - `UNION`
 - Storing multiple types in one column (like SQLite) - you can use the `blob`
   column type and [bluon], `core.write_json`, or `core.serialize` instead.
 - `SELECT 1 test WHERE test = 1` (PostgreSQL doesn't seem to support this
   either, but it works in SQLite)
 - Lots more

[bluon]: https://github.com/appgurueu/modlib/blob/master/doc/bluon.md#simple-example

### Transactions

Transactions are supported through the `BEGIN TRANSACTION`, `COMMIT`, and
`ROLLBACK` SQL commands. Only one transaction can happen at once in each
database.

Make sure you remember to `COMMIT` or `ROLLBACK`. To prevent accidental data
loss, Squill will crash the server if you forget.

A rollback is automatically done if there is a runtime error in a statement
(such as a constraint violation) to avoid leaving the database in an
inconsistent state. Note that parsing errors do not automatically roll back
transactions.

Statements that modify the schema can't be run in transactions.

## GUI

You must set `squill.enable_gui = true` in minetest.conf to enable the GUI.

You can try running queries on databases with the `/squill` command if you have
[flow](https://content.minetest.net/packages/luk3yx/flow) installed.

## Database names

Database names must either be in the form `modname:dbname` or `modname` to
avoid conflicts. Squill has its own internal database called `squill` that
should not be used (its format may change in the future).

## Connection API

If you're making a new mod and want to support both Squill and SQLite, you can
use this API to get a simple interface to both and to avoid having to deal with
mod security in your own mod.

First, you must obtain a database object with
`squill.connect(db_name[, supported_backends])`.

`supported_backends` is a list of backends (defaults to `{"squill"}`), for
example `{"sqlite", "squill"}` will allow both SQLite and Squill backends. Make
sure you test your mod with every database backend that you list here, since
different SQL implementations parse SQL slightly differently.

Unknown backends are ignored so that more backends can be added in the future.

The order of `supported_backends` is important - `{"squill", "sqlite"}` will
always use the Squill backend, except for existing worlds that already have a
SQLite database created for your mod.

This returns a database object with the following methods/values:

 - `db.dbms`: The DBMS in use, for example `"sqlite"` or `"squill"`.
 - `db:exec(stmt, ...)`: Executes `stmt`.
 - `db:prepare(stmt[, return_mode])`: Prepares `stmt` and returns a function
   that you can call to run the statement.
   [`return_mode` parameter documentation](#return-mode).
 - `db:blob(str)`: Returns a special object that can be passed to prepared
   statements to interpret the string as a `BLOB` instead of `TEXT` on SQLite.
   Returns the original string for backends that don't distinguish between
   `BLOB` and `TEXT`.

There is no API to close a connection - it is assumed that you will keep using
the same connection object until the server shuts down.

### Example

```lua
local db = squill.connect("mymod", {"sqlite", "squill"})
core.log("action", "[mymod] Connected to " .. db.dbms .. " database!")
db:exec([[
    CREATE TABLE IF NOT EXISTS my_table (
        x INTEGER PRIMARY KEY,
        y TEXT
    ) STRICT;
    INSERT INTO my_table (y) VALUES (current_timestamp);
]])

local stmt = db:prepare("SELECT * FROM my_table WHERE y LIKE ?", squill.RETURN_FIRST_ROW)
local row = stmt("20%")
print(row.x, row.y)
```

### Database backends

<details>
<summary><b>squill</b></summary>

Built into this mod, no setup required.

</details>

<details>
<summary><b>sqlite</b></summary>

Squill has a SQLite3 backend that uses LuaJIT's FFI library. Note that this
backend has not been tested much and likely has many bugs.

**Usage**

Provided you are on a Linux system and are using LuaJIT, you should just be
able to add `squill.enable_sqlite_ffi_backend = true` and
`secure.trusted_mods = squill` to minetest.conf.

**Notes**

 - I've only tested this backend on Linux.
 - LuaJIT is needed as its [FFI library](https://luajit.org/ext_ffi.html) is
   used.
 - SQLite doesn't have a separate boolean type, so booleans will be represented
   as either `1` or `0`.
 - All strings passed to prepared statements are bound as `TEXT`. You can do
   `db:blob(string)` to get a special object that will be bound as a BLOB.
 - This will not work if Luanti is statically linked, such as in the Flatpak
   package.
 - Databases are stored in the world folder, for example `mymod:my_db` would be
   stored as `worldpath/squill-sqlite/mymod-my_db.sqlite`.
 - Squill attempts to provide a secure API that does not allow other mods to
   access arbitrary files.
    - The `ATTACH` and `VACCUM INTO` commands are blocked.
 - Parameter binding with `$1`, `$2`, etc should work correctly.

</details>

## Database API wrappers

If you have a mod that's already using a real SQL database and want to add a
fallback to Squill if your mod can't access the insecure environment, or if you
just like one of these APIs better, you can use one of the below API wrappers.

<details>
<summary><b>lsqlite3</b></summary>

To make it easier to use both Squill and lsqlite3 with the same code, only the
file name is used out of the database path if the argument looks like a path.
For example, `sqlite3.open("/path/to/mymod.sqlite")` will open the database
`"mymod"`.

Any boolean values that get returned are converted to integers to match
SQLite's behaviour.

Example code that tries to use lsqlite3 and falls back to Squill[^1]:

```lua
local ie = core.request_insecure_environment()
local sqlite
if ie then
    -- The mod is in secure.trusted_mods, use lsqlite3
    sqlite = ie.require("lsqlite3")

    -- Don't leak the sqlite3 global variable
    sqlite3 = nil
elseif core.global_exists("squill") then
    -- The mod is not in secure.trusted_mods, fall back to using Squill
    sqlite = squill.compat.lsqlite3
else
    error("Please add this mod to secure.trusted_mods or install Squill.")
end

-- (Mod code here)
local db = sqlite.open(core.get_worldpath() .. "/mymod.sqlite")

local function check(retval)
    if retval ~= sqlite.OK then
        error(db:errmsg(), 2)
    end
end

check(db:exec([[
    CREATE TABLE IF NOT EXISTS my_table (
        a INTEGER PRIMARY KEY,
        b TEXT,
        c TEXT
    ) STRICT;

    INSERT INTO my_table (b, c) VALUES ('Hello world!', 'test');
]]))

local stmt = db:prepare("SELECT a, b FROM my_table where c = ?")
if not stmt then
    error(db:errmsg())
end
check(stmt:bind_values("test"))

for a, b in stmt:urows() do
    print(a, b)
end

check(stmt:finalize())

db:close()
```

Some more advanced functions from lsqlite3 aren't implemented, such as hooks.

Note that this API wrapper won't work with anything that depends on SQLite's
ability to store values of multiple types in columns.


[^1]: You will need to add `squill` to `optional_depends` in your mod.conf file
      (or maybe just to `depends` if you want to make ContentDB installs easier
      and don't mind requiring it even if it isn't being used). The code is
      adapted from the lsqlite3 example on the [Luanti Modding Book].

[Luanti Modding Book]: https://rubenwardy.com/minetest_modding_book/en/map/storage.html

</details>

<details>
<summary><b>LuaSQL</b></summary>

Squill provides an API that mimics a LuaSQL driver.

Example (loading real LuaSQL drivers is not shown):

```lua
local driver = squill.compat.luasql
local env = driver.squill()
local conn = assert(env:connect("mymod"))

local cur = assert(conn:execute("SELECT a, b FROM my_table WHERE c = 'test'"))
print(cur:fetch())

cur:close()
conn:close()
env:close()
```

See [LuaSQL's manual](https://lunarmodules.github.io/luasql/manual.html) for
more usage instructions. The `conn:escape(str)`, `cur:numrows()`, and
[parameter binding](https://github.com/lunarmodules/luasql/pull/128) extensions
are supported.

If you can (i.e. if other drivers you want to support also support it), you
should use parameter binding so that Squill doesn't have to re-parse statements
every time you run them.

</details>

<details>
<summary><b>pgmoon</b></summary>

Example (loading pgmoon itself is not covered here):

```lua
local pgmoon = squill.compat.pgmoon

-- Unsupported options such as "host", "port", and "username" are ignored
local pg = pgmoon.connect({database = "mymod"})
assert(pg:connect())

local rows = assert(pg:query("SELECT a, b FROM my_table WHERE c = $1", "test"))
print(dump(rows))

assert(pg:disconnect())
```

Note that you will probably need Squill-specific code for `CREATE TABLE` to use
Squill's types.

</details>

## Non-object-oriented API

The below API functions take a database name as a parameter.

### `squill.exec(db_name, sql)`

You can use `squill.exec` to run one or more SQL statements.

```lua
-- You can use squill.exec for debugging and to create tables
squill.exec("mymod:mydb", [[
    CREATE TABLE IF NOT EXISTS my_table (
        key text,
        other_key text
    );
    INSERT INTO my_table (key, other_key) VALUES ('hello', 'world');
]])
```

You can pass in parameters with `?` or `$1`, `$2`, etc:

```lua
squill.exec("mymod:mydb", "SELECT $1 * $2", 5, 2)
```

Single statements will be cached for future calls, and removed from the cache
if they're not used for a while. Note that only single statements are cached
at the moment, so you should try and avoid using multiple statements with
`squill.exec` in performance-sensitive code.

Two values are returned by `squill.exec`: The first value is a list of
`squill.RETURN_ALL_ROWS` values for each statement in the code, and the second
is a list of returned column lists (see `squill.prepare_statement`'s
documentation)

### `squill.prepare_statement(db_name, sql[, return_mode])`

`squill.prepare_statement` compiles the SQL statement and returns a function
that you can call to run the statement.

Creating or altering the database schema (e.g. `CREATE TABLE`) will invalidate
any existing prepared statements, make sure any schema changes are done at load
time.

Example:

```lua
local get_keys_except = squill.prepare_statement("mymod:mydb", [[
    SELECT key, other_key FROM my_table WHERE key <> ?
]])

for _, row in ipairs(get_keys_except("hello")) do
    print(row.key, row.other_key)
end
```

Prepared statements are cached in case `squill.prepare_statement` or
`squill.exec` is called with exactly the same string.

There's a second return value to `squill.prepare_statement` which indicates what
columns the SQL statement will return, and is something like `{"col1", "col2"}`
or `{affected_rows = true}`. Usually this value should just get ignored, as done
in the example. You must not modify the second return value as it is cached.

## `return_mode` parameter

`return_mode` adjusts what the function will return, and can be one of:

 - `squill.RETURN_ALL_ROWS` (default)
    - Returns all rows.
    - For example, `SELECT * FROM my_table ORDER BY key` would return
      `{{key = "key1", other_key = "data"}, {key = "key2", other_key = "more_data"}}`.
    - `UPDATE` and `DELETE` statements will return `{affected_rows = rows}`.

 - `squill.RETURN_FIRST_ROW`
    - Returns the first row, for example `{key = "key1", other_key = "data"}`.
    - Returns nil if no rows are returned by the query.

 - `squill.RETURN_SINGLE_COLUMN`
    - Returns the first column, for example
      `SELECT key FROM my_table ORDER BY key` would return `{"key1", "key2"}`.
    - Errors if the statement does not return exactly one column.

 - `squill.RETURN_SINGLE_VALUE`
    - Returns a single value from the first row, for example
      `SELECT key FROM my_table ORDER BY key` would return `"key1"`.
    - Returns `nil` if no rows are returned.
    - Errors if the statement does not return exactly one column.

## Making SQL backups of a database

You can use `squill.dump(db_name, file_object)` to export a backup of a
database. This backup can be imported on a new server with `squill.exec` (or
maybe it could be used to migrate to SQLite). `file_object` can be anything
that supports `:write()`, for example a [modlib.table.rope] object.

You should probably either do this or back up your mod storage database
regularly in case some bug corrupts data.

[modlib.table.rope]: https://github.com/appgurueu/modlib/blob/f6de802d6f50cab6eee1f0d760a7f2f737d81a42/table.lua#L84-L95

## Performance tips

 - Use `squill.prepare_statement` (or the equivalent in the database API
   wrapper you are using) and compile every statement you need at load time.
    - If you don't prepare statements ahead of time, Squill will try and do
      some caching, but queries not run at least once per minute will be
      re-parsed each time.
 - Use LuaJIT.
 - Use query parameters instead of escaping where possible, even if you're not
   preparing statements ahead of time.
 - SELECTs/JOINs on non-null values in singular `UNIQUE`/`PRIMARY KEY` columns
   are fast since they can just do a O(1) lookup if a cached index is in memory
    - Note: This doesn't work all the time, notably with `or` conditions, `in`,
      or with non-trivial `ON` clauses in LEFT JOINs.
 - Transactions can speed up bulk inserts and updates, as Squill will only
   write everything to mod storage after `COMMIT`, instead of after each
   statement.
 - Avoid `SELECT *` and only request the columns that you actually need.
 - Use `blob` for large data so that it doesn't get loaded into RAM all the
   time.
 - You can enable the `squill.aggressive_caching` setting to make Squill keep
   caches in memory for longer. I'm not sure how much this will improve
   performance by.
