-- Simple code to extract interesting values from JSON while parsing as little
-- JSON as possible.

local assert = assert
local byte, find, sub = string.byte, string.find, string.sub

local DOLLARS, LBRACKET, RBRACKET, QUOTE, BACKSLASH, LBRACE, RBRACE,
    ZERO, NINE, DOT, T, F, N, HYPHEN = byte('$[]"\\{}09.tfn-', 1, -1)

local function find_string_end(json, quote_idx)
    local idx = quote_idx + 1
    local has_escapes = false
    repeat
        local chr = byte(json, idx)
        if chr == BACKSLASH then
            -- Skip escaped quotes
            idx = idx + 1
            has_escapes = true
        elseif chr == QUOTE then
            -- Found end quote
            return idx, has_escapes
        end
        idx = idx + 1
    until chr == nil
end

local find_value_end
local function object_iter(json, idx)
    -- End of object
    if byte(json, idx) == RBRACE then return end

    local _, key_start = find(json, "^%s-[\"%}]", idx + 1)
    assert(key_start, "Invalid JSON")

    -- Empty object
    if byte(json, key_start) == RBRACE then return key_start end
    local key_end, has_escapes = find_string_end(json, key_start)


    local _, value_start = find(json, "^%s-:%s-%S", key_end + 1)
    assert(value_start, "Invalid JSON")

    local value_end = find_value_end(json, value_start)

    local _, separator_idx = find(json, "^%s-[,}]", value_end + 1)
    assert(separator_idx, "Invalid JSON")
    return separator_idx, key_start, key_end, has_escapes, value_start, value_end
end

local function array_iter(json, idx)
    if byte(json, idx) == RBRACKET then return end

    local value_start = find(json, "%S", idx + 1)
    assert(value_start, "Invalid JSON")
    if byte(json, value_start) == RBRACKET then return value_start end

    local value_end = find_value_end(json, value_start)

    local _, separator_idx = find(json, "^%s-[,%]]", value_end + 1)
    assert(separator_idx, "Invalid JSON")
    return separator_idx, value_start, value_end
end

function find_value_end(json, idx)
    local chr = byte(json, idx)
    if chr == QUOTE then
        return find_string_end(json, idx)
    elseif chr == LBRACE or chr == LBRACKET then
        local iter_func = chr == LBRACE and object_iter or array_iter
        for sep_idx in iter_func, json, idx do
            idx = sep_idx
        end
        return idx
    elseif (chr >= ZERO and chr <= NINE) or chr == DOT or chr == HYPHEN then
        -- Numbers
        local _, end_idx = find(json, "[0-9%.e%+%-]+", idx)
        assert(end_idx)
        return end_idx
    elseif chr == T then
        assert(sub(json, idx, idx + 3) == "true")
        return idx + 3
    elseif chr == F then
        assert(sub(json, idx, idx + 4) == "false")
        return idx + 4
    elseif chr == N then
        assert(sub(json, idx, idx + 3) == "null")
        return idx + 3
    end
    error("Invalid JSON: " .. sub(json, idx))
end

local function json_get(json, query)
    if json == nil then return end

    local key_type = type(query)
    if key_type == "string" then
        if query == "$" then
            return json
        end

        assert(byte(query, 1) ~= DOLLARS, "Unsupported: MySQL-style queries")
        local _, idx = find(json, "^%s-{")
        assert(idx, "Not a JSON object")
        for _, key_start, key_end, has_escapes, value_start, value_end in
                object_iter, json, idx do
            if key_start then
                local key
                if has_escapes then
                    -- Using core.parse_json is easier than trying to parse escapes
                    key = core.parse_json(sub(json, key_start, key_end))
                else
                    key = sub(json, key_start + 1, key_end - 1)
                end

                if key == query then
                    return sub(json, value_start, value_end)
                end
            end
        end
    elseif key_type == "number" then
        local array_idx = 0
        local _, idx = find(json, "^%s-%[")
        assert(idx, "Not a JSON array")
        for _, value_start, value_end in array_iter, json, idx do
            if value_start then
                if query == array_idx then
                    return sub(json, value_start, value_end)
                end
                array_idx = array_idx + 1
            end
        end
    else
        error("Invalid JSON query")
    end
end

local function json_get_and_parse(json, key)
    local value = json_get(json, key)
    if value then
        local res = core.parse_json(value)
        assert(type(res) ~= "table", "Not a plain JSON value")
        return res
    end
end

return json_get, json_get_and_parse
