local sq = squill._internal
local format = string.format
local valid_identifier = sq.valid_identifier

local table_exists_stmt = sq.bootstrap_statement([[
    SELECT COUNT(*) FROM schema WHERE db_name = ? AND table = ?
]], squill.RETURN_SINGLE_VALUE)

local get_column_id = sq.bootstrap_statement([[
    SELECT id FROM schema WHERE db_name = ? AND table = ? AND column = ?
]], squill.RETURN_SINGLE_VALUE)

local get_columns = sq.bootstrap_statement([[
    SELECT column AS name, type FROM schema WHERE db_name = ? AND table = ?
]])

local rename_table_schema = sq.bootstrap_statement([[
    UPDATE schema SET table = $3 WHERE db_name = $1 AND table = $2
]])

local get_max_column_id = sq.bootstrap_statement([[
    SELECT MAX(id) FROM schema
]], squill.RETURN_SINGLE_VALUE)

local add_column = sq.bootstrap_statement([[
    INSERT INTO schema (db_name, table, column, type, id, not_null, primary_key)
    VALUES (?, ?, ?, ?, ?, false, false)
]])

local delete_column_stmt = sq.bootstrap_statement([[
    DELETE FROM schema WHERE db_name = ? AND table = ? AND column = ?
]])

local delete_column_uniques_stmt = sq.bootstrap_statement([[
    DELETE FROM uniques WHERE column_id = ?
    RETURNING unique_id
]], squill.RETURN_SINGLE_COLUMN)

local delete_foreign_keys_stmt = sq.bootstrap_statement([[
    DELETE FROM foreign_keys WHERE child_id = ?
]])

local get_other_uniques = sq.bootstrap_statement([[
    SELECT COUNT(*) FROM uniques u
    JOIN schema s ON u.column_id = s.id
    WHERE db_name = ? AND table = ? AND unique_id = ?
]], squill.RETURN_SINGLE_VALUE)

return function(self)
    self:expect("table")

    local if_not_exists = self:pop_if_equals("if")
    if if_not_exists then
        self:expect("not")
        self:expect("exists")
    end

    local table_name = self:next()
    self:assert(valid_identifier(table_name), "%q is not a valid table name", table_name)
    self:assert(table_name ~= "squill", "The 'squill' database is read-only")
    local db_name = self.db_name

    local cmd = self:next()
    if cmd == "rename" then
        self:expect("to")

        local new_name = self:next()
        self:assert(valid_identifier(new_name), "%q is not a valid table name", new_name)

        return function()
            sq.assert_no_transaction(db_name)

            if table_exists_stmt(db_name, table_name) == 0 then
                error(format("Table %q does not exist", table_name), 2)
            elseif table_exists_stmt(db_name, new_name) > 0 then
                error(format("Table %q already exists", new_name), 2)
            end

            -- Since Squill does not support multithreading, there's no need to
            -- start the transaction before table_exists_stmt
            sq.begin_transaction("squill")
            sq.begin_transaction(db_name)

            -- This isn't efficient, but it's probably good enough because
            -- table renaming won't happen often, and when it does the table
            -- probably just been copied anyway.
            for _, c in ipairs(get_columns(db_name, table_name)) do
                local column = sq.get_column(db_name, table_name, c.name, c.type, false)

                -- Let the blob metatable know that it should save every single
                -- item, no-op for other types
                for i = 1, column.length do
                    column[i] = column[i]
                end

                sq.set_column(db_name, new_name, c.name, column)
                sq.set_column(db_name, table_name, c.name, {
                    type = column.type, length = 0, _old_length = column.length
                })
            end

            rename_table_schema(db_name, table_name, new_name)
            sq.schema_vers[db_name] = (sq.schema_vers[db_name] or 0) + 1

            sq.commit_transaction(db_name)
            sq.commit_transaction("squill")

            return {}
        end
    elseif cmd == "add" then
        self:expect("column")

        local col_name = self:next()
        self:assert(valid_identifier(col_name), "%q is not a valid column name", col_name)

        local col_type = self:next()
        local internal_type = self:assert(sq.known_column_types[col_type],
            "Unknown column type: %q", col_type)

        return function()
            sq.assert_no_transaction(db_name)

            self:assert(get_column_id(db_name, table_name, col_name) == nil,
                "Column %q already exists", col_name)

            sq.begin_transaction("squill")
            sq.begin_transaction(db_name)

            local total_rows = squill.exec(
                db_name, "SELECT COUNT(*) FROM " .. table_name
            )[1][1]["?column?"]

            sq.set_column(db_name, table_name, col_name,
                {type = internal_type, length = total_rows})
            add_column(db_name, table_name, col_name, internal_type,
                get_max_column_id() + 1)

            sq.schema_vers[db_name] = (sq.schema_vers[db_name] or 0) + 1
            sq.commit_transaction(db_name)
            sq.commit_transaction("squill")

            return {}
        end
    elseif cmd == "drop" then
        self:expect("column")

        local col_name = self:next()
        self:assert(valid_identifier(col_name), "%q is not a valid column name", col_name)

        return function()
            sq.assert_no_transaction(db_name)

            local column_id = get_column_id(db_name, table_name, col_name)
            assert(column_id, "Column %q does not exist", col_name)

            sq.begin_transaction("squill")
            sq.begin_transaction(db_name)

            local column = sq.get_column(db_name, table_name, col_name, false)
            sq.set_column(db_name, table_name, col_name, {
                type = column.type, length = 0, _old_length = column.length
            })
            delete_foreign_keys_stmt(column_id)
            local removed_uniques = delete_column_uniques_stmt(column_id)
            for _, unique_id in ipairs(removed_uniques) do
                if get_other_uniques(db_name, table_name, unique_id) ~= 0 then
                    sq.rollback_transaction(db_name)
                    sq.rollback_transaction("squill")
                    error("Removing columns with multi-column UNIQUE " ..
                        "constraints is not supported.")
                end
            end

            delete_column_stmt(db_name, table_name, col_name)

            -- The second argument (true) forces the cache for modified columns
            -- to be cleared
            sq.schema_vers[db_name] = (sq.schema_vers[db_name] or 0) + 1
            sq.commit_transaction(db_name, true)
            sq.commit_transaction("squill")
        end
    else
        self:error("Command not implemented: " .. cmd)
    end
end
