-- Global environment for compiled SQL statements used at runtime

local abs = math.abs
local format = string.format
local select = select
local type = type

local sq = squill._internal
local operators = sq.operators
local coerce_to_boolean = sq.coerce_to_boolean
local coerce_to_number = sq.coerce_to_number
local coerce_to_string = sq.coerce_to_string
local schema_vers = sq.schema_vers

local current_db_name
local function make_coerce_helper(coerce)
    return function(value, qualname)
        if value == nil then return nil end

        local res = coerce(value)
        if res == nil then
            sq.rollback_transaction(current_db_name, true)
            error(format("Cannot store value %s (of type %s) in column %s",
                dump(value), type(value), qualname), 3)
        end

        return res
    end
end

local str_coerce_or_error = make_coerce_helper(coerce_to_string)
sq.stmt_env = setmetatable({
    OPS = operators,

    -- Make assert and error automatically roll back transactions
    assert = function(value, msg)
        if not value then
            sq.rollback_transaction(current_db_name, true)
            error(msg, 3)
        end
        return value
    end,
    error = function(msg)
        sq.rollback_transaction(current_db_name, true)
        error(msg, 3)
    end,

    -- ipairs is not included to prevent its accidental usage over column.length
    -- (since ipairs cannot handle null values)
    pairs = pairs,
    table_sort = table.sort,
    get_column = sq.get_column,
    set_column = sq.set_column,
    get_unique_index = sq.get_unique_index,
    begin_stmt = function(db_name, ver)
        if (schema_vers[db_name] or 0) ~= ver then
            sq.rollback_transaction(db_name, true)
            error("Prepared statement is no longer valid, the database " ..
                "schema has been modified since it was created.", 3)
        end

        -- Store current database name for automatic rollbacks
        current_db_name = db_name
    end,
    coerce_to_string = coerce_to_string,
    coerce_or_error = {
        [sq.NUMBERS] = make_coerce_helper(coerce_to_number),
        [sq.INTEGERS] = make_coerce_helper(function(n)
            n = coerce_to_number(n)
            if n and n % 1 == 0 and abs(n) < 2^53 then
                return n
            end
        end),
        [sq.BOOLEANS] = make_coerce_helper(coerce_to_boolean),
        [sq.STRINGS] = str_coerce_or_error,
        [sq.BLOBS] = str_coerce_or_error,
    },
    autoincrement_set_min = sq.bootstrap_statement([[
        UPDATE schema SET autoincrement = $4 + 1
        WHERE db_name = $1 AND table = $2 AND column = $3 AND autoincrement <= $4
    ]]),

    str_upper = function(s)
        s = coerce_to_string(s)
        return s and s:upper()
    end,
    str_lower = function(s)
        s = coerce_to_string(s)
        return s and s:lower()
    end,
    coalesce = function(...)
        for i = 1, select("#", ...) do
            local arg = select(i, ...)
            if arg ~= nil then
                return arg
            end
        end
        return nil
    end,

    -- Wrap math functions to make the null behaviour match SQL
    math = setmetatable({}, {
        __index = function(sq_math, k)
            -- TODO: Does this need to be optimised?
            local value = math[k]
            if type(value) == "function" then
                local real_func = value
                function value(...)
                    for i = 1, select("#", ...) do
                        local arg = select(i, ...)
                        if arg == nil then
                            return nil
                        elseif type(arg) ~= "number" then
                            -- SQLite lets you use min/max with strings
                            error(
                                "Type coercion is not supported for math " ..
                                "functions"
                            )
                        end
                    end
                    return real_func(...)
                end
            end

            -- Cache so that the next access doesn't go to __index
            sq_math[k] = value
            return value
        end,
    }),
    os_date = os.date,

    autoincrement_helper = sq.bootstrap_statement([[
        UPDATE schema SET autoincrement = autoincrement + 1
        WHERE id = ?
        RETURNING autoincrement - 1
    ]], squill.RETURN_SINGLE_VALUE),
}, {
    __index = function(_, k)
        error(format(
            "Attempt to get undefined variable %q in Squill runtime env", k
        ))
    end,
    __newindex = function(_, k)
        error(format(
            "Attempt to create global variable %q in Squill runtime env", k
        ))
    end
})
