local find, format = string.find, string.format

local modpath = core.get_modpath("squill")

local sq = squill._internal

local operators = {}

operators["="] = function(a, b)
    if a == nil or b == nil then return nil end
    return a == b
end

operators["<>"] = function(a, b)
    if a == nil or b == nil then return nil end
    return a ~= b
end

local function coerce_to_comparable(n)
    if n == true then
        return 1
    elseif n == false then
        return 0
    else
        return n
    end
end

operators[">"] = function(a, b)
    a, b = coerce_to_comparable(a), coerce_to_comparable(b)
    return a and b and a > b
end

operators["<"] = function(a, b)
    a, b = coerce_to_comparable(a), coerce_to_comparable(b)
    return a and b and a < b
end

operators[">="] = function(a, b)
    a, b = coerce_to_comparable(a), coerce_to_comparable(b)
    return a and b and a >= b
end

operators["<="] = function(a, b)
    a, b = coerce_to_comparable(a), coerce_to_comparable(b)
    return a and b and a <= b
end

operators["between"] = function(a, b, c)
    a, b, c = coerce_to_comparable(a), coerce_to_comparable(b), coerce_to_comparable(c)
    return a and b and a >= b and c and a <= c
end

local function coerce_to_number(n)
    return tonumber(coerce_to_comparable(n))
end

operators["unary +"] = coerce_to_number
operators["unary -"] = function(n)
    n = coerce_to_number(n)
    return n and -n
end
operators["^"] = function(a, b)
    a, b = coerce_to_number(a), coerce_to_number(b)
    return a and b and a ^ b
end
operators["+"] = function(a, b)
    a, b = coerce_to_number(a), coerce_to_number(b)
    return a and b and a + b
end
operators["-"] = function(a, b)
    a, b = coerce_to_number(a), coerce_to_number(b)
    return a and b and a - b
end
operators["*"] = function(a, b)
    a, b = coerce_to_number(a), coerce_to_number(b)
    return a and b and a * b
end
operators["/"] = function(a, b)
    a, b = coerce_to_number(a), coerce_to_number(b)
    return a and b and a / b
end
operators["%"] = function(a, b)
    a, b = coerce_to_number(a), coerce_to_number(b)
    return a and b and a % b
end

local function coerce_to_string(value)
    local value_type = type(value)
    if value_type == "string" then
        return value
    elseif value_type == "number" then
        return format("%.17g", value)
    elseif value_type == "boolean" then
        return value and "TRUE" or "FALSE"
    end
end

local function compile_like_pattern(pattern, glob, already_compiled)
    if already_compiled then return pattern end

    local match_many, match_one = "%", "_"
    if glob then
        match_many, match_one = "*", "?"
    end

    local in_brackets = false
    pattern = pattern:gsub("[%(%)%.%%%+%-%*%?%[%^%$_%]]", function(chr)
        if in_brackets then
            -- When inside brackets, wildcards don't get evaluated
            if chr == "]" then
                in_brackets = false
                return chr
            elseif chr == "-" then
                return chr
            end
        elseif chr == match_many then
            return ".-"
        elseif chr == match_one then
            return "."
        elseif glob and chr == "[" then
            in_brackets = true
            return chr
        end

        return "%" .. chr
    end)

    assert(not in_brackets, "Mismatched brackets in GLOB pattern")

    if not glob then
        pattern = pattern:gsub("%a", function(chr)
            return format("[%s%s]", chr:upper(), chr:lower())
        end)
    end

    pattern = pattern:sub(1, 2) == ".-" and pattern:sub(3) or "^" .. pattern
    pattern = pattern:sub(-2) == ".-" and pattern:sub(1, -3) or pattern .. "$"

    return pattern
end
sq.compile_like_pattern = compile_like_pattern

for _, op in ipairs({"like", "glob"}) do
    local glob = op == "glob"

    operators[op] = function(str, pattern, compiled)
        str, pattern = coerce_to_string(str), coerce_to_string(pattern)
        if str == nil or pattern == nil then return nil end
        return find(str, compile_like_pattern(pattern, glob, compiled)) ~= nil
    end

    operators["not " .. op] = function(str, pattern, compiled)
        str, pattern = coerce_to_string(str), coerce_to_string(pattern)
        if str == nil or pattern == nil then return nil end
        return find(str, compile_like_pattern(pattern, glob, compiled)) == nil
    end
end

operators["||"] = function(a, b)
    a, b = coerce_to_string(a), coerce_to_string(b)
    return a and b and a .. b
end

local boolean_vals = {
    [true] = true, ["true"] = true, yes = true, on = true, ["1"] = true, [1] = true,
    [false] = false, ["false"] = false, no = false, off = false, ["0"] = false, [0] = false,
}

local function coerce_to_boolean(v)
    local bool = boolean_vals[v]
    if bool == nil and type(v) == "number" then
        bool = true
    end
    return bool
end

operators["to_boolean"] = coerce_to_boolean

operators["unary not"] = function(v)
    local bool = coerce_to_boolean(v)
    if bool == nil then return nil end
    return not bool
end

-- AND and OR is implemented with two helper functions so that "NULL and FALSE"
-- returns FALSE and not NULL, but in a way that we can keep short circuiting
-- where possible.
local and_is_null
operators["and_left"] = function(v)
    local bool = coerce_to_boolean(v)
    and_is_null = bool == nil

    -- Return true if null was returned so that the right-hand side of the
    -- expression gets evaluated.
    return and_is_null or bool
end

operators["and_right"] = function(v)
    local bool = coerce_to_boolean(v)
    -- Deal with "NULL and FALSE"
    if and_is_null and bool then
        return nil
    end
    return bool
end

-- So "NULL or FALSE" returns NULL instead of FALSE
local or_is_null
operators["or_left"] = function(v)
    local bool = coerce_to_boolean(v)
    or_is_null = bool == nil
    return bool
end

operators["or_right"] = function(v)
    local bool = coerce_to_boolean(v)
    if or_is_null and not bool then
        return nil
    end
    return bool
end

local function in_operator(value, inverse, ...)
    if value == nil then return nil end

    local res = inverse
    for i = 1, select("#", ...) do
        local arg = select(i, ...)
        if arg == nil then
            res = nil
        elseif arg == value then
            return not inverse
        end
    end
    return res
end

operators["in"] = function(value, ...)
    return in_operator(value, false, ...)
end

operators["not in"] = function(value, ...)
    return in_operator(value, true, ...)
end

operators["->"], operators["->>"] = dofile(modpath .. "/json.lua")

sq.coerce_to_boolean = coerce_to_boolean
sq.coerce_to_number = coerce_to_number
sq.coerce_to_string = coerce_to_string
sq.operators = operators
