-- pgmoon compatibility layer

local unpack = table.unpack or unpack

local pgmoon = {}
local pg = {}
pg.__index = pg

pg.NULL = setmetatable({"NULL"}, {__tostring = function() return "NULL" end})

function pgmoon.new(options)
    return setmetatable({
        _db_name = assert(options.database),
        _convert_null = options.convert_null,
    }, pg)
end

function pg:connect()
    self._connected = true
    return true
end

function pg:disconnect()
    self._connected = false
    return true
end

function pg:settimeout() end

local function format_result(self, res, returned_columns)
    if #returned_columns == 0 and not returned_columns.affected_rows then
        -- pgmoon returns true for statements like CREATE TABLE and DROP TABLE
        return true
    end

    if self._convert_null then
        for _, col in ipairs(returned_columns) do
            for _, row in ipairs(res) do
                if row[col] == nil then
                    row[col] = self.NULL
                end
            end
        end
    end

    return res
end

function pg:query(query_string, ...)
    assert(self._connected, "Not connected")

    -- Simple query protocol
    local argc = select("#", ...)
    if argc == 0 then
        local ok, all_rows, all_cols = pcall(squill.exec, self._db_name,
            query_string)
        if not ok then
            return nil, all_rows
        end

        -- Translate result into pgmoon's expected format
        local num_queries = #all_cols
        for i = 1, num_queries do
            all_rows[i] = format_result(self, all_rows[i], all_cols[i])
        end

        -- Return row directly if there's only one query
        if num_queries == 1 then
            all_rows = all_rows[1]
        end

        return all_rows, num_queries, nil, nil
    end

    -- Convert pgmoon.NULL back into nil
    local argv = {...}
    for i = 1, argc do
        if argv[i] == self.NULL then
            argv[i] = nil
        end
    end

    -- Extended query protocol
    local prepare_ok, stmt, returned_columns = pcall(
        squill.prepare_statement, self._db_name, query_string
    )
    if not prepare_ok then
        return nil, stmt
    end
    local ok, res = pcall(stmt, unpack(argv, 1, argc))
    if not ok then
        return nil, res
    end

    return format_result(self, res, returned_columns), 1, nil, nil
end

function pg:escape_literal(val)
    if val == self.NULL then
        val = nil
    end
    return squill.dump_value(val)
end

function pg:escape_identifier(val)
    assert(string.find(val, "^[a-z_][a-z0-9_]*$"),
        "Squill does not support identifiers with special charcaters")
    return val
end

return pgmoon
