local function objs_eq(t1, t2)
    if type(t1) ~= "table" or type(t2) ~= "table" then return t1 == t2 end

    for k, v in pairs(t1) do
        if not objs_eq(v, t2[k]) then
            return false
        end
    end

    for k in pairs(t2) do
        if t1[k] == nil then
            return false
        end
    end

    return true
end

local function assert_eq(t1, t2)
    if not objs_eq(t1, t2) then
        error("Objects are not equal! " .. dump(t1) .. " ~= " .. dump(t2), 2)
    end
end


-- Clean up old test databases
local get_test_dbs = squill.prepare_statement("squill", [[
    SELECT DISTINCT db_name, table FROM schema
    WHERE db_name LIKE 'squill:test_%'

    -- Sort tables for foreign key tests
    ORDER BY table DESC
]])
local function cleanup()
    for _, row in ipairs(get_test_dbs()) do
        squill.exec(row.db_name, "DROP TABLE " .. row.table)
    end
end
cleanup()


local get_id_offset = squill.prepare_statement("squill", [[
    SELECT MAX(id) FROM schema
]], squill.RETURN_SINGLE_VALUE)


local function register_test(name, sql, func)
    mtt.register("squill:" .. name, function(callback)
        local db_name = "squill:test_" .. name
        local id_offset = get_id_offset()
        local retval
        if sql then
            retval = squill.exec(db_name, sql)
        end
        func(db_name, retval, id_offset)
        cleanup()
        callback()
    end)
end

local expr_tests = {}
local function expr_test(sql, expected)
    expr_tests[#expr_tests + 1] = function(db_name)
        core.log("action", "Running expression test: " .. sql)
        local retval = squill.exec(db_name, "SELECT " .. sql)
        assert_eq(retval[1][1]["?column?"], expected)
    end
end

-- Run all the expression tests at once to speed up test execution
register_test("exprs", nil, function(db_name)
    for _, func in ipairs(expr_tests) do
        func(db_name)
    end
end)

expr_test("5 + 6 / 3 * (2 - 1)", 7)
expr_test("1 = 2", false)
expr_test("1 = 1", true)
expr_test("1 <> 2", true)
expr_test("TRUE + '1' / 2", 1.5)

expr_test("1 = null", nil)
expr_test("null = 1", nil)
expr_test("1 <> null", nil)
expr_test("null <> 1", nil)

expr_test("null = null", nil)
expr_test("null <> null", nil)
expr_test("null is null", true)
expr_test("5 is null", false)
expr_test("null is not null", false)
expr_test("5 is not null", true)
expr_test("(5 + 4 + NULL) / 3", nil)
expr_test("null and true", nil)
expr_test("null or true", true)
expr_test("null and false", false)
expr_test("true or null", true)
expr_test("true and false", false)
expr_test("false and true", false)
expr_test("true and true", true)
expr_test("null or false", nil)
expr_test("false or false", false)
expr_test("false or null", nil)
expr_test("true and null", nil)

expr_test("'hello' LIKE 'h%'", true)
expr_test("'world' NOT LIKE 'h%'", true)
expr_test("'world' NOT LIKE '%d'", false)
expr_test("'hello' LIKE 'w_rld'", false)
expr_test("'world' LIKE 'w_rld'", true)

expr_test("'HELLO' LIKE 'h%o'", true)
expr_test("'WORLD' NOT LIKE 'h%o'", true)

-- GLOBs are case sensitive
expr_test("'hello world%' GLOB 'h*r?d[%?]'", true)
expr_test("'hello world?' NOT GLOB 'h*r?d[%a]'", true)
expr_test("'hello woRld%' GLOB 'h*r?d%'", false)
expr_test("'abc' GLOB 'a[bc]c'", true)
expr_test("'abc' GLOB 'a[cd]c'", false)
expr_test("'b' GLOB '[a-z]'", true)
expr_test("'a' GLOB '[?]'", false)
expr_test("'?' GLOB '[?]'", true)

expr_test("1 LIKE '1'", true)
expr_test("NULL LIKE 'a'", nil)
expr_test("'a' LIKE NULL", nil)

expr_test("CONCAT('Hello, ', 'world!')", "Hello, world!")
expr_test("'a' || 'b' || 'c'", "abc")
expr_test("'a' || null", nil)
expr_test("null || 'a'", nil)

expr_test("UPPER('Hello')", "HELLO")
expr_test("LOWER('Hello')", "hello")
expr_test("UPPER(NULL)", nil)
expr_test("LOWER(NULL)", nil)

expr_test("3 ^ 2", 9)
expr_test("pow(3, 2)", 9)
expr_test("degrees(pi())", 180)
expr_test("pi()", math.pi)
expr_test("sin(pi() / 2)", 1)
expr_test("ceiling(1.5)", 2)
expr_test("floor(1.5)", 1)
expr_test("log10(1000)", 3)
expr_test("min(2, 4, 6)", 2)
expr_test("max(2, 4, 6)", 6)
expr_test("max(2, null, 3)", nil)

expr_test("2 between 3 and 5", false)
expr_test("3 between 3 and 5", true)
expr_test("4 between 3 and 5", true)
expr_test("5 between 3 and 5", true)
expr_test("6 between 3 and 5", false)

expr_test("3 in (1, null, 3)", true)
expr_test("4 in (1, null, 3)", nil)
expr_test("4 in (1, 2, 3)", false)

expr_test("3 not in (1, null, 3)", false)
expr_test("4 not in (1, null, 3)", nil)
expr_test("4 not in (1, 2, 3)", true)

expr_test("CASE null WHEN null THEN 1 ELSE 0 END", 0)

-- JSON
expr_test([['{
    "key": "value \"with escape\"",
    "key2": ["value2", 3e+4, 4e-5, 5],
    "key3": 4
}' -> 'key2' ->> 2]], 4e-5)
expr_test([['{"t":true,"f":false,"n":null}'->'t']], "true")
expr_test([['{"t":true,"f":false,"n":null}'->'f']], "false")
expr_test([['{"t":true,"f":false,"n":null}'->'n']], "null")
expr_test("'{}' -> 'key1' -> 'key2' ->> 'key3'", nil)
expr_test("'3' ->> '$'", 3)

expr_test([[
    CASE
        WHEN 1 == 2 THEN 'bad'
        WHEN 1 == 1 THEN 'ok'
        ELSE 'else'
    END
]], "ok")

register_test("create_insert", [[
    CREATE TABLE test (
        first int NOT NULL,
        second text,
        third blob
    );
    -- This is a line comment
    INSERT INTO test (first, second, third) VALUES (1, 'two', 'three');
    /*
    This is a block comment!
    */
    INSERT INTO test (first, second, third) VALUES (4, 'five', 'six');
    SELECT * FROM test;
]], function(db_name, retval, id_offset)
    assert_eq(retval[4], {
        {first = 1, second = "two", third = "three"},
        {first = 4, second = "five", third = "six"},
    })

    local schema_rows = squill.prepare_statement("squill", [[
        SELECT table, column, type, not_null, id
        FROM schema WHERE db_name = ?
        ORDER BY id
    ]])(db_name)

    assert_eq(schema_rows, {
        {table = "test", column = "first", type = "I", not_null = true, id = id_offset + 1},
        {table = "test", column = "second", type = "S", not_null = false, id = id_offset + 2},
        {table = "test", column = "third", type = "B", not_null = false, id = id_offset + 3},
    })
end)

register_test("update", [[
    CREATE TABLE test (
        first int,
        second text,
        third blob
    );
    INSERT INTO test (first, second, third) VALUES (1, 'two', 'three');
    INSERT INTO test (first, second, third) VALUES (4, 'five', 'six');
    INSERT INTO test (first, second, third) VALUES (NULL, 'seven', 'eight');
    INSERT INTO test (first, second, third) VALUES (4, 'nine', 'ten');

    UPDATE test SET second = 'updated!' WHERE first = 4;
    UPDATE test SET second = 'bad' WHERE first = NULL;

    SELECT * from test;
]], function(_, retval)
    assert_eq(retval[6].affected_rows, 2)
    assert_eq(retval[7].affected_rows, 0)
    assert_eq(retval[8], {
        {first = 1, second = "two", third = "three"},
        {first = 4, second = "updated!", third = "six"},
        {second = "seven", third = "eight"},
        {first = 4, second = "updated!", third = "ten"},
    })
end)


register_test("delete", [[
    CREATE TABLE test (
        first int NOT NULL,
        second text,
        third blob
    );
    INSERT INTO test (first, second, third) VALUES (1, 'two', 'three');
    INSERT INTO test (first, second, third) VALUES (4, 'five', 'six');
    INSERT INTO test (first, second, third) VALUES (6, '7', '8');

    DELETE FROM test WHERE first = 4 OR first = 1;

    SELECT * from test;
]], function(_, retval)
    assert_eq(retval[5].affected_rows, 2)
    assert_eq(retval[6], {
        {first = 6, second = "7", third = "8"},
    })
end)


register_test("transactions", [[
    CREATE TABLE test (
        first int NOT NULL,
        second text,
        third blob
    );
    INSERT INTO test (first, second, third) VALUES (1, 'two', 'three');

    BEGIN;
    INSERT INTO test (first, second, third) VALUES (4, 'five', 'six');
    INSERT INTO test (first, second, third) VALUES (6, '7', '8');
    COMMIT;

    BEGIN;
    DELETE FROM test WHERE first <> 6;
    UPDATE test SET third = 'test' WHERE true;
    INSERT INTO test (third, first) VALUES (8, 9);
    SELECT * FROM test;
    ROLLBACK;

    SELECT * FROM test;
]], function(_, retval)
    assert_eq(retval[#retval - 2], {
        {first = 6, second = "7", third = "test"},
        {first = 9, third = "8"},
    })
    assert_eq(retval[#retval], {
        {first = 1, second = "two", third = "three"},
        {first = 4, second = "five", third = "six"},
        {first = 6, second = "7", third = "8"},
    })
end)

register_test("aggregate", [[
    CREATE TABLE test (a int, b int);
    INSERT INTO test (a, b) VALUES (123, 10);
    INSERT INTO test (a, b) VALUES (500, 10);
    INSERT INTO test (a, b) VALUES (NULL, 9);
    INSERT INTO test (a, b) VALUES (246, 9);

    SELECT SUM(a) as c, b FROM test GROUP BY b HAVING b = 10;

    SELECT COUNT(*) as c, COUNT(a) as not_null FROM test;
    SELECT SUM(a) AS s, MIN(a) as m, MAX(b) as mb FROM test;

    SELECT SUM(a) as c, b FROM test GROUP BY b ORDER BY b;
]], function(_, retval)
    assert_eq(retval[#retval - 3], {
        {c = 123 + 500, b = 10},
    })
    assert_eq(retval[#retval - 2], {
        {c = 4, not_null = 3},
    })
    assert_eq(retval[#retval - 1], {
        {s = 123 + 500 + 246, m = 123, mb = 10},
    })
    assert_eq(retval[#retval], {
        {c = 246, b = 9},
        {c = 123 + 500, b = 10},
    })
end)

register_test("aggregate_multi", [[
    CREATE TABLE test (a int, b int, c int);
    INSERT INTO test (a, b, c) VALUES (123, 10, 1);
    INSERT INTO test (a, b, c) VALUES (500, 10, 1);
    INSERT INTO test (a, b, c) VALUES (NULL, 9, 1);
    INSERT INTO test (a, b, c) VALUES (246, 9, 1);
    INSERT INTO test (a, b, c) VALUES (123, 10, 2);
    INSERT INTO test (a, b, c) VALUES (NULL, 9, 2);
    INSERT INTO test (a, b, c) VALUES (246, 9, 2);

    SELECT a, b, COUNT(*) * 3 as c FROM test GROUP BY a, b ORDER BY a;

    SELECT a, b, COUNT(a) as not_null FROM test GROUP BY a, b
    HAVING SUM(b) = 18 ORDER BY a;

    SELECT a, b, COUNT(a) as not_null FROM test GROUP BY a, b
    HAVING b = 10 ORDER BY a;

    SELECT COUNT(DISTINCT a) as d FROM test;
    SELECT COUNT(DISTINCT a) as d FROM test GROUP BY c ORDER BY c;
]], function(_, retval)
    assert_eq(retval[#retval - 4], {
        {b = 9, c = 6},
        {a = 123, b = 10, c = 6},
        {a = 246, b = 9, c = 6},
        {a = 500, b = 10, c = 3},
    })
    assert_eq(retval[#retval - 3], {
        {b = 9, not_null = 0},
        {a = 246, b = 9, not_null = 2},
    })
    assert_eq(retval[#retval - 2], {
        {a = 123, b = 10, not_null = 2},
        {a = 500, b = 10, not_null = 1},
    })
    assert_eq(retval[#retval - 1], {{d = 3}})
    assert_eq(retval[#retval], {{d = 3}, {d = 2}})
end)

register_test("order_by", [[
    CREATE TABLE test (a int NOT NULL, b int NOT NULL);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 1);
    INSERT INTO test (a, b) VALUES (3, 1);
    INSERT INTO test (a, b) VALUES (5, 2);
    SELECT * FROM test ORDER BY a;
    SELECT * FROM test ORDER BY a DESC;
    SELECT * FROM test ORDER BY b DESC, a;
    SELECT * FROM test ORDER BY a = 3 DESC, a LIMIT 2;
]], function(_, retval)
    assert_eq(retval[6], {
        {a = 1, b = 2},
        {a = 2, b = 1},
        {a = 3, b = 1},
        {a = 5, b = 2},
    })
    assert_eq(retval[7], {
        {a = 5, b = 2},
        {a = 3, b = 1},
        {a = 2, b = 1},
        {a = 1, b = 2},
    })
    assert_eq(retval[8], {
        {a = 1, b = 2},
        {a = 5, b = 2},
        {a = 2, b = 1},
        {a = 3, b = 1},
    })
    assert_eq(retval[9], {
        {a = 3, b = 1},
        {a = 1, b = 2},
    })
end)

register_test("rename_table", [[
    CREATE TABLE test (a int unique not null, b blob unique not null);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 3);
    INSERT INTO test (a, b) VALUES (3, 4);
    INSERT INTO test (a, b) VALUES (8, 5);

    ALTER TABLE test RENAME TO mytable;
    UPDATE mytable SET b = b * 6 WHERE b <> 5;
    SELECT * FROM mytable;
]], function(db_name, retval)
    assert(not pcall(squill.exec, db_name, "SELECT * FROM test"))
    assert_eq(retval[#retval], {
        {a = 1, b = "12"},
        {a = 2, b = "18"},
        {a = 3, b = "24"},
        {a = 8, b = "30"},
    })

    assert_eq(squill.exec("squill", [[
        SELECT COUNT(*) FROM schema WHERE table = 'test';
        SELECT COUNT(*) FROM schema WHERE table = 'mytable';
    ]]), {{{["?column?"] = 0}}, {{["?column?"] = 2}}})
end)

register_test("add_column", [[
    CREATE TABLE test (a int unique not null, b int);
    INSERT INTO test (a) VALUES (1);
    INSERT INTO test (a) VALUES (2);
    INSERT INTO test (a) VALUES (3);
    INSERT INTO test (a) VALUES (4);

    ALTER TABLE test ADD COLUMN c int;
    UPDATE test SET c = a * 2 WHERE a <> 2;
    SELECT * FROM test;
]], function(db_name, retval, id_offset)
    assert_eq(retval[#retval], {
        {a = 1, c = 2},
        {a = 2},
        {a = 3, c = 6},
        {a = 4, c = 8},
    })

    assert_eq(squill.exec("squill", [[
        SELECT column, id FROM schema
        WHERE db_name = ? AND table = 'test'
        ORDER BY id
    ]], db_name), {
        {
            {column = "a", id = id_offset + 1},
            {column = "b", id = id_offset + 2},
            {column = "c", id = id_offset + 3},
        }
    })
end)

register_test("drop_column", [[
    CREATE TABLE test (
        a int unique not null,
        b blob unique not null check(b is not null)
    );
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 4);
    INSERT INTO test (a, b) VALUES (3, 6);
    INSERT INTO test (a, b) VALUES (4, 8);

    ALTER TABLE test DROP COLUMN b;
    SELECT * FROM test;

    CREATE TABLE test2 (
        c int REFERENCES test (a)
    );
    ALTER TABLE test2 DROP COLUMN c;
]], function(db_name, retval, id_offset)
    assert_eq(retval[#retval - 2], {
        {a = 1},
        {a = 2},
        {a = 3},
        {a = 4},
    })

    assert_eq(squill.exec("squill", [[
        SELECT column, id FROM schema
        WHERE db_name = $1 AND table = 'test'
        ORDER BY id;

        SELECT column_id FROM uniques u
        JOIN schema s ON u.column_id = s.id
        WHERE db_name = $1 AND table = 'test';

        SELECT COUNT(*) as c FROM foreign_keys k
        JOIN schema s ON k.child_id = s.id
        WHERE db_name = $1 AND table = 'test2';
    ]], db_name), {
        {
            {column = "a", id = id_offset + 1},
        },
        {
            {column_id = id_offset + 1},
        },
        {
            {c = 0}
        }
    })

    -- Make sure the blob data got deleted by adding the column back and
    -- checking for it
    assert_eq(squill.exec(db_name, [[
        ALTER TABLE test ADD COLUMN b blob;
        SELECT COUNT(*) FROM test WHERE b IS NOT NULL;
    ]])[2], {
        {["?column?"] = 0}
    })
end)

register_test("unique", [[
    CREATE TABLE test (a int unique not null, b int);
    BEGIN; -- Make sure insert or ignore doesn't corrupt transation variables
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (3, 4);
    INSERT OR IGNORE INTO test (a, b) VALUES (3, 5);
    SELECT b FROM test WHERE a = 3;
    COMMIT;
]], function(db_name, retval)
    assert_eq(retval[#retval - 2], {affected_rows = 0})
    assert_eq(retval[#retval - 1], {{b = 4}})
    assert(not pcall(squill.exec, db_name, [[
        INSERT INTO test (a, b) VALUES (3, 5);
    ]]))
    assert(not pcall(squill.exec, db_name, [[
        UPDATE test SET a = 1 WHERE b = 4;
    ]]))

    assert_eq(squill.exec(db_name, "SELECT * FROM test")[1], {
        {a = 1, b = 2},
        {a = 3, b = 4},
    })

    squill.drop_query_cache()
    squill.drop_column_cache()
    assert_eq(squill.exec(db_name, "SELECT * FROM test")[1], {
        {a = 1, b = 2},
        {a = 3, b = 4},
    })
end)

register_test("multi_unique", [[
    CREATE TABLE test (
        a int,
        b int,
        c int,
        UNIQUE (a, b)
    );
    INSERT INTO test (a, b, c) VALUES (1, 2, 3);
    BEGIN;
    INSERT INTO test (a, b, c) VALUES (5, 4, 3);
    INSERT INTO test (a, b, c) VALUES (1, 3, 4);
    INSERT OR REPLACE INTO test (a, b, c) VALUES (1, 3, 5);
    INSERT OR IGNORE INTO test (a, b, c) VALUES (1, 3, 6);
    COMMIT;
]], function(db_name, _)
    assert(not pcall(squill.exec, db_name, [[
        INSERT INTO test (a, b, c) VALUES (5, 4, 0);
    ]]))
    assert_eq(squill.exec(db_name, "SELECT * FROM test")[1], {
        {a = 1, b = 2, c = 3},
        {a = 5, b = 4, c = 3},
        {a = 1, b = 3, c = 5},
    })
end)

register_test("check", [[
    CREATE TABLE test (
        a int CHECK (test.a > b),
        b int
    );
    INSERT INTO test (a, b) VALUES (3, 1);
    INSERT INTO test (a, b) VALUES (10, 5);
    UPDATE TEST SET b = 2 WHERE a = 3;
]], function(db_name, _)
    assert(not pcall(squill.exec, db_name, [[
        INSERT INTO test (a, b) VALUES (3, 5);
    ]]))
    assert(not pcall(squill.exec, db_name, [[
        UPDATE TEST SET b = 10 WHERE a = 3;
    ]]))
    assert_eq(squill.exec(db_name, "SELECT * FROM test")[1], {
        {a = 3, b = 2},
        {a = 10, b = 5},
    })
end)

register_test("join", [[
    CREATE TABLE test1 (
        name text,
        second int unique
    );
    CREATE TABLE test2 (
        name2 text,
        second int,
        third int
    );
    CREATE TABLE test3 (
        third int,
        name3 text
    );

    INSERT INTO test1 (name, second) VALUES ('First', 2);
    INSERT INTO test2 (name2, second, third) VALUES ('One', 2, 5);
    INSERT INTO test3 (name3, third) VALUES ('1', 5);

    INSERT INTO test1 (name, second) VALUES ('Second', 10);
    INSERT INTO test2 (name2, second, third) VALUES ('Two', 10, 4);
    INSERT INTO test3 (name3, third) VALUES ('2', 4);

    SELECT name, name2, name3, second FROM test1 JOIN test2 USING (second)
    JOIN test3 ON test2.third = test3.third
    ORDER BY test3.third;
]], function(_, retval)
    assert_eq(retval[#retval], {
        {
            name = "Second",
            name2 = "Two",
            name3 = "2",
            second = 10,
        },
        {
            name = "First",
            name2 = "One",
            name3 = "1",
            second = 2,
        },
    })
end)

register_test("autoincrement", [[
    CREATE TABLE test (
        key INTEGER PRIMARY KEY AUTOINCREMENT,
        value text
    );

    INSERT INTO test (value) VALUES ('Hello,');
    INSERT INTO test (value) VALUES ('world!');
    INSERT INTO test (value) VALUES ('This is a test');
    INSERT INTO test (key, value) VALUES (123, 'Autoincrement increased');
    INSERT INTO test (value) VALUES ('test');
    INSERT INTO test (key, value) VALUES (10, 'test2');
    INSERT INTO test (value) VALUES ('test3');

    SELECT * FROM test ORDER BY key;
]], function(_, retval)
    assert_eq(retval[#retval], {
        {key = 1, value = "Hello,"},
        {key = 2, value = "world!"},
        {key = 3, value = "This is a test"},
        {key = 10, value = "test2"},
        {key = 123, value = "Autoincrement increased"},
        {key = 124, value = "test"},
        {key = 125, value = "test3"},
    })
end)

register_test("case", [[
    CREATE TABLE test (
        k INTEGER PRIMARY KEY AUTOINCREMENT,
        a int not null,
        b text
    );

    INSERT INTO test (a) values (1);
    INSERT INTO test (a) values (123);
    INSERT INTO test (a) values (53);
    INSERT INTO test (a) values (512);
    INSERT INTO test (a) values (5321);
    INSERT INTO test (a) values (10);

    SELECT MAX(a) as x, COUNT(*) as y FROM test GROUP BY CASE
        WHEN a < 100 THEN 1
        WHEN a < 500 THEN 2
        ELSE 3
    END ORDER BY a;

    UPDATE test SET b = CASE
        WHEN a < 50 THEN '<50'
        WHEN a < 100 THEN '50-99'
        WHEN a < 1000 THEN CASE a
            -- Nested CASE
            WHEN 512 THEN '512'
            ELSE '100-999'
        END
        ELSE '1000+'
    END WHERE a <> 1;

    SELECT * FROM test ORDER BY k;
]], function(_, retval)
    assert_eq(retval[#retval - 2], {
        {x = 53, y = 3},
        {x = 123, y = 1},
        {x = 5321, y = 2},
    })

    assert_eq(retval[#retval], {
        {k = 1, a = 1},
        {k = 2, a = 123, b = "100-999"},
        {k = 3, a = 53, b = "50-99"},
        {k = 4, a = 512, b = "512"},
        {k = 5, a = 5321, b = "1000+"},
        {k = 6, a = 10, b = "<50"},
    })
end)

register_test("select_where_simple", [[
    CREATE TABLE test (a integer primary key, b blob);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 4);
    INSERT INTO test (a, b) VALUES (3, 6);
    INSERT INTO test (a, b) VALUES (4, 8);

    SELECT a, b FROM test WHERE a = 3;
]], function(_, retval)
    assert_eq(retval[#retval], {
        {a = 3, b = "6"},
    })
end)

-- Make sure that INSERT OR statements don't corrupt the index
register_test("insert_or_replace_updates_index", [[
    CREATE TABLE test (a integer primary key, b int unique);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 4);
    INSERT INTO test (a, b) VALUES (3, 6);
    INSERT INTO test (a, b) VALUES (4, 8);
    INSERT OR REPLACE INTO test (a, b) VALUES (1, 6);
    INSERT INTO test (a, b) VALUES (5, 10);

    SELECT a, b FROM test WHERE a = 1;
    SELECT a, b FROM test WHERE b = 8;
]], function(_, retval)
    assert_eq(retval[#retval - 1], {{a = 1, b = 6}})
    assert_eq(retval[#retval], {{a = 4, b = 8}})
end)

register_test("insert_or_ignore_updates_index", [[
    CREATE TABLE test (a integer primary key, b int unique);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 4);
    INSERT INTO test (a, b) VALUES (3, 6);
    INSERT INTO test (a, b) VALUES (4, 8);
    INSERT OR IGNORE INTO test (a, b) VALUES (1, 6);
    INSERT INTO test (a, b) VALUES (5, 10);

    SELECT a, b FROM test WHERE a = 1;
    SELECT a, b FROM test WHERE b = 8;
]], function(_, retval)
    assert_eq(retval[#retval - 1], {{a = 1, b = 2}})
    assert_eq(retval[#retval], {{a = 4, b = 8}})
end)

register_test("update_updates_index", [[
    CREATE TABLE test (a integer primary key, b int unique);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, 4);
    INSERT INTO test (a, b) VALUES (3, 6);
    INSERT INTO test (a, b) VALUES (4, 8);
    INSERT INTO test (a, b) VALUES (5, 10);

    UPDATE test SET a = 6 WHERE b = 8;
    SELECT a, b FROM test WHERE a = 6;

    UPDATE test SET b = 3 WHERE b = 2;
    SELECT a, b FROM test WHERE b = 2;
    SELECT a, b FROM test WHERE b = 3;
]], function(db_name, retval)
    assert_eq(retval[#retval - 3], {{a = 6, b = 8}})
    assert_eq(retval[#retval - 1], {})
    assert_eq(retval[#retval], {{a = 1, b = 3}})

    assert(not pcall(squill.exec, db_name, [[
        UPDATE test SET a = 6 WHERE b = 3;
    ]]))

    assert_eq(squill.exec(db_name, [[
        SELECT a, b FROM test WHERE a = 2;
        SELECT a, b FROM test WHERE b = 3;
    ]]), {
        {{a = 2, b = 4}},
        {{a = 1, b = 3}},
    })
end)

register_test("index_handles_nulls", [[
    CREATE TABLE test (a integer primary key, b int unique);
    INSERT INTO test (a, b) VALUES (1, 2);
    INSERT INTO test (a, b) VALUES (2, null);
    INSERT INTO test (a, b) VALUES (3, null);

    SELECT * FROM test WHERE b = 2;
    SELECT * FROM test WHERE b is null;

    UPDATE test SET b = 3 WHERE a = 2;
    SELECT * FROM test WHERE b = 3;
]], function(_, retval)
    assert_eq(retval[#retval - 3], {{a = 1, b = 2}})
    assert_eq(retval[#retval - 2], {{a = 2}, {a = 3}})
    assert_eq(retval[#retval], {{a = 2, b = 3}})
end)

register_test("foreign_keys", [[
    CREATE TABLE two (
        one_a integer,
        c int,
        FOREIGN KEY (one_a) REFERENCES one (a)
    );
    CREATE TABLE one (a integer primary key, b int);

    INSERT INTO one VALUES (1, 2);
    INSERT INTO one VALUES (2, 3);
    INSERT INTO one VALUES (3, 4);
    INSERT INTO two VALUES (2, 5);
]], function(db_name, _)
    local insert2 = squill.prepare_statement(db_name, [[
        INSERT INTO two VALUES (?, ?)
    ]])
    local update1 = squill.prepare_statement(db_name, [[
        UPDATE one SET a = 10 * $1 WHERE a = $1
    ]])
    local delete1 = squill.prepare_statement(db_name, [[
        DELETE FROM one WHERE a = $1
    ]])
    insert2(1, 4)
    assert(not pcall(update1, 1))
    update1(3)

    assert(not pcall(delete1, 1))
    assert_eq(delete1(10).affected_rows, 0)
    delete1(30)

    assert_eq(squill.exec(db_name, [[
        SELECT * FROM one ORDER BY a;
        SELECT * FROM two ORDER BY c;
    ]]), {
        {{a = 1, b = 2}, {a = 2, b = 3}},
        {{one_a = 1, c = 4}, {one_a = 2, c = 5}},
    })
end)

register_test("connection_api", [[
    CREATE TABLE test (x int, y int);
    INSERT INTO test VALUES (123, 456);
]], function(db_name, _)
    local db = squill.connect(db_name)
    assert_eq(db:prepare([[
        SELECT * FROM test
    ]])(), {{x = 123, y = 456}})
    assert_eq(db:prepare("SELECT 1234", squill.RETURN_SINGLE_VALUE)(), 1234)
end)

register_test("insert_or_replace", [[
    CREATE TABLE a (
        x INTEGER PRIMARY KEY,
        y TEXT UNIQUE
    );

    CREATE TABLE b (
        x INTEGER REFERENCES a (x),
        w TEXT
    );

    INSERT INTO a VALUES (1, 'first');
    INSERT INTO a VALUES (2, 'second');
    INSERT INTO a VALUES (3, 'third');
    INSERT INTO b VALUES (1, 'firstw');
    INSERT INTO b VALUES (2, 'secondw');
]], function(db_name, _)
    local func = squill.prepare_statement(db_name, [[
        INSERT OR REPLACE INTO a VALUES (?, ?) RETURNING x, y
    ]], squill.RETURN_FIRST_ROW)

    assert_eq(func(1, "testing"), {x = 1, y = "testing"})
    assert(not pcall(func, 4, 'second'))
    assert_eq(func(4, "third"), {x = 4, y = "third"})

    assert_eq(squill.exec(db_name, [[
        SELECT * FROM a ORDER BY x
    ]]), {{
        {x = 1, y = 'testing'},
        {x = 2, y = 'second'},
        {x = 4, y = 'third'},
    }})
end)

register_test("union_order_by", [[
    SELECT 1 a, 2 b, 3 c
    UNION ALL
    SELECT 4, 5, 6
    UNION ALL
    SELECT 3, 7, 8
    ORDER BY a;
]], function(_, res)
    assert_eq(res, {{
        {a = 1, b = 2, c = 3},
        {a = 3, b = 7, c = 8},
        {a = 4, b = 5, c = 6},
    }})
end)

register_test("nested_select", [[
    CREATE TABLE test (
        first int NOT NULL,
        second text,
        third blob
    );
    INSERT INTO test (first, second, third) VALUES (1, 'two', 'three');
    INSERT INTO test (first, second, third) VALUES (4, 'five', 'six');

    SELECT first a, second b, third c FROM (
        SELECT first, second, third FROM test
        UNION ALL
        SELECT 3, 'four', 'five'
    ) ORDER BY first;

    SELECT b + 100 a FROM (
        SELECT c + 20 b FROM (
            SELECT 1 a, 2 b, 3 c
        )
    );
]], function(_, res)
    assert_eq(res[#res - 1], {
        {a = 1, b = 'two', c = 'three'},
        {a = 3, b = 'four', c = 'five'},
        {a = 4, b = 'five', c = 'six'},
    })
    assert_eq(res[#res], {
        {a = 123}
    })
end)

register_test("default_value", [[
    CREATE table test (
        first text DEFAULT 'default',
        second int NOT NULL
    );

    INSERT INTO test (second) VALUES (1);
    INSERT INTO test VALUES ('non-default', 2);

    SELECT * FROM test ORDER BY second;
]], function(_, res)
    assert_eq(res[#res], {
        {first = 'default', second = 1},
        {first = 'non-default', second = 2},
    })
end)
