local S = core.get_translator("m16b_mystic_stones")
local mob_class = mcl_mobs.mob_class
local posing_humanoid = mcl_mobs.posing_humanoid

local light_skeleton = table.merge (posing_humanoid, {
	description = S("Lightning Skeleton"),
	type = "monster",
	spawn_class = "hostile",
	_spawn_category = "monster",
	hp_min = 20,
	hp_max = 20,
	xp_min = 6,
	xp_max = 6,
	breath_max = -1,
	armor = {undead = 100, fleshy = 100},
	collisionbox = {-0.3, 0.0, -0.3, 0.3, 1.99, 0.3},
	head_swivel = "Head_Control",
	bone_eye_height = 2.38,
	head_eye_height = 1.74,
	curiosity = 6,
	visual = "mesh",
	mesh = "mobs_mc_skeleton.b3d",
	wears_armor = "no_pickup",
	armor_drop_probability = {
		head = 0.085,
		torso = 0.085,
		legs = 0.085,
		feet = 0.085,
	},
	makes_footstep_sound = true,
	textures = {
			"mobs_mc_empty.png", -- armor
			"m16b_light_skeleton_overlay.png", -- overlay
			"m16b_light_skeleton.png", -- texture
	},
	movement_speed = 5.0,
	sounds = {
		random = "mobs_mc_skeleton_random",
		death = "mobs_mc_skeleton_death",
		damage = "mobs_mc_skeleton_hurt",
		distance = 16,
	},
	damage = 2,
	reach = 2,
	drops = {
		{
			name = "mcl_bows:arrow",
			chance = 1,
			min = 0,
			max = 2,
			looting = "common",
		},
		{
			name = "mcl_mobitems:bone",
			chance = 1,
			min = 0,
			max = 2,
			looting = "common",
		},
--[[
		{
			name = "mcl_heads:skeleton",
			chance = 1,
			min = 0,
			max = 0,
			mob_head = true,
		},
]]--
	},
	animation = {
		stand_speed = 15,
		stand_start = 0,
		stand_end = 40,
		walk_speed = 15,
		walk_start = 40,
		walk_end = 60,
	},
	ignited_by_sunlight = false,
	avoids_sunlight = true,
	floats = 0,
	attack_type = "bowshoot",
	specific_attack = {
		"mobs_mc:iron_golem",
	},
	arrow = "mcl_bows:arrow_entity",
	shoot_interval = 1,
	shoot_offset = 1.5,
	harmed_by_heal = true,
	can_wield_items = "no_pickup",
	wielditem_info = {
		toollike_position = vector.new (1.1, 2.1, 0),
		toollike_rotation = vector.new (0, 0, -45),
		bow_position = vector.new (0, 2.1, -0.2),
		bow_rotation = vector.new (-7.107, 7.053, -45.439),
		crossbow_position = vector.new (0.2, 2.1, -0.2),
		crossbow_rotation = vector.new (-97, 45, -95),
		blocklike_position = vector.new (0.4, 2.1, 0),
		blocklike_rotation = vector.new (180, 45, 0),
		position = vector.new (0.2, 2.1, 0),
		rotation = vector.new (-90, 0, 0),
		bone = "wield_item",
		rotate_bone = true,
	},
	wielditem_drop_probability = 0.085,
	_humanoid_superclass = mob_class,
	_mcl_freeze_damage = 0,
	_frozen_time = 0,
})

------------------------------------------------------------------------
-- Lightning Skeleton visuals.
------------------------------------------------------------------------

local light_skeleton_poses = {
	default = {
		["arm.right"] = {},
		["arm.left"] = {},
	},
	shoot = {
		["arm.right"] = {
			nil,
			vector.new (90, 0, 90),
		},
		["arm.left"] = {
			nil,
			vector.new (110, 0, 90),
		},
	},
	attack = {
		["arm.right"] = {
			nil,
			vector.new (90, 0, 90),
		},
		["arm.left"] = {
			nil,
			vector.new (90, 0, 90),
		},
	},
}

light_skeleton._arm_poses = light_skeleton_poses

function light_skeleton:wielditem_transform (info, stack)
	local rot, pos, size
		= mob_class.wielditem_transform (self, info, stack)
	size.x = size.x / 3
	size.y = size.y / 3
	return rot, pos, size
end

function light_skeleton:select_arm_pose ()
	local basic_pose = "default"

	if self.attack and self.attack_type == "bowshoot" then
		basic_pose = "shoot"
	elseif self.attack then
		basic_pose = "attack"
	end

	if self.jockey_vehicle then
		return "jockey_" .. basic_pose
	else
		return basic_pose
	end
end

------------------------------------------------------------------------
-- Lightning Skeleton mechanics.
------------------------------------------------------------------------

local pr = PcgRandom (os.time () + 332)

function light_skeleton:on_spawn ()
	local self_pos = self.object:get_pos ()
	local mob_factor = mcl_worlds.get_special_difficulty (self_pos)
	-- Enable picking up armor for a random subset of
	-- light_skeleton.
	if math.random () < 0.55 * mob_factor then
		self.wears_armor = true
		self.can_wield_items = true
	end
	self:skelly_generate_default_equipment (mob_factor)
	return true
end

function light_skeleton:skelly_generate_default_equipment (mob_factor)
	self:generate_default_equipment (mob_factor, true, false)
	self:set_wielditem (ItemStack ("mcl_bows:bow"))
	self:enchant_default_weapon (mob_factor, pr)
end

------------------------------------------------------------------------
-- Lightning Skeleton AI.
------------------------------------------------------------------------

function light_skeleton:validate_waypoints (waypoints)
	local self_pos = self.object:get_pos ()
	if self.armor_list.head == ""
		and self:endangered_by_sunlight ()
		and not mcl_weather.is_outdoor (self_pos)
		and #waypoints > 0 then
		local safe_waypoints = {}
		local n_waypoints = #waypoints
		for i = 0, n_waypoints - 1 do
			local r = n_waypoints - i
			if mcl_weather.is_outdoor (waypoints[r]) then
				break
			end
			table.insert (safe_waypoints, waypoints[r])
		end
		self.waypoints = safe_waypoints
		return
	end

	return mob_class.validate_waypoints (self, waypoints)
end

function light_skeleton:reconfigure_attack_type (wielditem)
	local name = wielditem:get_name ()

	if name ~= "" and core.get_item_group (name, "bow") > 0 then
		self:reset_attack_type ("bowshoot")
	else
		self:reset_attack_type ("melee")
	end
end

function light_skeleton:mob_activate (staticdata, dtime)
	if not posing_humanoid.mob_activate (self, staticdata, dtime) then
		return false
	end
	self:reconfigure_attack_type (self:get_wielditem ())
	return true
end

function light_skeleton:set_wielditem (stack, drop_probability)
	mob_class.set_wielditem (self, stack, drop_probability)
	self:reconfigure_attack_type (stack)
end

function light_skeleton:ai_step (dtime)
	mob_class.ai_step (self, dtime)

	if self.conversion_step then
		self:conversion_step (dtime)
	end

	if mcl_vars.difficulty < 3 then
		self.shoot_interval = 2
	else
		self.shoot_interval = 1
	end
end

function light_skeleton:shoot_arrow (pos, dir)
	local wielditem = self:get_wielditem ()
	mcl_bows.shoot_arrow ("mcl_bows:arrow", pos, dir,
			self:get_yaw (), self.object, 0.5333333, nil,
			false, wielditem)
end

light_skeleton.ai_functions = {
	mob_class.check_avoid_sunlight,
	mob_class.check_avoid,
	mob_class.check_attack,
	mob_class.check_pace,
}

mcl_mobs.register_mob ("m16b_mystic_stones:light_skeleton", light_skeleton)
mobs_mc.light_skeleton = light_skeleton

-------------------------------------------------------------------
--Spawning
-------------------------------------------------------------------
mcl_mobs.spawn_setup ({
	name = "m16b_mystic_stones:light_skeleton",
	type_of_spawning = "ground",
	dimension = "overworld",
	aoc = 2,
	biomes = {
		"LightningForest",
	},
	chance = 800,
})

mcl_mobs.register_egg ("m16b_mystic_stones:light_skeleton", S("Lightning Skeleton"), "#8f9134", "#b0b240", 0)
-------------------------------------------------------------------
-- Spawning 2????
-------------------------------------------------------------------
local light_skeleton_biomes = {}
local lightning_biomes = {
	"LightningForest",
}

for _, biome in pairs (mobs_mc.monster_biomes) do
	if table.indexof (lightning_biomes, biome) == -1 then
		table.insert (light_skeleton_biomes, biome)
	end
end

local light_skeleton_spawner = table.merge (mobs_mc.monster_spawner, {
	name = "m16b_mystic_stones:light_skeleton",
	weight = 80,
	pack_max = 4,
	pack_min = 4,
	biomes = lightning_biomes,
})

local monster_spawner = mobs_mc.monster_spawner

function light_skeleton_spawner:test_spawn_position (spawn_pos, node_pos, sdata, node_cache)
	return mcl_weather.is_outdoor (node_pos)
		and monster_spawner.test_spawn_position (self, spawn_pos,
							 node_pos, sdata,
							 node_cache)
end

mcl_mobs.register_spawner (light_skeleton_spawner)
