local modname = minetest.get_current_modname()
local S = minetest.get_translator(modname)
local D = mcl_util.get_dynamic_translator(modname)

m16b_infernal_shields = {
	types = {
		mob = true,
		player = true,
		arrow = true,
		generic = true,
		explosion = true,
		dragon_breath = true,
	},
	enchantments = {"mending", "unbreaking"},
	players = {},
}

local interact_priv = minetest.registered_privileges.interact
interact_priv.give_to_singleplayer = false
interact_priv.give_to_admin = false

local overlay = mcl_enchanting.overlay
local hud = "m16b_infernal_shield_hud.png"

minetest.register_tool("m16b_infernal_shields:infernal_shield", {
	description = S("infernal shield"),
	_doc_items_longdesc = S("A infernal_shield is a tool used for protecting the player against attacks."),
	inventory_image = "m16b_infernal_shield_48.png",
	stack_max = 1,
	groups = {
		infernal_shield = 1,
		weapon = 2,
		enchantability = -1,
		offhand_item = 1,
	},
	sound = {breaks = "default_tool_breaks"},
	_repair_material = "mcl_nether:netherite_ingot",
	wield_scale = vector.new(2, 2, 2),
	_mcl_wieldview_item = "",
	_placement_class = "infernal_shield",
	_mcl_uses = 336
})

local function wielded_item(obj, i)
	local itemstack = obj:get_wielded_item()
	if i == 1 then
		itemstack = mcl_offhand.get_offhand(obj)
	end
	return itemstack:get_name(), itemstack
end

local function set_wielded_item(player, stack, i)
	if i ~= 1 then
		player:set_wielded_item(stack)
	else
		mcl_offhand.set_offhand(player, stack)
	end
end

function m16b_infernal_shields.wielding_shield(obj, i)
	return wielded_item(obj, i):find("m16b_infernal_shields:infernal_shield")
end

local function infernal_shield_is_enchanted(obj, i)
	return mcl_enchanting.is_enchanted(wielded_item(obj, i))
end

local rgb_to_unicolor

local function migrate_custom_infernal_shield_texture(texture)
	-- Build colour mapping, required to parse layer info from old texture
	if not rgb_to_unicolor then
		rgb_to_unicolor = {}
		for _, v in pairs( mcl_dyes.colors ) do
			rgb_to_unicolor[v.rgb:lower()] = v.unicolor
		end
	end
	-- Rebuild layers from texture.
	-- Example: (m16b_infernal_shield_base_nopattern.png^m16b_infernal_shield_pattern_base.png^[mask:m16b_infernal_shield_base_nopattern.png^m16b_infernal_shield_pattern_base.png)^((m16b_infernal_shield_base_nopattern.png^m16b_infernal_shield_pattern_base.png^[colorize:#f1b216:224)^[mask:m16b_infernal_shield_pattern_base.png)^((m16b_infernal_shield_pattern_rhombus.png^[colorize:#912222:255)^[mask:m16b_infernal_shield_pattern_rhombus.png)^((m16b_infernal_shield_pattern_globe.png^[colorize:#60ac19:255)^[mask:m16b_infernal_shield_pattern_globe.png)
	local layers = {}
	for layer in texture:gmatch("m16b_infernal_shield_pattern_([%w_]+%.png%^%[colorize:#[%w]+)") do
		-- layer = base.png^[colorize:#f1b216, rhombus.png^[colorize:#912222, globe.png^[colorize:#60ac19
		local i,j = layer:find( "%.png%^%[colorize:" )
		local pattern, colour = layer:sub(1, i-1), layer:sub(j+1):lower()
		if pattern ~= "base" then -- Base colour already coded in itemstring, only need layers.
			if not rgb_to_unicolor[colour] then
				core.log("warning", "Cannot migrate old infernal_shield banner pattern: "..colour.." not found in dye")
				return nil
			end
			table.insert(layers, { color = "unicolor_"..rgb_to_unicolor[colour], pattern = pattern } )
		end
	end
	return layers
end

local infernal_shield_texture_builder = {
	-- luacheck will flag this style of "function table" as non-standard global hence add an exception
	blank = function() return infernal_shield_texture_builder.combine("mcl_banners_banner_base.png","") end, -- luacheck: globals infernal_shield_texture_builder
	base = function (rgb, ratio)
		local banner = "mcl_banners_banner_base.png"
		if rgb then banner = "(" .. banner .. "^[colorize:"..rgb..":"..ratio .. ")" end
		return banner -- Passed as "base" in combine()
	end,
	combine = function (base, layers)
		local escape = mcl_banners.escape_texture
		-- Enlarge base texture for banner placement.  Banner patterns need to be resized and offset to leave only front.
		local infernal_shield = "[combine:128x128:0,0=m16b_infernal_shield_base_nopattern.png\\^[resize\\:128x128"
		return infernal_shield .. ":4,4=" .. escape("[combine:20x40:-1,-1=" .. escape(base .. layers .."^[resize:64x64"))
	end,
}

local function set_infernal_shield_layers(itemstack, layers)
	if not itemstack then return end
	local itemname, meta = itemstack:get_name(), itemstack:get_meta()
	local def = core.registered_items[itemname]
	if not meta or not def or not def._infernal_shield_color_key then return end
	local b, base_colour = mcl_banners, def._infernal_shield_color_key

	if layers and #layers > 0 then mcl_banners.write_layers(meta, layers) end
	b.update_description(itemstack)

	local item_image = b.make_banner_texture(base_colour, layers, "item")
	item_image = item_image:gsub("mcl_banners_item_base_48.png", "m16b_infernal_shield_48.png")
	meta:set_string("inventory_overlay", item_image)

	local texture = b.make_banner_texture(base_colour, layers, infernal_shield_texture_builder)
	meta:set_string("m16b_infernal_shields:banner_texture", texture)
	return texture
end

minetest.register_entity("m16b_infernal_shields:infernal_shield_entity", {
	initial_properties = {
		visual = "mesh",
		mesh = "m16b_infernal_shield.obj",
		physical = false,
		pointable = false,
		collide_with_objects = false,
		textures = {"m16b_infernal_shield_base_nopattern.png"},
		visual_size = vector.new(1, 1, 1),
	},
	_blocking = false,
	_infernal_shield_number = 2,
	_texture_copy = "",
	on_step = function(self, _, _)
		local player = self.object:get_attach()
		if not player then
			self.object:remove()
			return
		end
		local infernal_shield_texture = "m16b_infernal_shield_base_nopattern.png"
		local i = self._infernal_shield_number
		local item, itemstack = wielded_item(player, i)

		if item ~= "m16b_infernal_shields:infernal_shield" and item ~= "m16b_infernal_shields:infernal_shield_enchanted" then -- Bannered infernal_shield?
			local meta = itemstack:get_meta()
			local meta_texture = meta:get_string("m16b_infernal_shields:banner_texture")
			if meta_texture and meta_texture ~= "" then
				infernal_shield_texture = meta_texture
			else
				local layers
				meta:set_string("wield_overlay", "") -- Clear inner face (wield_texture) to show raw infernal_shield.
				local custom_texture = meta:get_string("m16b_infernal_shields:infernal_shield_custom_pattern_texture")
				if custom_texture and custom_texture ~= "" then -- Parse layers from custom standalone pattern texture.
					infernal_shield_texture = custom_texture
					layers = migrate_custom_infernal_shield_texture(custom_texture) -- May be nil
					if layers then -- Item image would be broken on downgrade anyway, may as well remove old cache.
						meta:set_string("m16b_infernal_shields:infernal_shield_custom_pattern_texture", nil)
					end
				else
					layers = mcl_banners.read_layers(meta) -- Non-nil
				end
				if layers then
					local texture = set_infernal_shield_layers(itemstack, layers)
					if texture then
						infernal_shield_texture = texture
					end
				end
				meta:set_string("m16b_infernal_shields:banner_texture", infernal_shield_texture)
				set_wielded_item(player, itemstack, i)
			end
		end

		if infernal_shield_is_enchanted(player, i) then
			infernal_shield_texture = infernal_shield_texture .. overlay
		end

		if self._texture_copy ~= infernal_shield_texture then
			self.object:set_properties({textures = {infernal_shield_texture}})
			self._texture_copy = infernal_shield_texture
		end
	end,
})

for _, e in pairs(m16b_infernal_shields.enchantments) do
	mcl_enchanting.enchantments[e].secondary.infernal_shield = true
end

function m16b_infernal_shields.is_blocking(obj)
	if not obj:is_player() then return end
	if m16b_infernal_shields.players[obj] then
		local blocking = m16b_infernal_shields.players[obj].blocking
		if blocking <= 0 then return end
		local _, infernal_shieldstack = wielded_item(obj, blocking)
		return blocking, infernal_shieldstack
	end
end

mcl_damage.register_modifier(function(obj, damage, reason)
	local type = reason.type
	local damager = reason.direct
	local blocking, infernal_shieldstack = m16b_infernal_shields.is_blocking(obj)

	if not (obj:is_player() and blocking and m16b_infernal_shields.types[type] and damager) then
		return
	end

	local entity = damager:get_luaentity()
	if entity and entity._shooter then
		damager = entity._shooter
	end

	local dpos = damager:get_pos()

	-- Used for removed / killed entities before the projectile hits the player
	if entity and not entity._shooter and entity._saved_shooter_pos then
		dpos = entity._saved_shooter_pos
	end

	if not dpos or vector.dot(obj:get_look_dir(), vector.subtract(dpos, obj:get_pos())) < 0 then
		return
	end

	local durability = 500
	local unbreaking = mcl_enchanting.get_enchantment(infernal_shieldstack, m16b_infernal_shields.enchantments[2])
	if unbreaking > 0 then
		durability = durability * (unbreaking + 1)
	end

	if not minetest.is_creative_enabled(obj:get_player_name()) and damage >= 3 then
		infernal_shieldstack:add_wear(65535 / durability) ---@diagnostic disable-line: need-check-nil
		if blocking == 2 then
			obj:set_wielded_item(infernal_shieldstack)
		else
			obj:get_inventory():set_stack("offhand", 1, infernal_shieldstack)
			mcl_inventory.update_inventory_formspec(obj)
		end
	end
	minetest.sound_play({name = "mcl_block"}, {pos = obj:get_pos(), max_hear_distance = 16})
	return 0
end)

local function modify_infernal_shield(player, vpos, vrot, i)
	local arm = "Right"
	if i == 1 then
		arm = "Left"
	end
	local infernal_shield = m16b_infernal_shields.players[player].infernal_shields[i]
	if infernal_shield then
		infernal_shield:set_attach(player, "Arm_" .. arm, vpos, vrot, false)
	end
end

local function set_infernal_shield(player, block, i)
	if block then
		if i == 1 then
			modify_infernal_shield(player, vector.new(-9, 4, 0.5), vector.new(80, 100, 0), i) -- TODO
		else
			modify_infernal_shield(player, vector.new(-8, 4, -2.5), vector.new(80, 80, 0), i)
		end
	else
		if i == 1 then
			modify_infernal_shield(player, vector.new(-3, -5, 0), vector.new(0, 180, 0), i)
		else
			modify_infernal_shield(player, vector.new(3, -5, 0), vector.new(0, 0, 0), i)
		end
	end
	local infernal_shield = m16b_infernal_shields.players[player].infernal_shields[i]
	if not infernal_shield then return end

	local luaentity = infernal_shield:get_luaentity()
	if not luaentity then return end

	luaentity._blocking = block
end

local function set_interact(player, interact)
	local player_name = player:get_player_name()
	local privs = minetest.get_player_privs(player_name)
	if privs.interact == interact then
		return
	end
	local meta = player:get_meta()

	if interact and meta:get_int("m16b_infernal_shields:interact_revoked") ~= 0 then
		meta:set_int("m16b_infernal_shields:interact_revoked", 0)
		privs.interact = true
	elseif not interact then
		meta:set_int("m16b_infernal_shields:interact_revoked", privs.interact and 1 or 0)
		privs.interact = nil
	end

	minetest.set_player_privs(player_name, privs)
end

-- Prevent player from being able to circumvent interact privilage removal by
-- using infernal_shield.
minetest.register_on_priv_revoke(function(name, revoker, priv)
	if priv == "interact" and revoker then
		local player = minetest.get_player_by_name(name)
		if not player then
			return
		end
		local meta = player:get_meta()
		meta:set_int("m16b_infernal_shields:interact_revoked", 0)
	end
end)

local infernal_shield_hud = {}

local function remove_infernal_shield_hud(player)
	set_interact(player, true)
	playerphysics.remove_physics_factor(player, "speed", "infernal_shield_speed")

	player:hud_remove(infernal_shield_hud[player])
	infernal_shield_hud[player] = nil
	set_infernal_shield(player, false, 1)
	set_infernal_shield(player, false, 2)

	local hf = player:hud_get_flags()
	if not hf.wielditem then
		player:hud_set_flags({wielditem = true})
	end
end

local function add_infernal_shield_entity(player, i)
	local infernal_shield = minetest.add_entity(player:get_pos(), "m16b_infernal_shields:infernal_shield_entity")
	if infernal_shield and infernal_shield:get_pos() then
		infernal_shield:get_luaentity()._infernal_shield_number = i
		m16b_infernal_shields.players[player].infernal_shields[i] = infernal_shield
		set_infernal_shield(player, false, i)
	end
end

local function remove_infernal_shield_entity(player, i)
	local infernal_shields = m16b_infernal_shields.players[player].infernal_shields
	if infernal_shields[i] then
		infernal_shields[i]:remove()
		infernal_shields[i] = nil
	end
end

local function is_node_stack(itemstack)
	return itemstack:get_definition().drawtype -- only node's definition table contains element "drawtype"
end

local function is_rmb_conflicting_node(nodename)
	local nodedef = minetest.registered_nodes[nodename]
	return nodedef and nodedef.on_rightclick
end

function m16b_infernal_shields.set_blocking (player, blocking)
	local player_infernal_shield = m16b_infernal_shields.players[player]
	if player_infernal_shield then
		player_infernal_shield.blocking = blocking
	end
end

local function handle_blocking(player)
	local player_infernal_shield = m16b_infernal_shields.players[player]
	if not player_infernal_shield then return end

	if mcl_serverplayer.is_csm_at_least (player, 1) then
		local infernal_shield_in_offhand
			= m16b_infernal_shields.wielding_shield (player, 1)
		local infernal_shield_in_hand
			= m16b_infernal_shields.wielding_shield (player)
		if not infernal_shield_in_hand and not infernal_shield_in_offhand then
			player_infernal_shield.blocking = 0
		end
		return
	end

	local rmb = player:get_player_control().RMB
	if not rmb then
		if player_infernal_shield.blocking ~= 0 then
			mcl_serverplayer.handle_blocking (player, 0)
		end
		player_infernal_shield.blocking = 0
		return
	end

	local infernal_shield_in_offhand = m16b_infernal_shields.wielding_shield(player, 1)
	local infernal_shield_in_hand = m16b_infernal_shields.wielding_shield(player)
	local not_blocking = player_infernal_shield.blocking == 0

	if infernal_shield_in_hand then
		if not_blocking then
			minetest.after(0.05, function()
				if (not_blocking or not infernal_shield_in_offhand) and infernal_shield_in_hand and rmb then
					if player_infernal_shield.blocking ~= 2 then
						mcl_serverplayer.handle_blocking (player, 2)
					end
					player_infernal_shield.blocking = 2
					set_infernal_shield(player, true, 2)
				end
			end)
		elseif not infernal_shield_in_offhand then
			player_infernal_shield.blocking = 2
			if player_infernal_shield.blocking ~= 2 then
				mcl_serverplayer.handle_blocking (player, 2)
			end
		end
	elseif infernal_shield_in_offhand then
		local pointed_thing = mcl_util.get_pointed_thing(player, true)
		local wielded_stack = player:get_wielded_item()
		local offhand_can_block = (minetest.get_item_group(wielded_item(player), "bow") ~= 1
		and minetest.get_item_group(wielded_item(player), "crossbow") ~= 1)

		if pointed_thing and pointed_thing.type == "node" then
			local pointed_node = minetest.get_node(pointed_thing.under)
			if minetest.get_item_group(pointed_node.name, "container") > 1
			or is_rmb_conflicting_node(pointed_node.name)
			or is_node_stack(wielded_stack)
			then
				return
			end
		end

		if not offhand_can_block then
			return
		end
		if not_blocking then
			minetest.after(0.05, function()
				if (not_blocking or not infernal_shield_in_hand) and infernal_shield_in_offhand and rmb  and offhand_can_block then
					if player_infernal_shield.blocking ~= 1 then
						mcl_serverplayer.handle_blocking (player, 1)
					end
					player_infernal_shield.blocking = 1
					set_infernal_shield(player, true, 1)
				end
			end)
		elseif not infernal_shield_in_hand then
			if player_infernal_shield.blocking ~= 1 then
				mcl_serverplayer.handle_blocking (player, 1)
			end
			player_infernal_shield.blocking = 1
		end
	else
		if player_infernal_shield.blocking ~= 0 then
			mcl_serverplayer.handle_blocking (player, 0)
		end
		player_infernal_shield.blocking = 0
	end
end

local function update_infernal_shield_entity(player, blocking, i)
	local infernal_shield = m16b_infernal_shields.players[player].infernal_shields[i]
	if m16b_infernal_shields.wielding_shield(player, i) then
		if not infernal_shield then
			add_infernal_shield_entity(player, i)
		else
			if blocking == i then
				if infernal_shield:get_luaentity() and not infernal_shield:get_luaentity()._blocking then
					set_infernal_shield(player, true, i)
				end
			else
				set_infernal_shield(player, false, i)
			end
		end
	elseif infernal_shield then
		remove_infernal_shield_entity(player, i)
	end
end

local function add_infernal_shield_hud(infernal_shieldstack, player, blocking)
	local texture = hud
	if mcl_enchanting.is_enchanted(infernal_shieldstack:get_name()) then
		texture = texture .. overlay
	end
	local offset = 100
	if blocking == 1 then
		texture = texture .. "^[transform4"
		offset = -100
	else
		player:hud_set_flags({wielditem = false})
	end
	infernal_shield_hud[player] = player:hud_add({
		type = "image",
		position = {x = 0.5, y = 0.5},
		scale = {x = -101, y = -101},
		offset = {x = offset, y = 0},
		text = texture,
		z_index = -200,
	})
	playerphysics.add_physics_factor(player, "speed", "infernal_shield_speed", 0.5)
	set_interact(player, false)
end

local function update_infernal_shield_hud(player, blocking, infernal_shieldstack)
	local infernal_shieldhud = infernal_shield_hud[player]
	if not infernal_shieldhud then
		add_infernal_shield_hud(infernal_shieldstack, player, blocking)
		return
	end

	local wielditem = player:hud_get_flags().wielditem
	if blocking == 1 then
		if not wielditem then
			player:hud_change(infernal_shieldhud, "text", hud .. "^[transform4")
			player:hud_change(infernal_shieldhud, "offset", {x = -100, y = 0})
			player:hud_set_flags({wielditem = true})
		end
	elseif wielditem then
		player:hud_change(infernal_shieldhud, "text", hud)
		player:hud_change(infernal_shieldhud, "offset", {x = 100, y = 0})
		player:hud_set_flags({wielditem = false})
	end

	local image = player:hud_get(infernal_shieldhud).text
	local enchanted = hud .. overlay
	local enchanted1 = image == enchanted
	local enchanted2 = image == enchanted .. "^[transform4"
	if mcl_enchanting.is_enchanted(infernal_shieldstack:get_name()) then
		if not enchanted1 and not enchanted2 then
			if blocking == 1 then
				player:hud_change(infernal_shieldhud, "text", hud .. overlay .. "^[transform4")
			else
				player:hud_change(infernal_shieldhud, "text", hud .. overlay)
			end
		end
	elseif enchanted1 or enchanted2 then
		if blocking == 1 then
			player:hud_change(infernal_shieldhud, "text", hud .. "^[transform4")
		else
			player:hud_change(infernal_shieldhud, "text", hud)
		end
	end
end

minetest.register_globalstep(function()
	for player in mcl_util.connected_players() do

		handle_blocking(player)

		local blocking, infernal_shieldstack = m16b_infernal_shields.is_blocking(player)

		if blocking then
			update_infernal_shield_hud(player, blocking, infernal_shieldstack)
		elseif infernal_shield_hud[player] then --this function takes a long time. only run it when necessary
			remove_infernal_shield_hud(player)
		end

		for i = 1, 2 do
			update_infernal_shield_entity(player, blocking, i)
		end
	end
end)

minetest.register_on_dieplayer(function(player)
	set_interact(player, true)
	playerphysics.remove_physics_factor(player, "speed", "infernal_shield_speed")
	if not minetest.settings:get_bool("mcl_keepInventory") then
		remove_infernal_shield_entity(player, 1)
		remove_infernal_shield_entity(player, 2)
	end
end)

minetest.register_on_leaveplayer(function(player)
	infernal_shield_hud[player] = nil
	m16b_infernal_shields.players[player] = nil
end)

minetest.register_craft({
	output = "m16b_infernal_shields:infernal_shield",
	recipe = {
		{"mcl_nether:red_nether_brick", "mcl_nether:netherite_ingot", "mcl_nether:red_nether_brick"},
		{"mcl_nether:red_nether_brick", "mcl_nether:red_nether_brick", "mcl_nether:red_nether_brick"},
		{"", "mcl_nether:red_nether_brick", ""},
	}
})

for colorkey, colortab in pairs(mcl_banners.colors) do
	local color = colortab.color_key
	minetest.register_tool("m16b_infernal_shields:infernal_shield_" .. color, {
		description = D(colortab.color_name.." infernal_shield"), -- "Purple infernal_shield"
		_doc_items_longdesc = S("A infernal_shield is a tool used for protecting the player against attacks."),
		inventory_image = "m16b_infernal_shield_48.png^(mcl_banners_item_overlay_48.png^[colorize:" .. colortab.rgb ..")",
		wield_image = "m16b_infernal_shield_48.png",
		stack_max = 1,
		groups = {
			infernal_shield = 1,
			weapon = 1,
			enchantability = -1,
			not_in_creative_inventory = 1,
			offhand_item = 1,
		},
		sound = {breaks = "default_tool_breaks"},
		_repair_material = "group:wood",
		wield_scale = vector.new(2, 2, 2),
		_infernal_shield_color_key = colorkey,
		_mcl_wieldview_item = "",
		_mcl_generate_description = mcl_banners.update_description,
		_on_set_item_entity = function (stack)
			local meta = stack:get_meta()
			meta:set_string("m16b_infernal_shields:banner_texture", "") -- Force texture rebuild to clear wield texture.
			local pattern = meta:get_string("inventory_overlay")
			if pattern and pattern ~= "" then
				meta:set_string("wield_overlay", pattern) -- Set texture of dropped item.
			end
			return stack, {wield_item = stack:to_string()}
		end,
	})

	local banner = "mcl_banners:banner_item_" .. color
	minetest.register_craft({
		type = "shapeless",
		output = "m16b_infernal_shields:infernal_shield_" .. color,
		recipe = {"m16b_infernal_shields:infernal_shield", banner},
	})
	minetest.register_craft({
		type = "shapeless",
		output = "m16b_infernal_shields:infernal_shield_" .. color .. "_enchanted",
		recipe = {"m16b_infernal_shields:infernal_shield_enchanted", banner},
	})
end

local function craft_banner_on_infernal_shield(itemstack, player, old_craft_grid, _)
	if not string.find(itemstack:get_name(), "^m16b_infernal_shields:infernal_shield_") then
		return
	end

	local infernal_shield_stack, banner_stack
	for i = 1, player:get_inventory():get_size("craft") do
		local stack = old_craft_grid[i]
		local name = stack:get_name()
		if name ~= "" then
			if core.get_item_group(name, "infernal_shield") > 0 then
				if infernal_shield_stack then return end
				infernal_shield_stack = stack
			elseif core.get_item_group(name, "banner") > 0 then
				if banner_stack then return end
				banner_stack = stack
			else
				return
			end
			if infernal_shield_stack and banner_stack then break end
		end
	end
	if not infernal_shield_stack or not banner_stack then return end

	local b, e = mcl_banners, mcl_enchanting
	local banner_meta = banner_stack:get_meta()
	local layers = b.read_layers(banner_meta)
	if #layers > b.max_craftable_layers then
		return ItemStack("") -- Too many layers to be placed on a infernal_shield.
	end

	-- Data copy
	local item_meta, infernal_shield_meta = itemstack:get_meta(), infernal_shield_stack:get_meta()
	local banner_name, infernal_shield_name = banner_meta:get_string("name"), infernal_shield_meta:get_string("name")
	if infernal_shield_name and infernal_shield_name ~= "" then
		item_meta:set_string("name", infernal_shield_name)
	elseif banner_name and banner_name ~= "" then
		item_meta:set_string("name", banner_name)
	end
	if e.is_enchanted(infernal_shield_stack:get_name()) then
		e.set_enchantments(itemstack, e.get_enchantments(infernal_shield_stack))
	end
	set_infernal_shield_layers(itemstack, layers)
	itemstack:set_wear(infernal_shield_stack:get_wear())
end

minetest.register_craft_predict(function(itemstack, player, old_craft_grid, craft_inv)
	return craft_banner_on_infernal_shield(itemstack, player, old_craft_grid, craft_inv)
end)

minetest.register_on_craft(function(itemstack, player, old_craft_grid, craft_inv)
	return craft_banner_on_infernal_shield(itemstack, player, old_craft_grid, craft_inv)
end)

minetest.register_on_joinplayer(function(player)
	m16b_infernal_shields.players[player] = {
		infernal_shields = {},
		blocking = 0,
	}
	set_interact(player, true)
	playerphysics.remove_physics_factor(player, "speed", "infernal_shield_speed")
end)
