local S = minetest.get_translator("saw")
local REPAIR_COOLDOWN = 300 -- 5 минут в секундах
local repair_cooldowns = {}

-- Проверка наличия пилы в руке
local function is_holding_saw(player)
    if not player or not player:is_player() then return false end
    local wielded = player:get_wielded_item()
    return wielded and wielded:get_name() == "saw:saw"
end

-- Большое меню пилы
local function show_saw_menu(name)
    local player = minetest.get_player_by_name(name)
    if not player then return end
    
    if not is_holding_saw(player) then
        minetest.chat_send_player(name, minetest.colorize("#FF0000", S("Hold the saw in your hand!")))
        return
    end
    
    local cooldown_left = (repair_cooldowns[name] or 0) - os.time()
    local repair_label = S("Repair This Saw")
    
    if cooldown_left > 0 then
        repair_label = S("Repair Cooldown").."\n"..string.format("(%d:%02d)", 
                      math.floor(cooldown_left/60), cooldown_left%60)
    end
    
    local formspec = {
        "size[16,10]",
        "formspec_version[2]",
        "label[6,0.5;"..minetest.formspec_escape("Saw Control Panel").."]",
        "box[1,1;14,8;#333333]",
        "button[3,1;4,3.5;repair;"..minetest.formspec_escape(repair_label).."]",
        "button[9,2;4,1;upgrade;"..minetest.formspec_escape("Upgrade Saw").."]",
        "image[5,4;6,4;saw.png^[colorize:#0000FF:155]"
    }
    
    minetest.show_formspec(name, "saw:big_menu", table.concat(formspec))
end

-- Обработчик чат-команды
minetest.register_on_chat_message(function(name, message)
    -- Проверяем точное совпадение команды (регистронезависимо)
    if message:lower() ~= "open my saw" then return end
    
    local player = minetest.get_player_by_name(name)
    if not player then return end
    
    if not is_holding_saw(player) then
        minetest.chat_send_all("Saw: "..minetest.colorize("#FF0000", name..S(", hold the saw in your hand first")))
        return
    end
    
    minetest.chat_send_all("Saw: "..minetest.colorize("#00007F", "opened saw menu for "..name))
    minetest.after(0.1, function()
        show_saw_menu(name)
    end)
end)


-- Обработчик меню
minetest.register_on_player_receive_fields(function(player, formname, fields)
    if formname ~= "saw:big_menu" then return end
    
    local player_name = player:get_player_name()
    
    if fields.repair then
        local current_time = os.time()
        
        -- Проверка кулдауна
        if (repair_cooldowns[player_name] or 0) > current_time then
            minetest.chat_send_player(player_name, S("Repair is on cooldown!"))
            return true
        end
        
        -- Ремонт пилы
        local saw_stack = player:get_wielded_item()
        if saw_stack:get_name() == "saw:saw" then
            saw_stack:set_wear(0)
            player:set_wielded_item(saw_stack)
            repair_cooldowns[player_name] = current_time + REPAIR_COOLDOWN
            minetest.chat_send_player(player_name, minetest.colorize("#00FF00", S("Saw has been repaired!")))
            show_saw_menu(player_name)
        end
    end
    
    if fields.upgrade then
        minetest.chat_send_player(player_name, S("Upgrade function coming soon!"))
    end
    
    return true
end)

-- Автообновление кулдаунов
minetest.register_globalstep(function(dtime)
    for player_name, cooldown_end in pairs(repair_cooldowns or {}) do
        if os.time() >= cooldown_end then
            repair_cooldowns[player_name] = nil
            local player = minetest.get_player_by_name(player_name)
            if player then
                show_saw_menu(player_name)
            end
        end
    end
end)

-- Регистрация пилы
minetest.register_tool("saw:saw", {
    description = S("Saw").."\n"..minetest.colorize("#00A3FF", S("Type the secret passphrase to open the saw")).."\n"..minetest.colorize("#000068","Open my saw"),
    inventory_image = "saw.png",
    wield_image = "saw.png",
    wield_scale = {x=2, y=2, z=2},
    tool_capabilities = {
        full_punch_interval = 0.8,
        max_drop_level = 1,
        groupcaps = {
            choppy = {times={[1]=0.02, [2]=0.035, [3]=0.045}, uses=500, maxlevel=2},
            snappy = {times={[1]=0.015, [2]=0.025, [3]=0.035}, uses=500, maxlevel=2}
        },
        damage_groups = {fleshy=9},
    },
    sound = {breaks = "default_tool_breaks"},
    groups = {tool = 1}
})

-- Крафт пилы
minetest.register_craft({
    output = "saw:saw",
    recipe = {
        {"", "", "group:stick"},
        {"group:wood","group:wood","group:stick"},
        }
    })