local S = minetest.get_translator("saw")
local REPAIR_COOLDOWN = 300
local repair_cooldowns = {}
local upgrade_cost = {
    {level = 1, items = {"default:steel_ingot 2", "default:diamond 1"}},
    {level = 2, items = {"default:steelblock 1", "default:diamond 2"}},
    {level = 3, items = {"dye:blue 2","default:diamondblock 4"}},
    {level = 4, items = {"default:diamondblock 5","default:steelblock 3"}},
    {level = 5, items = {"default:mese_crystal 13","default:obsidian 2"}}
}

local function is_holding_saw(player)
    if not player or not player:is_player() then return false end
    local wielded = player:get_wielded_item()
    return wielded and (wielded:get_name():find("^saw:saw"))
end

-- Tool registrations (unchanged)
minetest.register_tool("saw:saw3", {
    description = S("Saw").."\n"..minetest.colorize("#00A3FF", "Level 3 | Type passphrase: open my saw"),
    inventory_image = "saw.png^[colorize:#0000FF:50",
    wield_image = "saw.png^[colorize:#0000FF:50",
    wield_scale = {x=2,y=2,z=2},
    tool_capabilities = {
        full_punch_interval = 0.7,
        max_drop_level = 3,
        groupcaps = {
            choppy = {times={[1]=0.02, [2]=0.035, [3]=0.045}, uses=600, maxlevel=3},
            snappy = {times={[1]=0.015, [2]=0.025, [3]=0.035}, uses=600, maxlevel=3},
            cracky = {times={[1]=0.5, [2]=0.1, [3]=0.3}, uses=700, maxlevel=3}
        },
        damage_groups = {fleshy=11},
    },
    sound = {breaks = "default_tool_breaks"},
    groups = {tool=1, not_in_creative_inventory=1}
})

minetest.register_tool("saw:saw4", {
    description = S("Saw").."\n"..minetest.colorize("#00A3FF", "Level 4 | Type passphrase: open my saw"),
    inventory_image = "saw.png^[colorize:#FF00FF:50",
    wield_image = "saw.png^[colorize:#FF00FF:50",
    wield_scale = {x=2,y=2,z=2},
    tool_capabilities = {
        full_punch_interval = 0.6,
        max_drop_level = 4,
        groupcaps = {
            choppy = {times={[1]=0.015, [2]=0.025, [3]=0.035}, uses=700, maxlevel=4},
            snappy = {times={[1]=0.01, [2]=0.02, [3]=0.03}, uses=700, maxlevel=4},
            cracky = {times={[1]=0.03, [2]=0.07, [3]=0.1}, uses=700, maxlevel=4},
            crumbly = {times={[1]=0.01, [2]=0.02, [3]=0.025}, uses=700, maxlevel=3}
        },
        damage_groups = {fleshy=13},
    },
    sound = {breaks = "default_tool_breaks"},
    groups = {tool=1, not_in_creative_inventory=1}
})

local function show_saw_menu(name)
    local player = minetest.get_player_by_name(name)
    if not player then return end
    
    if not is_holding_saw(player) then
        minetest.chat_send_player(name, minetest.colorize("#FF0000", S("Hold the saw in your hand!")))
        return
    end
    
    local cooldown_left = (repair_cooldowns[name] or 0) - os.time()
    local repair_label = S("Repair This Saw")
    
    if cooldown_left > 0 then
        repair_label = S("Repair Cooldown").."\n"..string.format("(%d:%02d)", 
                      math.floor(cooldown_left/60), cooldown_left%60)
    end
    
    local formspec = {
        "size[16,10]",
        "formspec_version[2]",
        "background[-1,-1;18,12;formspec.png]",
        "label[6,0.5;"..minetest.formspec_escape(S("Saw Control Panel")).."]",
        "image_button[3,2;4,1;formbut.png;repair;"..minetest.formspec_escape(repair_label).."]",
        "image_button[9,2;4,1;formbut.png;upgrade;"..minetest.formspec_escape(S("Upgrade Saw")).."]",
        "image[5,4;6,4;saw.png^[colorize:#0000FF:155]"
    }
    
    minetest.show_formspec(name, "saw:big_menu", table.concat(formspec))
end

minetest.register_tool("saw:saw5", {
    description = S("Saw").."\n"..minetest.colorize("#00A3FF", "Level 5 | Type passphrase: open my saw"),
    inventory_image = "saw.png^[colorize:#FF00FF:50",
    wield_image = "saw.png^[colorize:#FF00FF:50",
    wield_scale = {x=2,y=2,z=2},
    tool_capabilities = {
        full_punch_interval = 0.6,
        max_drop_level = 4,
        groupcaps = {
            choppy = {times={[1]=0.00015, [2]=0.00025, [3]=0.00035}, uses=700, maxlevel=4},
            snappy = {times={[1]=0.0001, [2]=0.0002, [3]=0.0003}, uses=700, maxlevel=4},
            cracky = {times={[1]=0.0003, [2]=0.0007, [3]=0.001}, uses=700, maxlevel=4},
            crumbly = {times={[1]=0.000001, [2]=0.0002, [3]=0.0025}, uses=700, maxlevel=3}
        },
        damage_groups = {fleshy=13},
    },
    sound = {breaks = "default_tool_breaks"},
    groups = {tool=1, not_in_creative_inventory=1}
})

local function show_upgrade_form(name)
    local player = minetest.get_player_by_name(name)
    if not player then return end
    
    local wielded = player:get_wielded_item()
    local meta = wielded:get_meta()
    local current_level = meta:get_int("level") or 1
    local next_level = current_level + 1
    
    if not upgrade_cost[next_level] then
        local maxlevel_WARN = {
            "size[16,10]",
            "formspec_version[2]",
            "background[-1,-1;18,12;formspec.png]",
            "label[4,4;"..minetest.colorize("#FF0000",S("Max level reached!")).."]",
            "image_button[15.2,-0.3;1,1;formbut.png;returnback;×]"
        }
        minetest.show_formspec(name, "saw:maxwarn", table.concat(maxlevel_WARN))
        minetest.register_on_player_receive_fields(function(formname, fields)
            if formname == "saw:maxwarn" and fields.returnback then
    minetest.close_formspec(name, formname)
    minetest.after(0.1, function()
        show_saw_menu(name)
    end)
            end
        end)
            return
        end

    local formspec = {
        "size[8,6]",
        "background[-1,-1;10,8;formspec.png]",
        "label[2.5,0.5;"..minetest.formspec_escape(S("Saw Upgrade")).."]",
        "label[1,1;"..minetest.formspec_escape(S("Current: @1", current_level)).."]",
        "label[1,1.5;"..minetest.formspec_escape(S("Next: @1", next_level)).."]",
        "box[1,2;6,0.1;#FFFFFF]",
        "label[1,2.5;"..minetest.formspec_escape(S("Required:")).."]"
    }

    local inv = player:get_inventory()
    local y = 3
    local has_all = true
    
    for _, item in ipairs(upgrade_cost[next_level].items) do
        local stack = ItemStack(item)
        local count = stack:get_count()
        local name = stack:get_name()
        local has_item = inv:contains_item("main", item)
        
        if not has_item then has_all = false end
        
        table.insert(formspec, "item_image[1,"..y..";1,1;"..name.."]")
        table.insert(formspec, "label[2,"..y..";"..count.."x "..
                      minetest.formspec_escape(minetest.registered_items[name].description).."]")
        
        table.insert(formspec, "label[6,"..y..";"..
            (has_item and minetest.colorize("#00FF00", S("Yes")) or minetest.colorize("#FF0000", S("No"))).."]")
        
        y = y + 0.8
    end

    if has_all then
        table.insert(formspec, "image_button[2.5,5;3,1;formbut.png;upgrade_confirm;"..S("Upgrade").."]")
    else
        table.insert(formspec, "label[2.5,5;"..minetest.colorize("#FFA400", S("Need more resources")).."]")
    end

    minetest.show_formspec(name, "saw:upgrade_menu", table.concat(formspec, ""))
end



minetest.register_on_chat_message(function(name, message)
    if message:lower() ~= "open my saw" then return end
    
    local player = minetest.get_player_by_name(name)
    if not player then return end
    
    if not is_holding_saw(player) then
        minetest.chat_send_all("sawwer: "..minetest.colorize("#FF0000", S("Hold the saw in your hand first")))
        return
    end
    minetest.after(0.1, function()
    minetest.chat_send_all("sawwer: "..minetest.colorize("#00007F", "Opening saw menu for "..name))
        show_saw_menu(name)
    end)
end)

minetest.register_on_player_receive_fields(function(player, formname, fields)
    local name = player:get_player_name()
    
    if formname == "saw:big_menu" then
        if fields.repair then
            local saw_stack = player:get_wielded_item()
            if is_holding_saw(player) then
                saw_stack:set_wear(0)
                player:set_wielded_item(saw_stack)
                repair_cooldowns[name] = os.time() + REPAIR_COOLDOWN
                minetest.chat_send_player(name, minetest.colorize("#00FF00", S("Saw has been repaired!")))
                show_saw_menu(name)
            end
        elseif fields.upgrade then
            show_upgrade_form(name)
        end
        return true
    end
    
    if formname == "saw:upgrade_menu" and fields.upgrade_confirm then
        local inv = player:get_inventory()
        local wielded = player:get_wielded_item()
        local meta = wielded:get_meta()
        local current_level = meta:get_int("level") or 1
        local next_level = current_level + 1
        
        -- Check if player has all required items
        for _, item in ipairs(upgrade_cost[next_level].items) do
            if not inv:contains_item("main", item) then
                minetest.chat_send_player(name, minetest.colorize("#FF0000", S("Resources changed!")))
                return true
            end
        end
        
        -- Remove required items
        for _, item in ipairs(upgrade_cost[next_level].items) do
            inv:remove_item("main", item)
        end
        
        -- Create the new saw tool based on level
        local new_saw
        if next_level == 2 then
            new_saw = ItemStack("saw:saw")
        elseif next_level == 3 then
            new_saw = ItemStack("saw:saw3")
        elseif next_level == 4 then
            new_saw = ItemStack("saw:saw4")
        elseif next_level == 5 then
            new_saw = ItemStack("saw:saw5")
        else
            new_saw = ItemStack("saw:saw")
        end
        
        -- Set metadata for the new saw
        local new_meta = new_saw:get_meta()
        new_meta:set_int("level", next_level)
        new_meta:set_string("description", S("Saw").."\n"..minetest.colorize("#00A3FF","Level "..next_level.." | Type passphrase: open my saw"))
        
        -- Give the player the new saw
        player:set_wielded_item(new_saw)
        minetest.chat_send_player(name, minetest.colorize("#00FF00", 
            S("Saw upgraded to level @1!",next_level)))
        show_saw_menu(name)
        return true
    end
end)

minetest.register_globalstep(function(dtime)
    local current_time = os.time()
    for player_name, cooldown_end in pairs(repair_cooldowns) do
        if current_time >= cooldown_end then
            repair_cooldowns[player_name] = nil
            local player = minetest.get_player_by_name(player_name)
            if player then
                show_saw_menu(player_name)
            end
        end
    end
end)

minetest.register_tool("saw:saw", {
    description = S("Saw").."\n"..minetest.colorize("#00A3FF", S("Level 1 | Type passphrase: open my saw")),
    inventory_image = "saw.png",
    wield_image = "saw.png",
    wield_scale = {x=2, y=2, z=2},
    tool_capabilities = {
        full_punch_interval = 0.8,
        max_drop_level = 1,
        groupcaps = {
            choppy = {times={[1]=0.02, [2]=0.035, [3]=0.045}, uses=500, maxlevel=2},
            snappy = {times={[1]=0.015, [2]=0.025, [3]=0.035}, uses=500, maxlevel=2}
        },
        damage_groups = {fleshy=9},
    },
    sound = {breaks = "default_tool_breaks"},
    groups = {tool = 1}
})

minetest.register_craft({
    output = "saw:saw",
    recipe = {
        {"", "", "group:stick"},
        {"group:wood","group:wood","group:stick"}
    }
})