-- IndustrialTest
-- Copyright (C) 2023 mrkubax10

-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

local S=minetest.get_translator("industrialtest")

local function onTreetapUse(user,pointed)
	local node=minetest.get_node_or_nil(pointed.under)
	if not node then
		return false
	end
	-- Note: if more nodes from which treetap can extract will be added then they shouldn't be added here
	-- Instead they should have additional entry in definition which will denote that treetap can be used on them
	if node.name=="industrialtest:rubber_wood_with_rubber" then
		local inv=user:get_inventory()
		inv:add_item("main",ItemStack("industrialtest:sticky_resin"))
		minetest.set_node(pointed.under,{name="industrialtest:rubber_wood"})
		return true
	end
	return false
end
local definition={	
	description=S("Treetap"),
	inventory_image="industrialtest_treetap.png",
	tool_capabilities={
		full_punch_interval=1,
		uses=50
	},
	on_place=function(itemstack,user,pointed)
		if pointed.type=="node" and user and user:is_player() and onTreetapUse(user,pointed) then
			industrialtest.api.afterToolUse(itemstack)
			return itemstack
		end
		return nil
	end,
	_industrialtest_tool=true
}
if industrialtest.mtgAvailable then
	definition.groups={
		flammable=2
	}
elseif industrialtest.mclAvailable then
	definition.groups={
		tool=1
	}
	definition._repair_material="group:wood"
	definition._mcl_toollike_wield=true
end
minetest.register_tool("industrialtest:treetap",definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:treetap",
	recipe={
		{"","group:wood",""},
		{"group:wood","group:wood","group:wood"},
		{"group:wood","",""}
	}
})
definition={
	description=S("Electric Treetap"),
	inventory_image="industrialtest_electric_treetap.png",
	on_place=function(itemstack,user,pointed)
		local meta=itemstack:get_meta()
		if meta:get_int("industrialtest.powerAmount")>=50 and user and user:is_player() and onTreetapUse(user,pointed) then
			industrialtest.api.addPowerToItem(itemstack,-50)
			return itemstack
		end
		return nil
	end,
	_industrialtest_powerStorage=true,
	_industrialtest_powerCapacity=10000,
	_industrialtest_powerFlow=industrialtest.api.lvPowerFlow
}
if industrialtest.mclAvailable then
	definition.groups={
		tool=1
	}
	definition._mcl_toollike_wield=true
end
minetest.register_tool("industrialtest:electric_treetap",definition)
minetest.register_craft({
	type="shapeless",
	output="industrialtest:electric_treetap",
	recipe={
		"industrialtest:treetap",
		"industrialtest:electronic_circuit",
		"industrialtest:re_battery"
	}
})

local function onWrenchUse(user,pointed)
	local node=minetest.get_node_or_nil(pointed.under)
	if not node then
		return false
	end
	local def=minetest.registered_nodes[node.name]
	if not def or not def.groups or not def.groups._industrialtest_wrenchUnmountable or (def.can_dig and not def.can_dig(pointed.under)) then
		return false
	end
	local inv=user:get_inventory()
	if def.after_dig_node then
		def.after_dig_node(pointed.under,node,minetest.get_meta(pointed.under):to_table())
	end
	minetest.remove_node(pointed.under)
	local name=node.name
	if string.sub(name,-7)=="_active" then
		name=string.sub(name,1,-8)
	end
	inv:add_item("main",ItemStack(name))
	return true
end
definition={
	description=S("Wrench"),
	inventory_image="industrialtest_wrench.png",
	tool_capabilities={
		full_punch_interval=1,
		uses=200
	},
	on_use=function(itemstack,user,pointed)
		if pointed.type=="node" and user and user:is_player() and onWrenchUse(user,pointed) then
			industrialtest.api.afterToolUse(itemstack)
			return itemstack
		end
		return nil
	end,
	_industrialtest_tool=true
}
if industrialtest.mclAvailable then
	definition.groups={
		tool=1
	}
	definition._mcl_toollike_wield=true
end
minetest.register_tool("industrialtest:wrench",definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:wrench",
	recipe={
		{industrialtest.elementKeys.bronzeIngot,"",industrialtest.elementKeys.bronzeIngot},
		{industrialtest.elementKeys.bronzeIngot,industrialtest.elementKeys.bronzeIngot,industrialtest.elementKeys.bronzeIngot},
		{"",industrialtest.elementKeys.bronzeIngot,""}
	}
})
definition={
	description=S("Electric Wrench"),
	inventory_image="industrialtest_electric_wrench.png",
	on_use=function(itemstack,user,pointed)
		local meta=itemstack:get_meta()
		if meta:get_int("industrialtest.powerAmount")>=20 and user and user:is_player() and onWrenchUse(user,pointed) then
			industrialtest.api.addPowerToItem(itemstack,-20)
			return itemstack
		end
		return nil
	end,
	_industrialtest_powerStorage=true,
	_industrialtest_powerCapacity=10000,
	_industrialtest_powerFlow=industrialtest.api.lvPowerFlow
}
if industrialtest.mclAvailable then
	definition.groups={
		tool=1
	}
	definition._mcl_toollike_wield=true
end
minetest.register_tool("industrialtest:electric_wrench",definition)
minetest.register_craft({
	type="shapeless",
	output="industrialtest:electric_wrench",
	recipe={
		"industrialtest:wrench",
		"industrialtest:electronic_circuit",
		"industrialtest:re_battery"
	}
})

local registeredElectricChainsaws={}
local function registerElectricChainsaw(config)
	local definition={
		description=S(config.displayName),
		inventory_image="industrialtest_"..config.name..".png",
		_industrialtest_powerStorage=true,
		_industrialtest_powerCapacity=10000,
		_industrialtest_powerFlow=industrialtest.api.lvPowerFlow
	}
	if industrialtest.mtgAvailable then
		definition.tool_capabilities={
			full_punch_interval=0.5,
			max_drop_level=config.maxDropLevel,
			groupcaps={
				choppy={
					times=config.inactiveTimes,
					maxlevel=config.maxLevel
				}
			}
		}
		definition.groups={
			axe=1
		}
	elseif industrialtest.mclAvailable then
		definition.tool_capabilities={
			full_punch_interval=0.5,
			max_drop_level=config.maxDropLevel,
		}
		definition.groups={
			tool=1,
			dig_speed_class=config.digSpeedClass,
		}
		definition.on_place=function(itemstack,user,pointed)
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 then
				local itemstackCopy=itemstack
				if itemstack:get_wear()~=industrialtest.internal.mclMakeStrippedTrunk(itemstackCopy,user,pointed,true):get_wear() then
					industrialtest.api.addPowerToItem(itemstack,-20)
					return itemstack
				end
			end
			return nil
		end
		definition.after_use=function(itemstack)
			-- Hack to make sure that chainsaw won't be destroyed when has 0 EU in MCL
			return nil
		end
		definition._mcl_diggroups={
			axey={
				speed=config.inactiveDigSpeed,
				level=config.digLevel,
				uses=-1
			}
		}
		definition._mcl_toollike_wield=true
	end
	minetest.register_tool("industrialtest:"..config.name,definition)
	definition=table.copy(definition)
	if industrialtest.mtgAvailable then
		definition.tool_capabilities.groupcaps.choppy.times=config.activeTimes
	elseif industrialtest.mclAvailable then
		definition._mcl_diggroups.axey.speed=config.activeDigSpeed
	end
	definition.groups.not_in_creative_inventory=1
	definition.on_use=nil
	definition.after_use=function(itemstack,user,node,digparams)
		industrialtest.api.addPowerToItem(itemstack,-20)
		itemstack:set_name("industrialtest:"..config.name)
		return itemstack
	end
	minetest.register_tool("industrialtest:"..config.name.."_active",definition)
	registeredElectricChainsaws["industrialtest:"..config.name]=true
end
definition={
	name="electric_chainsaw",
	displayName="Electric Chainsaw"
}
if industrialtest.mtgAvailable then
	definition.maxDropLevel=1
	definition.inactiveTimes={[1]=10,[2]=5.6,[3]=4}
	definition.maxLevel=2
	definition.activeTimes={[1]=2.2,[2]=1.1,[3]=0.7}
elseif industrialtest.mclAvailable then
	definition.digSpeedClass=4
	definition.maxDropLevel=4
	definition.inactiveDigSpeed=1
	definition.digLevel=4
	definition.activeDigSpeed=7
end
registerElectricChainsaw(definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:electric_chainsaw",
	recipe={
		{"","industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot"},
		{"industrialtest:refined_iron_ingot","industrialtest:electronic_circuit","industrialtest:refined_iron_ingot"},
		{"industrialtest:re_battery","industrialtest:refined_iron_ingot",""}
	}
})
definition={
	name="diamond_electric_chainsaw",
	displayName="Diamond Electric Chainsaw"
}
if industrialtest.mtgAvailable then
	definition.maxDropLevel=1
	definition.inactiveTimes={[1]=10,[2]=5.6,[3]=4}
	definition.maxLevel=3
	definition.activeTimes={[1]=2.0,[2]=0.8,[3]=0.4}
elseif industrialtest.mclAvailable then
	definition.digSpeedClass=5
	definition.maxDropLevel=5
	definition.inactiveDigSpeed=1
	definition.digLevel=5
	definition.activeDigSpeed=9
end
registerElectricChainsaw(definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:diamond_electric_chainsaw",
	recipe={
		{"",industrialtest.elementKeys.diamond,""},
		{industrialtest.elementKeys.diamond,"industrialtest:electric_chainsaw",industrialtest.elementKeys.diamond}
	}
})

local registeredElectricHoes={}
local function registerElectricHoe(config)
	local definition={
		description=S(config.displayName),
		inventory_image="industrialtest_"..config.name..".png",
		_industrialtest_powerStorage=true,
		_industrialtest_powerCapacity=10000,
		_industrialtest_powerFlow=industrialtest.api.lvPowerFlow
	}
	if industrialtest.mtgAvailable then
		-- Taken and adapted from farming mod from Minetest Game
		local function onUse(user,pointed)
			local pt = pointed
			-- check if pointing at a node
			if not pt then
				return false
			end
			if pt.type ~= "node" then
				return false
			end

			local under = minetest.get_node(pt.under)
			local p = {x=pt.under.x, y=pt.under.y+1, z=pt.under.z}
			local above = minetest.get_node(p)

			-- return if any of the nodes is not registered
			if not minetest.registered_nodes[under.name] then
				return false
			end
			if not minetest.registered_nodes[above.name] then
				return false
			end

			-- check if the node above the pointed thing is air
			if above.name ~= "air" then
				return false
			end

			-- check if pointing at soil
			if minetest.get_item_group(under.name, "soil") ~= 1 then
				return false
			end

			-- check if (wet) soil defined
			local regN = minetest.registered_nodes
			if regN[under.name].soil == nil or regN[under.name].soil.wet == nil or regN[under.name].soil.dry == nil then
				return false
			end

			local player_name = user and user:get_player_name() or ""

			if minetest.is_protected(pt.under, player_name) then
				minetest.record_protection_violation(pt.under, player_name)
				return false
			end
			if minetest.is_protected(pt.above, player_name) then
				minetest.record_protection_violation(pt.above, player_name)
				return false
			end

			-- turn the node into soil and play sound
			minetest.set_node(pt.under, {name = regN[under.name].soil.dry})
			minetest.sound_play("default_dig_crumbly", {
				pos = pt.under,
				gain = 0.3,
			}, true)

			return true
		end
		definition.groups={
			hoe=1
		}
		definition.on_use=function(itemstack,user,pointed)
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 and onUse(user,pointed) then
				industrialtest.api.addPowerToItem(itemstack,-20)
				return itemstack
			end
			return nil
		end
	elseif industrialtest.mclAvailable then
		-- Taken from https://git.minetest.land/MineClone2/MineClone2/src/branch/master/mods/ITEMS/mcl_farming/hoes.lua#L3
		local function createSoil(pos)
			if pos == nil then
				return false
			end
			local node = minetest.get_node(pos)
			local name = node.name
			local above = minetest.get_node({x=pos.x, y=pos.y+1, z=pos.z})
			if minetest.get_item_group(name, "cultivatable") == 2 then
				if above.name == "air" then
					node.name = "mcl_farming:soil"
					minetest.set_node(pos, node)
					minetest.sound_play("default_dig_crumbly", { pos = pos, gain = 0.5 }, true)
					return true
				end
			elseif minetest.get_item_group(name, "cultivatable") == 1 then
				if above.name == "air" then
					node.name = "mcl_core:dirt"
					minetest.set_node(pos, node)
					minetest.sound_play("default_dig_crumbly", { pos = pos, gain = 0.6 }, true)
					return true
				end
			end
			return false
		end
		definition.tool_capabilities={
			full_punch_interval=0.5
		}
		definition.groups={
			tool=1
		}
		definition.on_place=function(itemstack,user,pointed)
			local node=minetest.get_node(pointed.under)
			if user and not user:get_player_control().sneak then
				if minetest.registered_nodes[node.name] and minetest.registered_nodes[node.name].on_rightclick then
					return minetest.registered_nodes[node.name].on_rightclick(pointed.under,node,user,itemstack) or itemstack
				end
			end
			if minetest.is_protected(pointed.under, user:get_player_name()) then
				minetest.record_protection_violation(pointed.under,user:get_player_name())
				return nil
			end
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 and createSoil(pointed.under) then
				industrialtest.api.addPowerToItem(itemstack,-20)
				return itemstack
			end
			return nil
		end
		definition.after_use=function(itemstack)
			-- Hack to make sure that chainsaw won't be destroyed when has 0 EU in MCL
			return nil
		end
		definition._mcl_toollike_wield=true
		definition._mcl_diggroups={
			hoey={
				speed=config.inactiveDigSpeed,
				level=config.digLevel,
				uses=-1
			}
		}
	end
	minetest.register_tool("industrialtest:"..config.name,definition)
	if industrialtest.mclAvailable then
		definition=table.copy(definition)
		definition._mcl_diggroups.hoey.speed=config.activeDigSpeed
		definition.groups.not_in_creative_inventory=1
		definition.after_use=function(itemstack)
			industrialtest.api.addPowerToItem(itemstack,-20)
			itemstack:set_name("industrialtest:"..config.name)
			return itemstack
		end
		minetest.register_tool("industrialtest:"..config.name.."_active",definition)
	end
	registeredElectricHoes["industrialtest:"..config.name]=true
end
definition={
	name="electric_hoe",
	displayName="Electric Hoe"
}
if industrialtest.mclAvailable then
	definition.inactiveDigSpeed=1
	definition.digLevel=2
	definition.activeDigSpeed=7
end
registerElectricHoe(definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:electric_hoe",
	recipe={
		{"industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot"},
		{"","industrialtest:electronic_circuit"},
		{"","industrialtest:re_battery"}
	}
})
minetest.register_craft({
	type="shaped",
	output="industrialtest:electric_hoe",
	recipe={
		{"industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot"},
		{"industrialtest:electronic_circuit",""},
		{"industrialtest:re_battery",""}
	}
})
if industrialtest.mclAvailable then
	registerElectricHoe({
		name="diamond_electric_hoe",
		displayName="Diamond Electric Hoe",
		inactiveDigSpeed=1,
		digLevel=5,
		activeDigSpeed=9
	})
	minetest.register_craft({
		type="shapeless",
		output="industrialtest:diamond_electric_hoe",
		recipe={
			"industrialtest:electric_hoe",
			industrialtest.elementKeys.diamond,
			industrialtest.elementKeys.diamond
		}
	})
end

local registeredElectricDrills={}
local function registerElectricDrill(config)
	local definition={
		description=S(config.displayName),
		inventory_image="industrialtest_"..config.name..".png",
		_industrialtest_powerStorage=true,
		_industrialtest_powerCapacity=10000,
		_industrialtest_powerFlow=industrialtest.api.lvPowerFlow
	}
	if industrialtest.mtgAvailable then
		definition.tool_capabilities={
			full_punch_interval=0.5,
			max_drop_level=config.maxDropLevel,
			groupcaps={
				cracky={
					times=config.inactiveTimes,
					maxlevel=config.maxLevel
				},
				crumbly={
					times=config.inactiveTimes,
					maxlevel=config.maxLevel
				}
			}
		}
		definition.groups={
			pickaxe=1,
			shovel=1
		}
	elseif industrialtest.mclAvailable then
		definition.tool_capabilities={
			full_punch_interval=0.5,
			max_drop_level=config.maxDropLevel
		}
		definition.groups={
			tool=1,
			dig_speed_class=config.digSpeedClass
		}
		definition.after_use=function()
			-- Hack to make sure that drill won't be destroyed when has 0 EU in MCL
			return nil
		end
		definition._mcl_diggroups={
			pickaxey={
				speed=config.inactiveDigSpeed,
				level=config.digLevel,
				uses=-1
			},
			shovely={
				speed=config.inactiveDigSpeed,
				level=config.digLevel,
				uses=-1
			}
		}
		definition._mcl_toollike_wield=true
	end
	minetest.register_tool("industrialtest:"..config.name,definition)
	definition=table.copy(definition)
	if industrialtest.mtgAvailable then
		definition.tool_capabilities.groupcaps.cracky.times=config.activeTimes
		definition.tool_capabilities.groupcaps.crumbly.times=config.activeTimes
	elseif industrialtest.mclAvailable then
		definition._mcl_diggroups.pickaxey.speed=config.activeDigSpeed
		definition._mcl_diggroups.shovely.speed=config.activeDigSpeed
	end
	definition.groups.not_in_creative_inventory=1
	definition.after_use=function(itemstack)
		industrialtest.api.addPowerToItem(itemstack,-20)
		itemstack:set_name("industrialtest:"..config.name)
		return itemstack
	end
	minetest.register_tool("industrialtest:"..config.name.."_active",definition)
	registeredElectricDrills["industrialtest:"..config.name]=true
end
definition={
	name="electric_drill",
	displayName="Electric Drill"
}
if industrialtest.mtgAvailable then
	definition.maxDropLevel=1
	definition.inactiveTimes={[1]=10,[2]=5.6,[3]=4}
	definition.maxLevel=2
	definition.activeTimes={[1]=2.0,[2]=0.8,[3]=0.4}
elseif industrialtest.mclAvailable then
	definition.digSpeedClass=4
	definition.maxDropLevel=4
	definition.inactiveDigSpeed=1
	definition.digLevel=4
	definition.activeDigSpeed=7
end
registerElectricDrill(definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:electric_drill",
	recipe={
		{"","industrialtest:refined_iron_ingot",""},
		{"industrialtest:refined_iron_ingot","industrialtest:electronic_circuit","industrialtest:refined_iron_ingot"},
		{"industrialtest:refined_iron_ingot","industrialtest:re_battery","industrialtest:refined_iron_ingot"}
	}
})
definition={
	name="diamond_electric_drill",
	displayName="Diamond Electric Drill"
}
if industrialtest.mtgAvailable then
	definition.maxDropLevel=1
	definition.inactiveTimes={[1]=10,[2]=5.6,[3]=4}
	definition.maxLevel=3
	definition.activeTimes={[1]=2.0,[2]=0.8,[3]=0.4}
elseif industrialtest.mclAvailable then
	definition.digSpeedClass=5
	definition.maxDropLevel=5
	definition.inactiveDigSpeed=1
	definition.digLevel=5
	definition.activeDigSpeed=9
end
registerElectricDrill(definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:diamond_electric_drill",
	recipe={
		{"",industrialtest.elementKeys.diamond,""},
		{industrialtest.elementKeys.diamond,"industrialtest:electric_drill",industrialtest.elementKeys.diamond}
	}
})

local registeredElectricSabers={}
local function registerElectricSaber(config)
	local definition={
		description=S(config.displayName),
		inventory_image="industrialtest_"..config.name..".png",
		_industrialtest_powerStorage=true,
		_industrialtest_powerCapacity=10000,
		_industrialtest_powerFlow=industrialtest.api.lvPowerFlow
	}
	if industrialtest.mtgAvailable then
		definition.groups={
			sword=1
		}
		definition.tool_capabilities={
			full_punch_interval=0.5,
			max_drop_level=config.maxDropLevel,
			groupcaps={
				snappy={
					times=config.inactiveTimes,
					maxlevel=config.maxLevel
				}
			},
			damage_groups={fleshy=config.inactiveDamage}
		}
	elseif industrialtest.mclAvailable then
		definition.groups={
			weapon=1,
			sword=1,
			dig_speed_class=config.digSpeedClass
		}
		definition.tool_capabilities={
			full_punch_interval=0.5,
			max_drop_level=config.maxDropLevel,
			damage_groups={fleshy=config.inactiveDamage}
		}
		definition.after_use=function()
			-- Hack to make sure that saber won't be destroyed when has 0 EU in MCL
			return nil
		end
		definition._mcl_toollike_wield=true
		definition._mcl_diggroups={
			swordy={
				speed=config.inactiveDigSpeed,
				level=config.digLevel,
				uses=-1
			},
			swordy_cobweb={
				speed=config.inactiveDigSpeed,
				level=config.digLevel,
				uses=-1
			}
		}
	end
	minetest.register_tool("industrialtest:"..config.name,definition)
	definition=table.copy(definition)
	if industrialtest.mtgAvailable then
		definition.tool_capabilities.groupcaps.snappy.times=config.activeTimes
		definition.tool_capabilities.damage_groups.fleshy=config.activeDamage
	elseif industrialtest.mclAvailable then
		definition.tool_capabilities.damage_groups.fleshy=config.activeDamage
		definition._mcl_diggroups.swordy.speed=config.activeDigSpeed
		definition._mcl_diggroups.swordy_cobweb.speed=config.activeDigSpeed
	end
	definition.groups.not_in_creative_inventory=1
	definition.after_use=function()
		return nil
	end
	minetest.register_tool("industrialtest:"..config.name.."_active",definition)
	registeredElectricSabers["industrialtest:"..config.name]=true
end
definition={
	name="electric_saber",
	displayName="Electric Saber"
}
if industrialtest.mtgAvailable then
	definition.maxDropLevel=1
	definition.inactiveTimes={[1]=10,[2]=5.6,[3]=4}
	definition.maxLevel=3
	definition.inactiveDamage=1
	definition.activeTimes={[1]=2.0,[2]=0.8,[3]=0.4}
	definition.activeDamage=6
elseif industrialtest.mclAvailable then
	definition.digSpeedClass=4
	definition.maxDropLevel=4
	definition.inactiveDamage=1
	definition.inactiveDigSpeed=1
	definition.digLevel=4
	definition.activeDamage=6
	definition.activeDigSpeed=7
end
registerElectricSaber(definition)
minetest.register_craft({
	type="shaped",
	output="industrialtest:electric_saber",
	recipe={
		{"industrialtest:refined_iron_ingot"},
		{"industrialtest:refined_iron_ingot"},
		{"industrialtest:re_battery"}
	}
})
definition={
	name="diamond_electric_saber",
	displayName="Diamond Electric Saber"
}
if industrialtest.mtgAvailable then
	definition.maxDropLevel=1
	definition.inactiveTimes={[1]=10,[2]=5.6,[3]=4}
	definition.maxLevel=3
	definition.inactiveDamage=1
	definition.activeTimes={[1]=2.0,[2]=0.8,[3]=0.4}
	definition.activeDamage=9
elseif industrialtest.mclAvailable then
	definition.digSpeedClass=5
	definition.maxDropLevel=5
	definition.inactiveDamage=1
	definition.inactiveDigSpeed=1
	definition.digLevel=5
	definition.activeDamage=9
	definition.activeDigSpeed=9
end
registerElectricSaber(definition)
minetest.register_craft({
	type="shapeless",
	output="industrialtest:diamond_electric_saber",
	recipe={
		"industrialtest:electric_saber",
		industrialtest.elementKeys.diamond,
		industrialtest.elementKeys.diamond
	}
})

-- Tool callbacks
minetest.register_on_punchnode(function(pos,node,user,pointed)
	if user then
		local itemstack=user:get_wielded_item()
		if registeredElectricChainsaws[itemstack:get_name()] then
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 then
				local def=minetest.registered_nodes[node.name]
				if (industrialtest.mtgAvailable and def.groups and def.groups.choppy) or (industrialtest.mclAvailable and def.groups and def.groups.axey) then
					itemstack:set_name(itemstack:get_name().."_active")
					user:set_wielded_item(itemstack)
				end
			end
		elseif industrialtest.mclAvailable and registeredElectricHoes[itemstack:get_name()] then
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 then
				local def=minetest.registered_nodes[node.name]
				if def.groups and def.groups.hoey then
					itemstack:set_name(itemstack:get_name().."_active")
					user:set_wielded_item(itemstack)
				end
			end
		elseif registeredElectricDrills[itemstack:get_name()] then
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 then
				local def=minetest.registered_nodes[node.name]
				if (industrialtest.mtgAvailable and def.groups and (def.groups.cracky or def.groups.crumbly)) or (industrialtest.mclAvailable and def.groups and (def.groups.pickaxey or def.groups.shovely)) then
					itemstack:set_name(itemstack:get_name().."_active")
					user:set_wielded_item(itemstack)
				end
			end
		elseif registeredElectricSabers[itemstack:get_name()] then
			local meta=itemstack:get_meta()
			if meta:get_int("industrialtest.powerAmount")>=20 then
				local def=minetest.registered_nodes[node.name]
				if (industrialtest.mtgAvailable and def.groups and def.groups.snappy) or (industrialtest.mclAvailable and def.groups and (def.groups.swordy or def.groups.swordy_cobweb)) then
					itemstack:set_name(itemstack:get_name().."_active")
					user:set_wielded_item(itemstack)
				end
			end
		end
	end
end)
minetest.register_on_punchplayer(function(player,hitter)
	local itemstack=hitter:get_wielded_item()
	if registeredElectricSabers[itemstack:get_name()] then
		local meta=itemstack:get_meta()
		if meta:get_int("industrialtest.powerAmount")>=20 then
			industrialtest.api.addPowerToItem(itemstack,-20)
			hitter:set_wielded_item(itemstack)
			local def=minetest.registered_tools[itemstack:get_name().."_active"]
			player:set_hp(player:get_hp()-def.tool_capabilites.damage_groups.fleshy)
			return true
		end
	end
	return false
end)
