-- IndustrialTest
-- Copyright (C) 2024 mrkubax10

-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

local S=minetest.get_translator("industrialtest")

function industrialtest.api.addTag(name,tag)
	if not industrialtest.api.tags[tag] then
		industrialtest.api.tags[tag]={}
	end
	table.insert(industrialtest.api.tags[tag],name)
end

-- \brief Registers dust of certain resource
-- \param name Technical name of resource
-- \param displayName Display name of resource
-- \param resources List of tables with following keys: <output>, <recipe>, [count(1)]
--                                                      <> - required, [] - optional, () - default value
-- \param color HTML color of dust
-- \param registerMaceratorRecipe If true macerator recipe for dust will be registered
-- \returns nil
function industrialtest.api.registerResourceDust(name,displayName,resources,color,registerMaceratorRecipe)
	minetest.register_craftitem("industrialtest:"..name.."_dust",{
		description=S(displayName.." Dust"),
		inventory_image="industrialtest_dust.png",
		color=color
	})
	if registerMaceratorRecipe then
		for _,value in ipairs(resources) do
			industrialtest.api.registerMaceratorRecipe({
				output="industrialtest:"..name.."_dust "..(value.count or 1),
				recipe=value.resource
			})
		end
	end
end

-- \brief Registers plate of certain resource
-- \param name Technical name of resource
-- \param displayName Display name of resource
-- \param resources List of tables with following keys: <output>, <recipe>, [count(1)]
--                                                      <> - required, [] - optional, () - default value
-- \param color HTML color of plate
-- \param registerCompressorRecipe If true compressor recipe for plate will be registered
-- \returns nil
function industrialtest.api.registerPlate(name,displayName,resources,color,registerCompressorRecipe)
	minetest.register_craftitem("industrialtest:"..name,{
		description=displayName,
		inventory_image="industrialtest_plate.png",
		inventory_overlay="industrialtest_plate_overlay.png",
		color=color
	})
	if registerCompressorRecipe then
		for _,value in ipairs(resources) do
			industrialtest.api.registerCompressorRecipe({
				output="industrialtest:"..name.." "..(value.count or 1),
				recipe=value.resource
			})
		end
	end
end

-- \brief Registers cell with certain fluid
-- \param name Technical name of cell
-- \param displayName Display name of cell
-- \param node Node which can be picked up with this cell
-- \returns nil
function industrialtest.api.registerStorageCell(name,displayName,node,modname,color)
	color = color or "#ffffffff"
	if not modname then
		modname="industrialtest"
	end
	minetest.register_craftitem("industrialtest:"..name.."_cell",{
		description=S(displayName.." Cell"),
		inventory_image="industrialtest_cell_fluid.png",
		inventory_overlay="industrialtest_cell_casing.png",
		color=color,
		on_place=function(itemstack,user,pointed)
			if pointed.type~="node" or not user or not user:is_player() then
				return nil
			end
			local node=minetest.get_node_or_nil(pointed.above)
			local storage=industrialtest.api.getStorageCell("industrialtest:"..name.."_cell")
			if storage.node then
				if node.name~="air" and node.name~=storage.node then
					return nil
				end
				minetest.set_node(pointed.above,{name=storage.node})
				if itemstack:get_count()==1 then
					itemstack:set_name("industrialtest:empty_cell")
				else
					local inv=user:get_inventory()
					inv:add_item("main",ItemStack("industrialtest:empty_cell"))
					itemstack:take_item()
				end
				return itemstack
			end
			return nil
		end
	})
	industrialtest.api.storageCells["industrialtest:"..name.."_cell"]={
		name="industrialtest:"..name.."_cell",
		node=node
	}
end

-- \brief Returns registred storage cell by name
-- \param name Storage cell name
-- \returns Table with following keys: name, node or nil in case of failure
function industrialtest.api.getStorageCell(name)
	return industrialtest.api.storageCells[name]
end

-- \brief Returns registered storage cells by node
-- \param node Node ID
-- \returns Table with following keys: name, node or nil in case of failure
function industrialtest.api.getStorageCellByNode(node)
	for _,value in pairs(industrialtest.api.storageCells) do
		if value.node==node then
			return value
		end
	end
	return nil
end

-- \brief Registers macerator recipe
-- \param config Table with keys: <output>, <recipe>, [time(2)]
-- \returns nil
function industrialtest.api.registerMaceratorRecipe(config)
	local definition={
		output=config.output or "",
		recipe=config.recipe or "",
		time=config.time or 2
	}
	industrialtest.api.maceratorRecipes[definition.recipe]=definition
end

-- \brief Returns macerator recipe result
-- \param recipe String ID of resulting conten
-- \returns Table with following keys: output, recipe, time
function industrialtest.api.getMaceratorRecipeResult(recipe)
	return industrialtest.api.maceratorRecipes[recipe]
end

-- \brief Registers compressor recipe
-- \param config Table with following keys: <output>, <recipe>, [time(2)], [count(1)]
-- \returns nil
function industrialtest.api.registerCompressorRecipe(config)
	local definition={
		output=config.output or "",
		recipe=config.recipe or "",
		time=config.time or 2,
		count=config.count or 1
	}
	industrialtest.api.compressorRecipes[definition.recipe]=definition
end

-- \brief Returns macerator recipe result
-- \param recipe String ID of resulting conten
-- \returns Table with following keys: output, recipe, time
function industrialtest.api.getCompressorRecipeResult(recipe)
	return industrialtest.api.compressorRecipes[recipe]
end

function industrialtest.api.registerExtractorRecipe(config)
	local definition={
		output=config.output or "",
		recipe=config.recipe or "",
		time=config.time or 2
	}
	industrialtest.api.extractorRecipes[definition.recipe]=definition
end

function industrialtest.api.getExtractorRecipeResult(recipe)
	return industrialtest.api.extractorRecipes[recipe]
end

function industrialtest.api.registerCableFormerRecipe(config)
	local definition={
		output=config.output or "",
		recipe=config.recipe or "",
		time=config.time or 2
	}
	industrialtest.api.cableFormerRecipes[definition.recipe]=definition
end

function industrialtest.api.getCableFormerRecipeResult(recipe)
	return industrialtest.api.cableFormerRecipes[recipe]
end

-- \brief Registers fuel that can be used in geothermal generator
-- \param fuel Table with following keys: <name>, <calorificValue>, <storageItems>
--        which is a table containing items which are tables with following keys: <name>, <leftover>
-- \returns nil
function industrialtest.api.registerGeothermalGeneratorFuel(config)
	local definition={
		name=config.name or "",
		calorificValue=config.calorificValue or 0,
		texture=config.texture or "industrialtest_gui_fluid_bg.png",
		storageItems=config.storageItems or {}
	}
	industrialtest.api.geothermalGeneratorFuels[definition.name]=definition
end

-- \brief Returns generator fuel information
-- \param name Name of fuel
-- \returns Table with following keys: name, calorificValue, storageItems
function industrialtest.api.getGeothermalGeneratorFuel(name)
	return industrialtest.api.geothermalGeneratorFuels[name]
end

-- \brief Returns generator fuel information by item name
-- \param name ID of item
-- \returns Table with following keys: name, calorificValue, storageItems or nil in case of failure
function industrialtest.api.getGeothermalGeneratorFuelByItem(name)
	for _,value in pairs(industrialtest.api.geothermalGeneratorFuels) do
		for _,item in ipairs(value.storageItems) do
			if item.name==name then
				return value
			end
		end
	end
	return nil
end

-- \brief Registers fuel that can be used in water mill
-- \param fuel Table with following keys: <name>, <calorificValue>, <storageItems>
--        which is a table containing items which are tables with following keys: <name>, <leftover>
-- \returns nil
function industrialtest.api.registerWaterMillFuel(config)
	local definition={
		name=config.name or "",
		calorificValue=config.calorificValue or 0,
		texture=config.texture or "industrialtest_gui_fluid_bg.png",
		storageItems=config.storageItems or {}
	}
	industrialtest.api.waterMillFuels[definition.name]=definition
end

-- \brief Returns water mill fuel information
-- \param name Name of fuel
-- \returns Table with following keys: name, calorificValue, storageItems
function industrialtest.api.getWaterMillFuel(name)
	return industrialtest.api.waterMillFuels[name]
end

-- \brief Returns water mill fuel information by item name
-- \param name ID of item
-- \returns Table with following keys: name, calorificValue, storageItems or nil in case of failure
function industrialtest.api.getWaterMillFuelByItem(name)
	for _,value in pairs(industrialtest.api.waterMillFuels) do
		for _,item in ipairs(value.storageItems) do
			if item.name==name then
				return value
			end
		end
	end
	return nil
end

-- \brief Registers Rotary Macerator recipe modifier
-- \param config table
-- \param omitPlaceholder bool, for internal use only
-- \returns nil
function industrialtest.api.registerRotaryMaceratorModifier(config,omitPlaceholder)
	local definition={
		name=config.name or "",
		modifier=config.modifier or "",
		output=config.output or "",
		time=config.time or 2,
		uses=config.uses or 1,
		modifierLeftover=config.modifierLeftover
	}

	if not omitPlaceholder and not config.modifierLeftover and string.len(definition.modifier)>0 then
		local delimiter,_=string.find(definition.modifier,":")
		definition.stackLeftover="industrialtest:"..string.sub(definition.modifier,1,delimiter-1).."_"..string.sub(definition.modifier,delimiter+1,-1).."_leftover"
		industrialtest.api.registerRotaryMaceratorModifier({
			name=definition.name,
			modifier=definition.stackLeftover,
			output=definition.output,
			time=definition.time,
			uses=definition.uses
		},true)
	end

	industrialtest.api.rotaryMaceratorModifiers[definition.name.." "..config.modifier]=definition
end

-- \brief Returns modified Rotary Macerator recipe by item and modifier
-- \param name string
-- \param modifier string
-- \returns table
function industrialtest.api.getRotaryMaceratorModifier(name,modifier)
	return industrialtest.api.rotaryMaceratorModifiers[name.." "..modifier]
end

minetest.register_on_mods_loaded(function()
	for _,def in pairs(industrialtest.api.rotaryMaceratorModifiers) do
		if def.stackLeftover then
			local leftoverDef=table.copy(minetest.registered_items[def.modifier])
			leftoverDef.groups=leftoverDef.groups or {}
			leftoverDef.groups.not_in_creative_inventory=1
			if industrialtest.mclAvailable then
				leftoverDef._doc_items_create_entry=false
			end
			-- Item name starts with : to prevent name checks, because it seems to fail in on_mods_loaded
			minetest.register_craftitem(":"..def.stackLeftover,leftoverDef)
		end
	end
end)
