-- IndustrialTest
-- Copyright (C) 2024 mrkubax10

-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

local S=minetest.get_translator("industrialtest")
industrialtest.RotaryMacerator=table.copy(industrialtest.ActivatedElectricMachine)
industrialtest.internal.unpackTableInto(industrialtest.RotaryMacerator,{
	name="industrialtest:rotary_macerator",
	description=S("Rotary Macerator"),
	tiles={
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png^industrialtest_macerator_front.png"
	},
	sounds="metal",
	requiresWrench=true,
	facedir=true,
	storageLists={
		"src",
		"dst",
		"upgrades",
		"powerStorage"
	},
	powerLists={
		{
			list="powerStorage",
			direction="i"
		}
	},
	active={
		tiles={
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png^industrialtest_macerator_front_active.png"
		}
	},
	capacity=industrialtest.api.lvPowerFlow*2,
	flow=industrialtest.api.lvPowerFlow,
	hasPowerInput=true,
	ioConfig="iiiiii",
	_opPower=60,
	_maintainSpeedOpPower=10
})

function industrialtest.RotaryMacerator.onConstruct(self,pos)
	local meta=minetest.get_meta(pos)
	local inv=meta:get_inventory()
	inv:set_size("src",1)
	inv:set_size("modifier",1)
	inv:set_size("powerStorage",1)
	inv:set_size("dst",1)
	inv:set_size("upgrades",4)
	meta:set_int("rpm",0)
	meta:set_float("srcTime",0)
	meta:set_float("maxSrcTime",0)
	meta:set_int("maintainSpeed",0)
	industrialtest.ActivatedElectricMachine.onConstruct(self,pos)
end

function industrialtest.RotaryMacerator.getFormspec(self,pos)
	local parentFormspec=industrialtest.ActivatedElectricMachine.getFormspec(self,pos)
	local meta=minetest.get_meta(pos)
	local powerPercent=meta:get_int("industrialtest.powerAmount")/meta:get_int("industrialtest.powerCapacity")*100
	local maxSrcTime=meta:get_float("maxSrcTime")
	local srcPercent=maxSrcTime>0 and meta:get_float("srcTime")/maxSrcTime*100 or 0
	local rpm=meta:get_int("rpm")
	local buttonMaintainSpeedText=meta:get_int("maintainSpeed")==1 and S("Don't maintain speed") or S("Maintain speed")
	local formspec={
		"list[context;src;3.8,1.8;1,1]",
		industrialtest.internal.getItemSlotBg(3.8,1.8,1,1),
		"list[context;modifier;4.9,1.8;1,1]",
		industrialtest.internal.getItemSlotBg(4.9,1.8,1,1),
		(powerPercent>0 and "image[3.8,2.8;1,1;industrialtest_gui_electricity_bg.png^[lowpart:"..powerPercent..":industrialtest_gui_electricity_fg.png]"
		 or "image[3.8,2.8;1,1;industrialtest_gui_electricity_bg.png]"),
		"list[context;powerStorage;3.8,3.9;1,1]",
		industrialtest.internal.getItemSlotBg(3.8,3.9,1,1),
		(srcPercent>0 and "image[4.9,2.8;1,1;gui_furnace_arrow_bg.png^[lowpart:"..srcPercent..":gui_furnace_arrow_fg.png^[transformR270]"
		 or "image[4.9,2.8;1,1;gui_furnace_arrow_bg.png^[transformR270]"),
		"list[context;dst;6,2.8;1,1;]",
		industrialtest.internal.getItemSlotBg(6,2.8,1,1),
		"list[context;upgrades;9,0.9;1,4]",
		industrialtest.internal.getItemSlotBg(9,0.9,1,4),
		"label[0.5,2.8;"..minetest.formspec_escape(S("Speed: @1",rpm)).."]",
		"button[0.5,3.4;3,0.8;maintainSpeed;"..minetest.formspec_escape(buttonMaintainSpeedText).."]",
		"listring[context;src]",
		"listring[context;dst]"
	}
	return parentFormspec..table.concat(formspec,"")
end

function industrialtest.RotaryMacerator.update(self,pos,elapsed,meta,inv)
	local shouldRerunTimer=false
	local shouldUpdateFormspec=false
	local rpm=meta:get_int("rpm")
	local maintainSpeed=meta:get_int("maintainSpeed")
	local powerAmount=meta:get_int("industrialtest.powerAmount")

	if maintainSpeed==1 and powerAmount>=self._maintainSpeedOpPower then
		local newRpm=math.max(rpm+10*elapsed,0)
		if newRpm>rpm then
			meta:set_int("rpm",newRpm)
			shouldUpdateFormspec=true
		end
		industrialtest.api.addPower(meta,-self._maintainSpeedOpPower)
		shouldRerunTimer=true
	elseif rpm>0 then
		meta:set_int("rpm",math.max(rpm-1000*elapsed,0))
		shouldRerunTimer=rpm>0
		shouldUpdateFormspec=true
	end
	
	return shouldRerunTimer,shouldUpdateFormspec
end

function industrialtest.RotaryMacerator.allowMetadataInventoryMove(self,pos,fromList,fromIndex,toList,count)
	if toList=="dst" then
		return 0
	end
	return industrialtest.ActivatedElectricMachine.allowMetadataInventoryMove(self,pos,fromList,fromIndex,toList,toIndex,count)
end

function industrialtest.RotaryMacerator.allowMetadataInventoryPut(self,pos,listname,index,stack)
	if listname=="dst" then
		return 0
	end
	return industrialtest.ActivatedElectricMachine.allowMetadataInventoryPut(self,pos,listname,index,stack)
end

function industrialtest.RotaryMacerator.onMetadataInventoryMove(self,pos,fromList,fromIndex,toList,toIndex,count)
	industrialtest.ActivatedElectricMachine.onMetadataInventoryMove(self,pos,fromList,fromIndex,toList,toIndex,count)
	if toList=="src" or toList=="modifier" or fromList=="dst" then
		self:triggerIfNeeded(pos)
	end
end

function industrialtest.RotaryMacerator.onMetadataInventoryPut(self,pos,listname,index,stack,player)
	industrialtest.ActivatedElectricMachine.onMetadataInventoryPut(self,pos,listname,index,stack,player)
	if listname=="src" or listname=="modifier" then
		self:triggerIfNeeded(pos)
	end
end

function industrialtest.RotaryMacerator.onMetadataInventoryTake(self,pos,listname,index,stack)
	industrialtest.ActivatedElectricMachine.onMetadataInventoryTake(self,pos,listname,index,stack)
	if listname=="dst" then
		self:triggerIfNeeded(pos)
	end
end

function industrialtest.RotaryMacerator.onReceiveFields(self,pos,formname,fields)
	if not fields.maintainSpeed then
		return
	end
	local meta=minetest.get_meta(pos)
	local maintainSpeed=meta:get_int("maintainSpeed")
	if maintainSpeed==1 then
		meta:set_int("maintainSpeed",0)
	elseif meta:get_int("industrialtest.powerAmount")>=self._maintainSpeedOpPower then
		meta:set_int("maintainSpeed",1)
		self:trigger(pos)
	end
	self:updateFormspec(pos)
end

function industrialtest.RotaryMacerator.shouldActivate(self,pos)
	local meta=minetest.get_meta(pos)
	local powerAmount=meta:get_int("industrialtest.powerAmount")
	if powerAmount<self._opPower then
		return false
	end

	local inv=meta:get_inventory()
	local result,_=self.getRecipeResult(pos)
	if not result or result.time==0 or not inv:room_for_item("dst",ItemStack(result.output)) then
		return false
	end

	return true
end

function industrialtest.RotaryMacerator.shouldDeactivate(self,pos)
	return not self:shouldActivate(pos)
end

function industrialtest.RotaryMacerator.afterDeactivation(self,pos)
	local meta=minetest.get_meta(pos)
	meta:set_float("srcTime",0)
	meta:set_float("maxSrcTime",0)
end

function industrialtest.RotaryMacerator.activeUpdate(self,pos,elapsed,meta,inv)
	local srcSlot=inv:get_stack("src",1)
	local dstSlot=inv:get_stack("dst",1)
	local powerAmount=meta:get_int("industrialtest.powerAmount")
	local rpm=meta:get_int("rpm")
	local maxSrcTime=meta:get_float("maxSrcTime")
	local speed=industrialtest.api.getMachineSpeed(meta)
	local requiredPower=elapsed*self._opPower*speed

	if maxSrcTime<=0 then
		local result,_=self.getRecipeResult(pos)
		meta:set_float("maxSrcTime",result.time)
		maxSrcTime=result.time
	end

	local srcTime=meta:get_float("srcTime")+elapsed*(1+rpm/7500)
	if srcTime>=maxSrcTime then
		local result,modified=self.getRecipeResult(pos)
		local resultStack=ItemStack(result.output)
		local multiplier=math.min(srcSlot:get_count(),speed)
		local prevCount=resultStack:get_count()
		resultStack:set_count(resultStack:get_count()*multiplier)
		local leftover=inv:add_item("dst",resultStack)
		meta:set_float("srcTime",0)
		srcSlot:take_item(multiplier-leftover:get_count()/prevCount)
		inv:set_stack("src",1,srcSlot)
		meta:set_int("rpm",math.min(rpm+750*elapsed,7500))
		if modified then
			local modifierSlot=inv:get_stack("modifier",1)
			local modifierMeta=modifierSlot:get_meta()
			local uses=result.uses
			local replace=false
			if modifierMeta:contains("uses") then
				uses=modifierMeta:get_int("uses")
			else
				replace=true
			end
			uses=math.max(uses-1,0)
			if uses==0 then
				if result.modifierLeftover then
					modifierSlot:set_name(result.modifierLeftover)
				else
					modifierSlot:take_item(1)
				end
				uses=result.uses
			end
			if not modifierSlot:is_empty() and not result.modifierLeftover then
				local modifierDef=modifierSlot:get_definition()
				modifierMeta:set_int("uses",uses)
				modifierMeta:set_string("description",string.format("%s (%s: %d)",modifierDef.description,S("Uses"),uses))
			end
			if replace then
				modifierSlot:set_name(result.stackLeftover)
			end
			inv:set_stack("modifier",1,modifierSlot)
		end
	else
		meta:set_float("srcTime",srcTime)
	end

	industrialtest.api.addPower(meta,-requiredPower)

	return true
end

industrialtest.RotaryMacerator:register()

function industrialtest.RotaryMacerator.getRecipeResult(pos)
	local meta=minetest.get_meta(pos)
	local inv=meta:get_inventory()
	local srcSlot=inv:get_stack("src",1)
	local modifierSlot=inv:get_stack("modifier",1)

	local result
	local modified=false
	if modifierSlot:is_empty() then
		result=industrialtest.api.getMaceratorRecipeResult(srcSlot:get_name())
	else
		result=industrialtest.api.getRotaryMaceratorModifier(srcSlot:get_name(),modifierSlot:get_name())
		modified=true
	end

	return result,modified
end

minetest.register_craft({
	type="shaped",
	output="industrialtest:rotary_macerator",
	recipe={
		{"industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot"},
		{"industrialtest:refined_iron_ingot","industrialtest:macerator","industrialtest:refined_iron_ingot"},
		{"industrialtest:refined_iron_ingot","industrialtest:advanced_machine_block","industrialtest:refined_iron_ingot"}
	}
})
