-- IndustrialTest
-- Copyright (C) 2023 mrkubax10

-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

local S=minetest.get_translator("industrialtest")

industrialtest.Cable={}

function industrialtest.Cable.onConstruct(self,pos)
	local connections=industrialtest.api.getConnections(pos)
	for _,conn in ipairs(connections) do
		local meta=minetest.get_meta(conn)
		if industrialtest.api.isNetworkMaster(meta) then
			industrialtest.api.createNetworkMapForNode(conn)
			local networkNode=minetest.get_node(conn)
			local def=minetest.registered_nodes[networkNode.name]
			if def and def._industrialtest_self then
				def._industrialtest_self:triggerIfNeeded(conn)
			else
				-- Support for bare definitions that don't use industrialtest pseudo-OOP
				minetest.get_node_timer(conn):start(industrialtest.config.updateDelay)
			end
		else
			local networks=industrialtest.api.isAttachedToNetwork(meta)
			if networks then
				for _,network in ipairs(networks) do
					industrialtest.api.createNetworkMapForNode(network)
					local networkNode=minetest.get_node(network)
					local def=minetest.registered_nodes[networkNode.name]
					if def and def._industrialtest_self then
						def._industrialtest_self:triggerIfNeeded(network)
					else
						-- Support for bare definitions that don't use industrialtest pseudo-OOP
						minetest.get_node_timer(network):start(industrialtest.config.updateDelay)
					end
				end
			end
		end
	end
end

function industrialtest.Cable.onDestruct(self,pos)
	local meta=minetest.get_meta(pos)
	local networks=industrialtest.api.isAttachedToNetwork(meta)
	if networks then
		for _,network in ipairs(networks) do
			industrialtest.api.createNetworkMapForNode(network,pos)
		end
	end
end

function industrialtest.Cable.createDefinitionTable(self,description,inventoryImage,tile,insulated)
	local size=(insulated and self.size+0.02 or self.size)
	local def={
		description=description,
		inventory_image=inventoryImage,
		wield_image=inventoryImage,
		tiles={tile},
		paramtype="light",
		sunlight_propagates=true,
		use_texture_alpha=(self.transparent and "clip" or "opaque"),
		drawtype="nodebox",
		node_box={
			type="connected",
			fixed={
				-size,
				-size,
				-size,
				size,
				size,
				size
			},
			connect_top={
				-size,
				-size,
				-size,
				size,
				0.5,
				size
			},
			connect_bottom={
				-size,
				-0.5,
				-size,
				size,
				size,
				size
			},
			connect_front={
				-size,
				-size,
				-0.5,
				size,
				size,
				size,
			},
			connect_left={
				-0.5,
				-size,
				-size,
				size,
				size,
				size
			},
			connect_back={
				-size,
				-size,
				-size,
				size,
				size,
				0.5
			},
			connect_right={
				-size,
				-size,
				-size,
				0.5,
				size,
				size
			}
		},
		connects_to={
			"group:_industrialtest_hasPowerInput",
			"group:_industrialtest_hasPowerOutput",
			"group:_industrialtest_cable"
		},
		on_construct=function(pos)
			self:onConstruct(pos)
		end,
		on_destruct=function(pos)
			self:onDestruct(pos)
		end,
		_industrialtest_cableFlow=self.flow
	}

	if industrialtest.mtgAvailable then
		def.groups={
			cracky=1,
			level=1,
			oddly_breakable_by_hand=1
		}
		def.sound=default.node_sound_metal_defaults()
	elseif industrialtest.mclAvailable then
		def.groups={
			handy=1,
			pickaxey=1
		}
		def._mcl_blast_resistance=1
		def._mcl_hardness=0.5
		def.sound=mcl_sounds.node_sound_metal_defaults()
	end
	def.groups._industrialtest_cable=1
	if insulated then
		def.groups._industrialtest_insulatedCable=1
	end

	return def
end

function industrialtest.Cable.register(self)
	local def=self:createDefinitionTable(self.description,self.inventoryImage,self.tile,self.safe)
	minetest.register_node(self.name,def)

	if self.insulated then
		def=self:createDefinitionTable(self.insulated.description,self.insulated.inventoryImage,self.insulated.tile,true)
		minetest.register_node(self.insulated.name,def)
	end
end

industrialtest.TinCable=table.copy(industrialtest.Cable)
industrialtest.internal.unpackTableInto(industrialtest.TinCable,{
	name="industrialtest:tin_cable",
	description=S("Tin Cable"),
	inventoryImage="industrialtest_tin_cable_inv.png",
	tile="industrialtest_tin_cable.png",
	size=0.19,
	flow=industrialtest.api.lvPowerFlow,
	insulated={
		name="industrialtest:insulated_tin_cable",
		description=S("Insulated Tin Cable"),
		inventoryImage="industrialtest_insulated_tin_cable_inv.png",
		tile="industrialtest_insulated_tin_cable.png"
	}
})

industrialtest.TinCable:register()

minetest.register_craft({
	type="shaped",
	output="industrialtest:tin_cable 6",
	recipe={
		{industrialtest.elementKeys.tinIngot,industrialtest.elementKeys.tinIngot,industrialtest.elementKeys.tinIngot}
	}
})

minetest.register_craft({
	type="shapeless",
	output="industrialtest:insulated_tin_cable",
	recipe={
		"industrialtest:tin_cable",
		industrialtest.elementKeys.rubber
	}
})

minetest.register_craft({
	type="shaped",
	output="industrialtest:insulated_tin_cable 6",
	recipe={
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber},
		{industrialtest.elementKeys.tinIngot,industrialtest.elementKeys.tinIngot,industrialtest.elementKeys.tinIngot},
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber}
	}
})

industrialtest.api.registerCableFormerRecipe({
	output="industrialtest:tin_cable 12",
	recipe=industrialtest.elementKeys.tinIngot,
	time=1
})

industrialtest.CopperCable=table.copy(industrialtest.Cable)
industrialtest.internal.unpackTableInto(industrialtest.CopperCable,{
	name="industrialtest:copper_cable",
	description=S("Copper Cable"),
	inventoryImage="industrialtest_copper_cable_inv.png",
	tile="industrialtest_copper_cable.png",
	size=0.15,
	flow=industrialtest.api.mvPowerFlow,
	insulated={
		name="industrialtest:insulated_copper_cable",
		description=S("Insulated Copper Cable"),
		inventoryImage="industrialtest_insulated_copper_cable_inv.png",
		tile="industrialtest_insulated_copper_cable.png"
	}
})

industrialtest.CopperCable:register()

minetest.register_craft({
	type="shaped",
	output="industrialtest:copper_cable 6",
	recipe={
		{industrialtest.elementKeys.copperIngot,industrialtest.elementKeys.copperIngot,industrialtest.elementKeys.copperIngot}
	}
})

minetest.register_craft({
	type="shapeless",
	output="industrialtest:insulated_copper_cable",
	recipe={
		"industrialtest:copper_cable",
		industrialtest.elementKeys.rubber
	}
})

minetest.register_craft({
	type="shaped",
	output="industrialtest:insulated_copper_cable 6",
	recipe={
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber},
		{industrialtest.elementKeys.copperIngot,industrialtest.elementKeys.copperIngot,industrialtest.elementKeys.copperIngot},
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber}
	}
})

industrialtest.api.registerCableFormerRecipe({
	output="industrialtest:copper_cable 12",
	recipe=industrialtest.elementKeys.copperIngot
})

industrialtest.GoldCable=table.copy(industrialtest.Cable)
industrialtest.internal.unpackTableInto(industrialtest.GoldCable,{
	name="industrialtest:gold_cable",
	description=S("Gold Cable"),
	inventoryImage="industrialtest_gold_cable_inv.png",
	tile="industrialtest_gold_cable.png",
	size=0.15,
	flow=industrialtest.api.hvPowerFlow,
	insulated={
		name="industrialtest:insulated_gold_cable",
		description=S("Insulated Gold Cable"),
		inventoryImage="industrialtest_insulated_gold_cable_inv.png",
		tile="industrialtest_insulated_gold_cable.png"
	}
})

industrialtest.GoldCable:register()

minetest.register_craft({
	type="shaped",
	output="industrialtest:gold_cable 6",
	recipe={
		{industrialtest.elementKeys.goldIngot,industrialtest.elementKeys.goldIngot,industrialtest.elementKeys.goldIngot}
	}
})

minetest.register_craft({
	type="shapeless",
	output="industrialtest:insulated_gold_cable",
	recipe={
		"industrialtest:gold_cable",
		industrialtest.elementKeys.rubber
	}
})

minetest.register_craft({
	type="shaped",
	output="industrialtest:insulated_gold_cable 6",
	recipe={
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber},
		{industrialtest.elementKeys.goldIngot,industrialtest.elementKeys.goldIngot,industrialtest.elementKeys.goldIngot},
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber}
	}
})

industrialtest.api.registerCableFormerRecipe({
	output="industrialtest:gold_cable 12",
	recipe=industrialtest.elementKeys.goldIngot
})

industrialtest.IronCable=table.copy(industrialtest.Cable)
industrialtest.internal.unpackTableInto(industrialtest.IronCable,{
	name="industrialtest:iron_cable",
	description=S("Iron Cable"),
	inventoryImage="industrialtest_iron_cable_inv.png",
	tile="industrialtest_iron_cable.png",
	size=0.15,
	flow=industrialtest.api.evPowerFlow,
	insulated={
		name="industrialtest:insulated_iron_cable",
		description=S("Insulated Iron Cable"),
		inventoryImage="industrialtest_insulated_iron_cable_inv.png",
		tile="industrialtest_insulated_iron_cable.png"
	}
})

industrialtest.IronCable:register()

minetest.register_craft({
	type="shaped",
	output="industrialtest:iron_cable 6",
	recipe={
		{"industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot"}
	}
})

minetest.register_craft({
	type="shapeless",
	output="industrialtest:insulated_iron_cable",
	recipe={
		"industrialtest:iron_cable",
		industrialtest.elementKeys.rubber
	}
})

minetest.register_craft({
	type="shaped",
	output="industrialtest:insulated_iron_cable 6",
	recipe={
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber},
		{"industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot","industrialtest:refined_iron_ingot"},
		{industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber,industrialtest.elementKeys.rubber}
	}
})

industrialtest.api.registerCableFormerRecipe({
	output="industrialtest:iron_cable 12",
	recipe="industrialtest:refined_iron_ingot",
	time=3
})

industrialtest.GlassFibreCable=table.copy(industrialtest.Cable)
industrialtest.internal.unpackTableInto(industrialtest.GlassFibreCable,{
	name="industrialtest:glass_fibre_cable",
	description=S("Glass Fibre Cable"),
	inventoryImage="industrialtest_glass_fibre_cable_inv.png",
	transparent=true,
	tile="industrialtest_glass_fibre_cable.png",
	safe=true,
	size=0.12,
	flow=industrialtest.api.ivPowerFlow
})

industrialtest.GlassFibreCable:register()

minetest.register_craft({
	type="shaped",
	output="industrialtest:glass_fibre_cable 4",
	recipe={
		{industrialtest.elementKeys.glass,industrialtest.elementKeys.glass,industrialtest.elementKeys.glass},
		{industrialtest.elementKeys.powerCarrier,industrialtest.elementKeys.diamond,industrialtest.elementKeys.powerCarrier},
		{industrialtest.elementKeys.glass,industrialtest.elementKeys.glass,industrialtest.elementKeys.glass}
	}
})
-- TODO: Add glass fibre cable craft with silver ingot

if industrialtest.config.electrocution then
	local electrocutionDelta=0
	minetest.register_globalstep(function(dtime)
		electrocutionDelta=electrocutionDelta+dtime
		if electrocutionDelta<industrialtest.config.updateDelay then
			return
		end
		electrocutionDelta=0

		local offsets={
			vector.new(0,0,0),
			vector.new(-0.7,0,0),
			vector.new(-0.7,1,0),
			vector.new(0,1,0),
			vector.new(0.7,0,0),
			vector.new(0.7,1,0),
			vector.new(0,0,-0.7),
			vector.new(0,1,-0.7),
			vector.new(0,0,0.7),
			vector.new(0,1,0.7)
		}

		local players=minetest.get_connected_players()
		for _,player in ipairs(players) do
			local pos=player:get_pos()
			for _,offset in ipairs(offsets) do
				local nodePos=vector.add(pos,offset)
				local node=minetest.get_node(nodePos)
				local def=minetest.registered_nodes[node.name]
				if def and def.groups and def.groups._industrialtest_cable and not def.groups._industrialtest_insulatedCable then
					local meta=minetest.get_meta(pos)
					local networks=industrialtest.api.isAttachedToNetwork(meta)
					if networks then
						local current=0
						for _,network in ipairs(networks) do
							current=current+industrialtest.api.getFlowingCurrent(network)
						end
						if current>0 then
							local removed=math.ceil(current/500)
							player:set_hp(player:get_hp()-removed,"electrocution")
							break
						end
					end
				end
			end
		end
	end)
end
