-- IndustrialTest
-- Copyright (C) 2024 mrkubax10

-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

local S=minetest.get_translator("industrialtest")
industrialtest.internal.chargepads={}
industrialtest.Chargepad=table.copy(industrialtest.ActivatedElectricMachine)
industrialtest.internal.unpackTableInto(industrialtest.Chargepad,{
	storageLists={
		"charged",
		"discharged"
	},
	powerLists={
		{
			list="charged",
			direction="o"
		},
		{
			list="discharged",
			direction="i"
		}
	},
	facedir=true,
	ioConfig="iiiioi",
	hasPowerInput=true,
	hasPowerOutput=true
})

function industrialtest.Chargepad.onConstruct(self,pos)
	local meta=minetest.get_meta(pos)
	local inv=meta:get_inventory()
	inv:set_size("charged",1)
	inv:set_size("discharged",1)
	meta:set_int("active",0)
	industrialtest.ActivatedElectricMachine.onConstruct(self,pos)
end

function industrialtest.Chargepad.getFormspec(self,pos)
	local meta=minetest.get_meta(pos)
	local parentFormspec=industrialtest.ActivatedElectricMachine.getFormspec(self,pos)
	local charged=meta:get_int("industrialtest.powerAmount")/meta:get_int("industrialtest.powerCapacity")
	local formspec={
		industrialtest.internal.getItemSlotBg(1,2.5,1,1),
		"list[context;charged;1,2.5;1,1]",
		"label[0.9,3.9;"..S("Charge").."]",
		industrialtest.internal.getItemSlotBg(3,2.5,1,1),
		"list[context;discharged;3,2.5;1,1]",
		"label[2.7,3.9;"..S("Discharge").."]",
		self.createPowerIndicatorWidget(charged,9,1),
		"listring[context;charged]",
		"listring[context;discharged]"
	}
	return parentFormspec..table.concat(formspec,"")
end

function industrialtest.Chargepad.register(self)
	industrialtest.ActivatedElectricMachine.register(self)
	table.insert(industrialtest.internal.chargepads,self.name)
	table.insert(industrialtest.internal.chargepads,self.name.."_active")
	minetest.register_craft({
		type="shaped",
		output=self.name,
		recipe={
			{"industrialtest:electronic_circuit",industrialtest.elementKeys.stoneSlab,"industrialtest:electronic_circuit"},
			{industrialtest.elementKeys.rubber,self._basePowerStorage,industrialtest.elementKeys.rubber}
		}
	})
end

function industrialtest.Chargepad.chargePlayer(meta,player,flow)
	local inv
	if industrialtest.mtgAvailable then
		_,inv=armor:get_valid_player(player,"")
		if not inv then
			return false
		end
	elseif industrialtest.mclAvailable then
		inv=player:get_inventory()
	end

	local armorList=inv:get_list("armor")
	local chargedSlots={}
	for i,stack in ipairs(armorList) do
		local stackMeta=stack:get_meta()
		if industrialtest.api.hasPowerStorage(stackMeta) and not industrialtest.api.isFullyCharged(stackMeta) then
			table.insert(chargedSlots,{
				index=i,
				stack=stack
			})
		end
	end
	local wielded=player:get_wielded_item()
	if not wielded:is_empty() then
		local wieldedMeta=wielded:get_meta()
		if industrialtest.api.hasPowerStorage(wieldedMeta) and not industrialtest.api.isFullyCharged(wieldedMeta) then
			table.insert(chargedSlots,{
				stack=wielded
			})
		end
	end

	if #chargedSlots==0 then
		return false
	end
	local distribution=math.min(flow,meta:get_int("industrialtest.powerAmount"))/#chargedSlots

	for _,chargedSlot in ipairs(chargedSlots) do
		industrialtest.api.transferPowerToItem(meta,chargedSlot.stack,distribution)
		if chargedSlot.index then
			inv:set_stack("armor",chargedSlot.index,chargedSlot.stack)
		else
			player:set_wielded_item(chargedSlot.stack)
		end
	end

	return true
end

function industrialtest.Chargepad.action(self,pos,node)
	local meta=minetest.get_meta(pos)
	local inv=meta:get_inventory()
	local chargedSlot=inv:get_stack("charged",1)
	local dischargedSlot=inv:get_stack("discharged",1)
	local _,shouldUpdateFormspec=industrialtest.api.powerFlow(pos)
	local flow=meta:get_int("industrialtest.powerFlow")

	if chargedSlot:get_count()>0 and meta:get_int("industrialtest.powerAmount")>0 and industrialtest.api.transferPowerToItem(meta,chargedSlot,flow)>0 then
		inv:set_stack("charged",1,chargedSlot)
		shouldUpdateFormspec=true
	end
	if dischargedSlot:get_count()>0 and not industrialtest.api.isFullyCharged(meta) and industrialtest.api.transferPowerFromItem(dischargedSlot,meta,flow)>0 then
		inv:set_stack("discharged",1,dischargedSlot)
		shouldUpdateFormspec=true
	end

	local players=minetest.get_connected_players()
	local p1=vector.offset(pos,-0.5,0,-0.5)
	local p2=vector.offset(pos,0.5,2,0.5)
	local playerFound=false
	for _,player in ipairs(players) do
		if vector.in_area(player:get_pos(),p1,p2) then
			playerFound=true
			shouldUpdateFormspec=shouldUpdateFormspec or self.chargePlayer(meta,player,flow)
			break
		end
	end
	local active=meta:get_int("active")==1
	if playerFound and not active then
		self:activate(pos)
		meta:set_int("active",1)
	elseif (not playerFound or meta:get_int("industrialtest.powerAmount")==0) and active then
		self:deactivate(pos)
		meta:set_int("active",0)
	end

	if shouldUpdateFormspec then
		self:updateFormspec(pos)
	end
end

industrialtest.BatboxChargepad=table.copy(industrialtest.Chargepad)
industrialtest.internal.unpackTableInto(industrialtest.BatboxChargepad,{
	name="industrialtest:batbox_chargepad",
	description=S("BatBox Chargepad"),
	tiles={
		"industrialtest_wood_machine_block.png^industrialtest_chargepad_top.png",
		"industrialtest_wood_machine_block.png",
		"industrialtest_wood_machine_block.png",
		"industrialtest_wood_machine_block.png",
		"industrialtest_wood_machine_block.png",
		"industrialtest_wood_machine_block.png^industrialtest_batbox_front.png"
	},
	sounds="wood",
	active={
		tiles={
			"industrialtest_wood_machine_block.png^industrialtest_chargepad_top_active.png",
			"industrialtest_wood_machine_block.png",
			"industrialtest_wood_machine_block.png",
			"industrialtest_wood_machine_block.png",
			"industrialtest_wood_machine_block.png",
			"industrialtest_wood_machine_block.png^industrialtest_batbox_front.png"
		}
	},
	capacity=25000,
	flow=industrialtest.api.lvPowerFlow,
	_basePowerStorage="industrialtest:batbox"
})
industrialtest.BatboxChargepad:register()

industrialtest.CESUChargepad=table.copy(industrialtest.Chargepad)
industrialtest.internal.unpackTableInto(industrialtest.CESUChargepad,{
	name="industrialtest:cesu_chargepad",
	description=S("CESU Chargepad"),
	tiles={
		"industrialtest_bronze_machine_block.png^industrialtest_chargepad_top.png",
		"industrialtest_bronze_machine_block.png",
		"industrialtest_bronze_machine_block.png",
		"industrialtest_bronze_machine_block.png",
		"industrialtest_bronze_machine_block.png",
		"industrialtest_bronze_machine_block.png^industrialtest_cesu_front.png"
	},
	sounds="metal",
	active={
		tiles={
			"industrialtest_bronze_machine_block.png^industrialtest_chargepad_top_active.png",
			"industrialtest_bronze_machine_block.png",
			"industrialtest_bronze_machine_block.png",
			"industrialtest_bronze_machine_block.png",
			"industrialtest_bronze_machine_block.png",
			"industrialtest_bronze_machine_block.png^industrialtest_cesu_front.png"
		}
	},
	capacity=400000,
	flow=industrialtest.api.mvPowerFlow,
	_basePowerStorage="industrialtest:cesu"
})
industrialtest.CESUChargepad:register()

industrialtest.MFEChargepad=table.copy(industrialtest.Chargepad)
industrialtest.internal.unpackTableInto(industrialtest.MFEChargepad,{
	name="industrialtest:mfe_chargepad",
	description=S("MFE Chargepad"),
	tiles={
		"industrialtest_machine_block.png^industrialtest_chargepad_top.png",
		"industrialtest_machine_block.png",
		"industrialtest_machine_block.png",
		"industrialtest_machine_block.png",
		"industrialtest_machine_block.png",
		"industrialtest_machine_block.png^industrialtest_mfe_front.png"
	},
	sounds="metal",
	requiresWrench=true,
	active={
		tiles={
			"industrialtest_machine_block.png^industrialtest_chargepad_top_active.png",
			"industrialtest_machine_block.png",
			"industrialtest_machine_block.png",
			"industrialtest_machine_block.png",
			"industrialtest_machine_block.png",
			"industrialtest_machine_block.png^industrialtest_mfe_front.png"
		}
	},
	capacity=3000000,
	flow=industrialtest.api.hvPowerFlow,
	_basePowerStorage="industrialtest:mfe"
})
industrialtest.MFEChargepad:register()

industrialtest.MFSUChargepad=table.copy(industrialtest.Chargepad)
industrialtest.internal.unpackTableInto(industrialtest.MFSUChargepad,{
	name="industrialtest:mfsu_chargepad",
	description=S("MFSU Chargepad"),
	tiles={
		"industrialtest_advanced_machine_block.png^industrialtest_chargepad_top.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png",
		"industrialtest_advanced_machine_block.png^industrialtest_mfsu_front.png"
	},
	sounds="metal",
	requiresWrench=true,
	active={
		tiles={
			"industrialtest_advanced_machine_block.png^industrialtest_chargepad_top_active.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png",
			"industrialtest_advanced_machine_block.png^industrialtest_mfsu_front.png"
		}
	},
	capacity=30000000,
	flow=industrialtest.api.evPowerFlow,
	_basePowerStorage="industrialtest:mfsu"
})
industrialtest.MFSUChargepad:register()

minetest.register_abm({
	label="Chargepad updating",
	nodenames=industrialtest.internal.chargepads,
	interval=industrialtest.config.updateDelay,
	chance=1,
	action=function(pos,node)
		local def=minetest.registered_nodes[node.name]
		def._industrialtest_self:action(pos,node)
	end
})
