-- IndustrialTest
-- Copyright (C) 2025 mrkubax10

-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

local S=minetest.get_translator("industrialtest")

-- \brief Creates hypertext table with rows
-- \param rows table
-- \param withTitleRow bool
-- \param description (optional) string
-- \returns string
local function createTable(rows,withTitleRow,description)
	local columnWidths={}
	for _,row in ipairs(rows) do
		for i,cell in ipairs(row) do
			-- Add 3 here because of cell padding (2 spaces at each side of column text) and right cell separator
			columnWidths[i]=math.max(columnWidths[i] or 0,string.len(cell)+3)
		end
	end

	-- Begin with width of 1 because of left edge
	local tableWidth=1
	for _,width in ipairs(columnWidths) do
		tableWidth=tableWidth+width
	end

	local horizontalEdge=string.rep("-",tableWidth).."\n"
	local hypertextRows={horizontalEdge}
	for i,row in ipairs(rows) do
		local hypertextRow={"|"}
		for o,cell in ipairs(row) do
			local cellWidth=string.len(cell)+3
			local rightPadding=string.rep(" ",columnWidths[o]-cellWidth)
			table.insert(hypertextRow,string.format(" %s%s |",cell,rightPadding))
		end
		table.insert(hypertextRow,"\n")
		table.insert(hypertextRows,table.concat(hypertextRow,""))
		-- Insert horizontal edge if title row is present
		if withTitleRow and i==1 then
			table.insert(hypertextRows,horizontalEdge)
		end
	end
	table.insert(hypertextRows,horizontalEdge)

	return "<mono>"..table.concat(hypertextRows,"")..(description or "").."</mono>\n"
end

-- \brief Creates table with machine information while providing few shortcuts for common fields
-- \param fields table
-- \param machineName string
-- \param num (optional) number
-- \returns string
local function createMachineInformationTable(fields,machineName,num)
	local fieldLabels={
		inputVoltage="Input voltage level",
		outputVoltage="Output voltage level",
		voltage="Voltage level",
		lowVoltage="Low voltage level",
		highVoltage="High voltage level",
		damageRed="Damage reduction",
		recipe="Recipe type",
		powerCapacity="Power capacity",
		opPower="Power per operation",
		powerPerFuel="Power per fuel unit",
		uses="Uses",
		efficiency="Default efficiency"
	}

	local rows={}
	for _,field in ipairs(fields) do
		local label=fieldLabels[field.name] or field.name
		table.insert(rows,{
			label,
			field.value
		})
	end

	return createTable(rows,false,S("Figure @1. Information for @2",(num or 1),machineName))
end

local pages={
	{
		name="introduction",
		title=S("Introduction"),
		icon=industrialtest.elementKeys.paper,
		content={
			[[
				<big>||Introduction||</big>
				<left>
					||IndustrialTest implements finite power functionality which can be used to perform various tasks using generators and consumers. Such features are not a new thing in Luanti, as there are older mods which do that already. This mod, however, proves to be more flexible.||
				</left>
				<img name="industrialtest_guide_introduction.png" width="{{IMAGE_WIDTH}}">
			]]
		}
	},

	{
		name="electricalNetwork",
		title=S("Electrical network"),
		icon="industrialtest:insulated_copper_cable",
		content={
			[[
				<big>||Electrical network||</big>
				<left>
					||In order to attach electricity consumers to power sources (like generators, batteries, further just called generators) electrical network is used. Consumers can get connected either directly to generators or through various cables. Each generator can output certain voltage (this is actually amperage, however since this mod doesn't simulate electricity accurately, it's called voltage), the same situation is for consumers as they can only handle up to certain voltage.||
					||Figure 1 shows voltage levels present in this mod.||
					||If voltage is too high for cable it will melt without dropping anything while consumer will explode also without dropping anything and possibly destroying terrain around. Power capacity is measured in EU (Energy Units) and the voltage is EU/update.||
					||Picture 1 shows environment after consumer explosion.||
				</left>
			]],
			createTable({
				{
					"Voltage level",
					"EU/update",
					"Cable"
				},
				{
					"LV",
					"600",
					"Tin"
				},
				{
					"MV",
					"2400",
					"Copper"
				},
				{
					"HV",
					"10200",
					"Gold"
				},
				{
					"EV",
					"40800",
					"Iron"
				},
				{
					"IV",
					"163800",
					"Glass Fibre"
				}
			},true,S("Figure 1. Voltage levels and respective cables")),
			[[
				<img name="industrialtest_guide_explosion.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Environment after consumer explosion||</left>
			]]
		}
	},

	{
		name="cableFormer",
		title=S("Cable Former"),
		icon="industrialtest:cable_former",
		content={
			[[
				<big>||Cable Former||</big>
				<left>
					||Cable Former can be used to produce cables with higher material efficiency. It can, however, only produce cables from metals so for example <item name="industrialtest:glass_fibre_cable" height="{{ITEM_HEIGHT}}"> <b>Glass Fibre Cable</b> can't be produced in it.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Cable forming"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.CableFormer.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.CableFormer.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.CableFormer.efficiency)
				}
			},S("Cable Former")),
			[[
				<img name="industrialtest_guide_cable_former.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Cable Former running||</left>
			]]
		}
	},

	{
		name="canningMachine",
		title=S("Canning Machine"),
		icon="industrialtest:canning_machine",
		content={
			[[
				<big>||Canning Machine||</big>
				<left>
					||Canning Machine is used to move certain fluid from one item to another. The common usage is to refill fuel in <item name="industrialtest:jetpack_v" height="{{ITEM_HEIGHT}}"> <b>Jetpack</b> or <item name="industrialtest:fuel_can" height="{{ITEM_HEIGHT}}"> <b>Fuel Can</b>.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.CanningMachine.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.CanningMachine._opPower)
				},
				{
					name="Canning completion time",
					value=string.format("%d seconds",industrialtest.CanningMachine._canningTime)
				}
			},S("Canning Machine")),
			[[
				<img name="industrialtest_guide_canning_machine.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Canning Machine refilling <item name="industrialtest:fuel_can" height="{{ITEM_HEIGHT}}"> <b>Fuel Can</b> with fuel from <item name="industrialtest:coalfuel_cell" height="{{ITEM_HEIGHT}}"> <b>Coalfuel Cell</b>||</left>
			]]
		}
	},

	{
		name="chargepads",
		title=S("Chargepads"),
		icon="industrialtest:batbox_chargepad",
		content={
			[[
				<big>||Chargepads||</big>
				<left>
					||Chargepad is similar in functionality to regular power storage like <item name="industrialtest:batbox" height="{{ITEM_HEIGHT}}"> <b>BatBox</b>. The additional feature is that it can charge equipped electric gear and selected item in hotbar while player is standing on it. There is a chargepad corresponding to each power storage node and they have exact same information so there is no need to list that again here. While charging the top of chargepad changes color to blue.||
				</left>
				<mono>
					----
				</mono>
				<img name="industrialtest_guide_chargepad.png">
				<left>||Picture 1. Unused <item name="industrialtest:batbox_chargepad" height="{{ITEM_HEIGHT}}"> <b>BatBox Chargepad</b> attached to <item name="industrialtest:solar_panel" height="{{ITEM_HEIGHT}}"> <b>Solar Panel</b>.||</left>
			]]
		}
	},

	{
		name="compressor",
		title=S("Compressor"),
		icon="industrialtest:compressor",
		content={
			[[
				<big>||Compressor||</big>
				<left>
					||Compressor as the name suggests is used to compress various materials. Mostly required to produce items for crafting other items.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Compressing"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Compressor.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Compressor.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.Compressor.efficiency)
				}
			},S("Compressor")),
			[[
				<img name="industrialtest_guide_compressor.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Compressor compressing <item name="industrialtest:plantball" height="{{ITEM_HEIGHT}}"> <b>Plantball</b> into <item name="industrialtest:compressed_plantball" height="{{ITEM_HEIGHT}}"> <b>Compressed Plantball</b>||</left>
			]]
		}
	},

	{
		name="electricFurnace",
		title=S("Electric Furnace"),
		icon="industrialtest:electric_furnace",
		content={
			[[
				<big>||Electric Furnace||</big>
				<left>
					||Electric Furnace like regular furnace can be used to smelt items. The difference is that Electric Furnace is slightly faster even without any upgrades and, well, it's electric so it doesn't require fuel.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Smelting"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricFurnace.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.ElectricFurnace.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.ElectricFurnace.efficiency)
				}
			},S("Electric Furnace")),
			[[
				<img name="industrialtest_guide_electric_furnace.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Electric Furnace smelting <item name="industrialtest:iron_dust" height="{{ITEM_HEIGHT}}"> <b>Iron Dust</b> into <item name="{{elementKeyIronIngot}}" height="{{ITEM_HEIGHT}}"> <b>Iron Ingot</b>||</left>
			]]
		}
	},

	{
		name="extractor",
		title=S("Extractor"),
		icon="industrialtest:extractor",
		content={
			[[
				<big>||Extractor||</big>
				<left>
					||Extractor is used to separate materials from within other materials, often when it's not possible by other means.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Extracting"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Extractor.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Extractor.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.Extractor.efficiency)
				}
			},S("Extractor")),
			[[
				<img name="industrialtest_guide_extractor.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Extractor extracting <item name="{{elementKeyRubber}}" height="{{ITEM_HEIGHT}}"> <b>Rubber</b> from <item name="{{elementKeyRubberWood}}" height="{{ITEM_HEIGHT}}"> <b>Rubber Wood</b>||</left>
			]]
		}
	},

	{
		name="generator",
		title=S("Generator"),
		icon="industrialtest:generator",
		content={
			[[
				<big>||Generator||</big>
				<left>
					||Generator produces energy from various fuels which can be used for smelting. It is the most basic way of powering electric machines and also the simplest to obtain.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Generator.capacity)
				},
				{
					name="powerPerFuel",
					value="200 EU"
				}
			},S("Generator")),
			[[
				<img name="industrialtest_guide_generator.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Generator producing power from <item name="{{elementKeyCoal}}" height="{{ITEM_HEIGHT}}"> <b>Coal</b>||</left>
			]]
		}
	},

	{
		name="geothermalGenerator",
		title=S("Geothermal Generator"),
		icon="industrialtest:geothermal_generator",
		content={
			[[
				<big>||Geothermal Generator||</big>
				<left>
					||Geothermal Generator produces energy using heat from fluids like lava. Hot fluids can be transferred into Geothermal Generator for example by using <item name="industrialtest:empty_cell" height="{{ITEM_HEIGHT}}"> <b>Empty Cell</b> or <item name="{{elementKeyBucket}}" height="{{ITEM_HEIGHT}}"> <b>Empty Bucket</b>.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.GeothermalGenerator.capacity)
				}
			},S("Geothermal Generator")),
			[[
				<img name="industrialtest_guide_geothermal_generator.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Geothermal Generator producing power from <b>Lava</b>||</left>
			]]
		}
	},

	{
		name="inductionFurnace",
		title=S("Induction Furnace"),
		icon="industrialtest:induction_furnace",
		content={
			[[
				<big>||Induction Furnace||</big>
				<left>
					||Induction Furnace like Electric Furnace uses electricity to smelt items. The difference between those is that Induction Furnace when running builds up heat internally over time. This property results in Induction Furnace smelting items faster the longer it's running. Once smelting is finished Induction Furnace begins to slowly lose heat. The heat level is indicated on the left of formspec.||
					||Furthermore Induction Furnace allows to smelt 2 different items at once increasing efficiency even more.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="MV"
				},
				{
					name="recipe",
					value="Smelting"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.InductionFurnace.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.InductionFurnace._opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.InductionFurnace._efficiency)
				}
			},S("Induction Furnace")),
			[[
				<img name="industrialtest_guide_induction_furnace.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Induction Furnace smelting <item name="industrialtest:copper_dust" height="{{ITEM_HEIGHT}}"> <b>Copper Dust</b> and <item name="industrialtest:iron_dust" height="{{ITEM_HEIGHT}}"> <b>Iron Dust</b> into <item name="{{elementKeyCopperIngot}}" height="{{ITEM_HEIGHT}}"> <b>Copper Ingot</b> and <item name="{{elementKeyIronIngot}}" height="{{ITEM_HEIGHT}}"> <b>Iron Ingot</b>.||</left>
			]]
		}
	},

	{
		name="ironFurnace",
		title=S("Iron Furnace"),
		icon="industrialtest:iron_furnace",
		content={
			[[
				<big>||Iron Furnace||</big>
				<left>
					||Iron Furnace works in similar way to regular furnace but it smelts items faster.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="recipe",
					value="Smelting"
				}
			},S("Iron Furnace")),
			[[
				<img name="industrialtest_guide_iron_furnace.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Iron Furnace smelting <item name="industrialtest:iron_dust" height="{{ITEM_HEIGHT}}"> <b>Iron Dust</b> into <item name="{{elementKeyIronIngot}}" height="{{ITEM_HEIGHT}}"> <b>Iron Ingot</b> using <item name="{{elementKeyCoal}}" height="{{ITEM_HEIGHT}}"> <b>Coal</b>.||</left>
			]]
		}
	},

	{
		name="macerator",
		title=S("Macerator"),
		icon="industrialtest:macerator",
		content={
			[[
				<big>||Macerator||</big>
				<left>
					||Macerator allows to crush various materials into their dust form. This is particularly useful with ores where it can be used for obtaining twice as much resources in some cases.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Macerating"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Macerator.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Macerator.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.Macerator.efficiency)
				}
			},S("Macerator")),
			[[
				<img name="industrialtest_guide_macerator.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Macerator crushing <item name="{{elementKeyIronLump}}" height="{{ITEM_HEIGHT}}"> <b>Iron Lump</b> into <item name="industrialtest:iron_dust" height="{{ITEM_HEIGHT}}"> <b>Iron Dust</b>.||</left>
			]]
		}
	},

	{
		name="magnetizer",
		title=S("Magnetizer"),
		icon="industrialtest:magnetizer",
		content={
			[[
				<big>||Magnetizer||</big>
				<left>
					||Magnetizer when used together with vertical rail made of <item name="industrialtest:iron_fence" height="{{ITEM_HEIGHT}}"> <b>Iron Fence</b> next to it creates small magnetic field around the fence. Such field allows for moving (as long as player wears some metal boots) upwards when jump key is pressed or downwards when sneak key is pressed. Each fence in rail adds to total power capacity that can be stored. Magnetizer automatically changes rotation to face detected fence rail.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU + %d EU per fence in rail",industrialtest.Magnetizer.capacity,industrialtest.Magnetizer._capacityPerFence)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Magnetizer._opPower)
				}
			},S("Magnetizer")),
			[[
				<img name="industrialtest_guide_magnetizer.png">
				<left>||Picture 1. Magnetizer powered by <item name="industrialtest:solar_panel" height="{{ITEM_HEIGHT}}"> <b>Solar Panel</b> with fence rail next to it.||</left>
			]]
		}
	},

	{
		name="massFabricator",
		title=S("Mass Fabricator"),
		icon="industrialtest:mass_fabricator",
		content={
			[[
				<big>||Mass Fabricator||</big>
				<left>
					||Mass Fabricator can transform energy and <item name="industrialtest:scrap" height="{{ITEM_HEIGHT}}"> <b>Scrap</b> into <item name="industrialtest:uu_matter" height="{{ITEM_HEIGHT}}"> <b>UU-Matter</b> which then can be later used to produce various materials.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="EV"
				},
				{
					name="recipe",
					value="Mass fabricating"
				},
				{
					name="Scraps per UU-Matter",
					value="34"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.MassFabricator.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.MassFabricator.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.MassFabricator.efficiency)
				}
			},S("Mass Fabricator")),
			[[
				<img name="industrialtest_guide_mass_fabricator.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Mass Fabricator producing <item name="industrialtest:uu_matter" height="{{ITEM_HEIGHT}}"> <b>UU-Matter</b> from <item name="industrialtest:scrap" height="{{ITEM_HEIGHT}}"> <b>Scrap</b>.||</left>
			]]
		}
	},

	{
		name="miner",
		title=S("Miner"),
		icon="industrialtest:miner",
		content={
			[[
				<big>||Miner||</big>
				<left>
					||Miner when placed on ground will use drill and <item name="industrialtest:mining_pipe" height="{{ITEM_HEIGHT}}"> <b>Mining Pipe</b> put in appropriate slots to dig below it. Besides that one can put <item name="industrialtest:ov_scanner" height="{{ITEM_HEIGHT}}"> <b>OV Scanner</b> or <item name="industrialtest:od_scanner" height="{{ITEM_HEIGHT}}"> <b>OD Scanner</b> to increase horizontal range of digging.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Miner.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Miner._opPower)
				},
				{
					name="Power per scanner operation",
					value=string.format("%d EU",industrialtest.Miner._scannerOpPower)
				}
			},S("Miner")),
			[[
				<img name="industrialtest_guide_miner_formspec.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Miner formspec with <item name="industrialtest:electric_drill" height="{{ITEM_HEIGHT}}"> <b>Electric Drill</b> and <item name="industrialtest:mining_pipe" height="{{ITEM_HEIGHT}}"> <b>Mining Pipe</b>.||</left>
				<mono>
				</mono>
				<img name="industrialtest_guide_miner_digging.png">
				<left>||Picture 2. Miner digging.||</left>
			]]
		}
	},

	{
		name="nuclearReactor",
		title=S("Nuclear Reactor"),
		icon="industrialtest:nuclear_reactor",
		content={
			[[
				<big>||Nuclear Reactor||</big>
				<left>
					||Nuclear Reactor produces energy via fission of <item name="industrialtest:uranium_cell" height="{{ITEM_HEIGHT}}"> <b>Uranium Cell</b>. It needs to be cooled using <item name="industrialtest:coolant_cell" height="{{ITEM_HEIGHT}}"> <b>Coolant Cell</b> or otherwise after keeping too much heat it will explode. The amount of generated energy depends on how many <item name="industrialtest:uranium_cell" height="{{ITEM_HEIGHT}}"> <b>Uranium Cell</b> is next to each other. It is possible to extend the capacity of the reactor to put more uranium and coolant by placing some <item name="industrialtest:nuclear_reactor_chamber" height="{{ITEM_HEIGHT}}"> <b>Nuclear Reactor Chamber</b> next to the reactor.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Reactor.capacity)
				}
			},S("Nuclear Reactor")),
			[[
				<img name="industrialtest_guide_nuclear_reactor_formspec.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Nuclear Reactor formspec with <item name="industrialtest:uranium_cell" height="{{ITEM_HEIGHT}}"> <b>Uranium Cell</b> and <item name="industrialtest:coolant_cell" height="{{ITEM_HEIGHT}}"> <b>Coolant Cell</b>.||</left>
				<mono>
				</mono>
				<img name="industrialtest_guide_nuclear_reactor_chambers.png">
				<left>||Picture 2. Nuclear Reactor with some <item name="industrialtest:nuclear_reactor_chamber" height="{{ITEM_HEIGHT}}"> <b>Nuclear Reactor Chamber</b>.||</left>
			]]
		}
	},

	{
		name="powerStorage",
		title=S("Power storage"),
		icon="industrialtest:batbox",
		content={
			[[
				<big>||Power storage||</big>
				<left>
					||This mod introduces a few nodes which allow for storing energy in them. It is useful in case some energy deficit happens or to charge electric items in them. Each of those have power input on each side except one marked with connector which is the output. Those are following:||
				</left>
				<mono>
				</mono>
				<left>
					<item name="industrialtest:batbox" height="{{ITEM_HEIGHT}}"> <b>||BatBox||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.BatBox.capacity)
				}
			},S("BatBox")),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:cesu" height="{{ITEM_HEIGHT}}"> <b>||CESU||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="MV"
				},
				{
					name="outputVoltage",
					value="MV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.CESU.capacity)
				}
			},S("CESU"),2),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:mfe" height="{{ITEM_HEIGHT}}"> <b>||MFE||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="HV"
				},
				{
					name="outputVoltage",
					value="HV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.MFE.capacity)
				}
			},S("MFE"),3),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:mfsu" height="{{ITEM_HEIGHT}}"> <b>||MFSU||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="EV"
				},
				{
					name="outputVoltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.MFSU.capacity)
				}
			},S("MFSU"),4),
			[[
				<mono></mono>
				<img name="industrialtest_guide_batbox.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. BatBox with some energy.||</left>
			]]
		}
	},

	{
		name="pump",
		title=S("Pump"),
		icon="industrialtest:pump",
		content={
			[[
				<big>||Pump||</big>
				<left>
					||Pump is used to pump fluids under it into internal fluid storage. It is later possible to move that fluid into some item, for example <item name="industrialtest:empty_cell" height="{{ITEM_HEIGHT}}"> <b>Empty Cell</b>. Pump when used together with <item name="industrialtest:miner" height="{{ITEM_HEIGHT}}"> <b>Miner</b> can pump fluids found.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Pump.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Pump._opPower)
				},
				{
					name="Fluid capacity",
					value=string.format("%d mB",industrialtest.Pump._fluidCapacity)
				},
				{
					name="Pump time",
					value=string.format("%d seconds",industrialtest.Pump._pumpTime)
				}
			},S("Pump")),
			[[
				<img name="industrialtest_guide_pump_formspec.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Pump with some fluid.||</left>
				<mono>
				</mono>
				<img name="industrialtest_guide_pump.png">
				<left>||Picture 1. Pump attached to <item name="industrialtest:miner" height="{{ITEM_HEIGHT}}"> <b>Miner</b>.||</left>
			]]
		}
	},

	{
		name="recycler",
		title=S("Recycler"),
		icon="industrialtest:recycler",
		content={
			[[
				<big>||Recycler||</big>
				<left>
					||Recycler with chance of 1/8 allows for transforming any item into <item name="industrialtest:scrap" height="{{ITEM_HEIGHT}}"> <b>Scrap</b>.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Recycling"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.Recycler.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.Recycler.opPower)
				},
				{
					name="efficiency",
					value=tostring(1/industrialtest.Recycler.efficiency)
				}
			},S("Recycler")),
			[[
				<img name="industrialtest_guide_recycler.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Recycler transforming some items.||</left>
			]]
		}
	},

	{
		name="rotaryMacerator",
		title=S("Rotary Macerator"),
		icon="industrialtest:rotary_macerator",
		content={
			[[
				<big>||Rotary Macerator||</big>
				<left>
					||Rotary Macerator works in similar way to regular Macerator with few differences. Like Induction Furnace the longer it runs the faster it works. Furthermore it provides additional slot for recipe modifier which can slightly alter the result.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="LV"
				},
				{
					name="recipe",
					value="Macerating + modifier"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.RotaryMacerator.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.RotaryMacerator._opPower)
				},
				{
					name="Power while maintaining energy",
					value=string.format("%d EU",industrialtest.RotaryMacerator._maintainSpeedOpPower)
				}
			},S("Rotary Macerator")),
			[[
				<img name="industrialtest_guide_rotary_macerator.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Rotary Macerator running with altered recipe.||</left>
			]]
		}
	},

	{
		name="solarPanels",
		title=S("Solar panels"),
		icon="industrialtest:solar_panel",
		content={
			[[
				<big>||Solar panels||</big>
				<left>
					||Solar panels are generators which allow for producting power from sun energy. There are few of them in this mod for different voltages. Those are:||
				</left>
				<mono>
				</mono>
				<left>
					<item name="industrialtest:solar_panel" height="{{ITEM_HEIGHT}}"> <b>||Solar Panel||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.SolarPanel.capacity)
				},
				{
					name="Generation multiplier",
					value=tostring(industrialtest.SolarPanel.multiplier)
				}
			},S("Solar Panel")),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:lv_solar_array" height="{{ITEM_HEIGHT}}"> <b>||LV Solar Array||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.LVSolarArray.capacity)
				},
				{
					name="Generation multiplier",
					value=tostring(industrialtest.LVSolarArray.multiplier)
				}
			},S("LV Solar Array"),2),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:mv_solar_array" height="{{ITEM_HEIGHT}}"> <b>||MV Solar Array||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="MV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.MVSolarArray.capacity)
				},
				{
					name="Generation multiplier",
					value=tostring(industrialtest.MVSolarArray.multiplier)
				}
			},S("MV Solar Array"),3),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:hv_solar_array" height="{{ITEM_HEIGHT}}"> <b>||HV Solar Array||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="HV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.HVSolarArray.capacity)
				},
				{
					name="Generation multiplier",
					value=tostring(industrialtest.HVSolarArray.multiplier)
				}
			},S("HV Solar Array"),4),
			[[
				<mono></mono>
				<img name="industrialtest_guide_solar_panel_full.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Solar Panel generating in full sunlight.||</left>
				<mono>
				</mono>
				<img name="industrialtest_guide_solar_panel_none.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 2. Solar Panel not generating.||</left>
			]]
		}
	},

	{
		name="toolWorkshop",
		title=S("Tool Workshop"),
		icon="industrialtest:tool_workshop",
		content={
			[[
				<big>||Tool Workshop||</big>
				<left>
					||Tool Workshop can be used to repair various tools with cost of energy. The process requires quite a lot energy and is slow.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="inputVoltage",
					value="HV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ToolWorkshop.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.ToolWorkshop._opPower)
				},
				{
					name="efficiency",
					value=tostring(industrialtest.ToolWorkshop._efficiency)
				}
			},S("Tool Workshop")),
			[[
				<img name="industrialtest_guide_tool_workshop.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Tool Workshop repairing <item name="{{elementKeyIronPickaxe}}" height="{{ITEM_HEIGHT}}"> <b>Iron Pickaxe</b>.||</left>
			]]
		}
	},

	{
		name="transformers",
		title=S("Transformers"),
		icon="industrialtest:lv_transformer",
		content={
			[[
				<big>||Transformers||</big>
				<left>
					||Transformer can be used to either bump up or lower the voltage between different voltage levels. Each transformer has one high voltage input/output side and five low voltage input/output sides. These are transformers in this mod:||
				</left>
				<mono>
				</mono>
				<left>
					<item name="industrialtest:lv_transformer" height="{{ITEM_HEIGHT}}"> <b>||LV Transformer||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="lowVoltage",
					value="LV"
				},
				{
					name="highVoltage",
					value="MV"
				}
			},S("LV Transformer")),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:mv_transformer" height="{{ITEM_HEIGHT}}"> <b>||MV Transformer||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="lowVoltage",
					value="MV"
				},
				{
					name="highVoltage",
					value="HV"
				}
			},S("MV Transformer"),2),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:hv_transformer" height="{{ITEM_HEIGHT}}"> <b>||HV Transformer||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="lowVoltage",
					value="HV"
				},
				{
					name="highVoltage",
					value="EV"
				}
			},S("HV Transformer"),3),
			[[
				<mono></mono>
				<left>
					<item name="industrialtest:ev_transformer" height="{{ITEM_HEIGHT}}"> <b>||EV Transformer||</b>
				</left>
			]],
			createMachineInformationTable({
				{
					name="lowVoltage",
					value="EV"
				},
				{
					name="highVoltage",
					value="IV"
				}
			},S("EV Transformer"),4),
			[[
				<mono></mono>
				<img name="industrialtest_guide_transformer.png">
				<left>||Picture 1. <item name="industrialtest:electric_furnace" height="{{ITEM_HEIGHT}}"> <b>Electric Furnace</b> connected to <item name="industrialtest:mv_solar_array" height="{{ITEM_HEIGHT}}"> <b>MV Solar Array</b> through <item name="industrialtest:lv_transformer" height="{{ITEM_HEIGHT}}"> <b>LV Transformer</b>.||</left>
			]]
		}
	},

	{
		name="waterMill",
		title=S("Water Mill"),
		icon="industrialtest:water_mill",
		content={
			[[
				<big>||Water Mill||</big>
				<left>
					||Water Mill is used to produce power from water kinetic force. There are two ways to supply water flow to Water Mill: putting it next to water or putting water in Water Mill for example by using <item name="industrialtest:empty_cell" height="{{ITEM_HEIGHT}}"> <b>Empty Cell</b>. The fluid level indicator in formspec shows the amount of water.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.WaterMill.capacity)
				}
			},S("Water Mill")),
			[[
				<img name="industrialtest_guide_water_mill_formspec.png" width="{{IMAGE_WIDTH}}">
				<left>||Picture 1. Water Mill with some water inside.||</left>
				<mono>
				</mono>
				<img name="industrialtest_guide_water_mill.png">
				<left>||Picture 2. Water Mill in water.||</left>
			]]
		}
	},

	{
		name="windMill",
		title=S("Wind Mill"),
		icon="industrialtest:wind_mill",
		content={
			[[
				<big>||Wind Mill||</big>
				<left>
					||Wind Mill can be used to produce energy from wind moving it's fan. The amount of produced power depends on height where Wind Mill is put, the higher the more power and it is indicated by icon in formspec. Wind Mill can't be put next to any node or otherwise it won't produce power.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="outputVoltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.WindMill.capacity)
				}
			},S("Wind Mill")),
			[[
				<img name="industrialtest_guide_wind_mill.png">
				<left>||Picture 1. Wind Mill producing some power.||</left>
			]]
		}
	},

	{
		name="upgrades",
		title=S("Machine upgrades"),
		icon="industrialtest:overclocker_upgrade",
		content={
			[[
				<big>||Machine upgrades||</big>
				<left>
					||Upgrades can be put into special slots in machine formspec for this purpose. They increase machine capabilities while changing other properties. Not all machines can accept upgrades, however this can be determined by presence of four slots on right of formspec where upgrades are put. The following are all machine upgrades:||
				</left>
				<left>
					||<item name="industrialtest:overclocker_upgrade" height="{{ITEM_HEIGHT}}"> <b>Overclocker Upgrade</b> - speeds up the machine.||
					||<item name="industrialtest:power_storage_upgrade" height="{{ITEM_HEIGHT}}"> <b>Power Storage Upgrade</b> - increases the amount of power that machine can store.||
					||<item name="industrialtest:transformer_upgrade" height="{{ITEM_HEIGHT}}"> <b>Transformer Upgrade</b> - allows machine to be connected to higher voltage by one level.||
				</left>
				<mono>--</mono>
				<img name="industrialtest_guide_upgrades.png">
				<left>||Picture 1. <item name="industrialtest:electric_furnace" height="{{ITEM_HEIGHT}}"> <b>Electric Furnace</b> with <item name="industrialtest:overclocker_upgrade" height="{{ITEM_HEIGHT}}"> <b>Overclocker Upgrade</b>.||</left>
			]]
		}
	},

	{
		name="batpacks",
		title=S("BatPacks"),
		icon="industrialtest:batpack_v",
		content={
			[[
				<big>||BatPacks||</big>
				<left>
					||BatPack when equipped charges currently wielded electric item. There are two types of BatPacks:||
				</left>
				<left>
					||<item name="industrialtest:batpack_v" height="{{ITEM_HEIGHT}}"> <b>BatPack</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.BatPack.capacity)
				}
			},S("BatPack")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:lappack_v" height="{{ITEM_HEIGHT}}"> <b>LapPack</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="MV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.LapPack.capacity)
				}
			},S("LapPack"),2)
		}
	},

	{
		name="electricToolset",
		title=S("Electric toolset"),
		icon="industrialtest:electric_drill",
		content={
			[[
				<big>||Electric toolset||</big>
				<left>
					||Electric toolset consists of typical tools like pickaxe but powered by electricity. While used they don't wear out but instead discharge which allow them to be reused. When completely discharged it's still possible to use the tool but it will be considerably slower. Those are (regular and advanced version):||
				</left>
				<left>
					||<item name="industrialtest:electric_chainsaw" height="{{ITEM_HEIGHT}}"> <b>Electric Chainsaw</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricChainsaw.capacity)
				}
			},S("Electric Chainsaw")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:electric_drill" height="{{ITEM_HEIGHT}}"> <b>Electric Drill</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricDrill.capacity)
				}
			},S("Electric Drill"),2),
			[[
				<mono></mono>
				||<item name="industrialtest:electric_hoe" height="{{ITEM_HEIGHT}}"> <b>Electric Hoe</b>||
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricHoe.capacity)
				}
			},S("Electric Hoe"),3),
			[[
				<mono></mono>
				||<item name="industrialtest:electric_saber" height="{{ITEM_HEIGHT}}"> <b>Electric Saber</b>||
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricSaber.capacity)
				}
			},S("Electric Saber"),4),
			[[
				<mono></mono>
				||<item name="industrialtest:electric_treetap" height="{{ITEM_HEIGHT}}"> <b>Electric Treetap</b>||
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricTreetap.capacity)
				}
			},S("Electric Treetap"),5)
		}
	},

	{
		name="itemPowerStorage",
		title=S("Item power storage"),
		icon="industrialtest:re_battery",
		content={
			[[
				<big>||Item power storage||</big>
				<left>
					||There are items which purpose is solely to store power in them. Those are following:||
				</left>
				<left>
					||<item name="industrialtest:re_battery" height="{{ITEM_HEIGHT}}"> <b>RE-Battery</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.REBattery.capacity)
				}
			},S("RE-Battery")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:advanced_re_battery" height="{{ITEM_HEIGHT}}"> <b>Advanced RE-Battery</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="MV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.AdvancedREBattery.capacity)
				}
			},S("Advanced RE-Battery"),2),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:energy_crystal" height="{{ITEM_HEIGHT}}"> <b>Energy Crystal</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="HV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.EnergyCrystal.capacity)
				}
			},S("Energy Crystal"),3),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:lapotron_crystal" height="{{ITEM_HEIGHT}}"> <b>Lapotron Crystal</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.LapotronCrystal.capacity)
				}
			},S("Lapotron Crystal"),4)
		}
	},

	{
		name="jetpacks",
		title=S("Jetpacks"),
		icon="industrialtest:jetpack_v",
		content={
			[[
				<big>||Jetpacks||</big>
				<left>
					||Jetpacks allow to combat against the gravity after pressing the jump key. They are equipped as chest part of armor. There are two jetpack types:||
				</left>
				<left>
					||<item name="industrialtest:jetpack_v" height="{{ITEM_HEIGHT}}"> <b>Jetpack</b> - regular jetpack which is powered by fuel, can be refilled in <item name="industrialtest:canning_machine" height="{{ITEM_HEIGHT}}"> <b>Canning Machine</b>.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="Fuel capacity",
					value=string.format("%d mB",industrialtest.Jetpack.capacity)
				}
			},S("Jetpack")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:electric_jetpack" height="{{ITEM_HEIGHT}}"> <b>Electric Jetpack</b> - jetpack powered by electricity.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="LV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.ElectricJetpack.capacity)
				}
			},S("Electric Jetpack"),2)
		}
	},

	{
		name="miningLaser",
		title=S("Mining Laser"),
		icon="industrialtest:mining_laser",
		content={
			[[
				<big>||Mining Laser||</big>
				<left>
					||Mining Laser is a tool that emits beam when used. Depending on the mode beam will cause different a result:||
					||Mode 1 - breaks nodes at full power, direction where player looks.||
					||Mode 2 - breaks nodes at low power, direction where player looks.||
					||Mode 3 - breaks nodes at full power, direction only horizontal.||
					||Mode 4 - explodes after touching node, direction where player looks.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="HV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.MiningLaser.capacity)
				},
				{
					name="Power per operation (Mode 1)",
					value="1250 EU"
				},
				{
					name="Power per operation (Mode 2)",
					value="100 EU"
				},
				{
					name="Power per operation (Mode 3)",
					value="100 EU"
				},
				{
					name="Power per operation (Mode 4)",
					value="5000 EU"
				}
			},S("Mining Laser"))
		}
	},

	{
		name="nanoSuit",
		title=S("Nano Suit"),
		icon="industrialtest:nano_bodyarmor",
		content={
			[[
				<big>||Nano Suit||</big>
				<left>
					||Nano Suit is an armor which with power of electricity can reduce damage which player takes. It consists of following parts:||
				</left>
				<left>
					||<item name="industrialtest:nano_helmet" height="{{ITEM_HEIGHT}}"> <b>Nano Helmet</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.NanoSuit.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.NanoSuit:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value="0.12"
				}
			},S("Nano Helmet")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:nano_bodyarmor" height="{{ITEM_HEIGHT}}"> <b>Nano Bodyarmor</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.NanoSuit.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.NanoSuit:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value="0.32"
				}
			},S("Nano Bodyarmor"),2),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:nano_leggings" height="{{ITEM_HEIGHT}}"> <b>Nano Leggings</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.NanoSuit.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.NanoSuit:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value="0.3"
				}
			},S("Nano Leggings"),3),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:nano_boots" height="{{ITEM_HEIGHT}}"> <b>Nano Boots</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="EV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.NanoSuit.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.NanoSuit:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value="0.24"
				}
			},S("Nano Boots"),4)
		}
	},

	{
		name="quantumSuit",
		title=S("Quantum Suit"),
		icon="industrialtest:quantum_bodyarmor",
		content={
			[[
				<big>||Quantum Suit||</big>
				<left>
					||Quantum Suit like Nano Suit is also capable of reducing damage with usage of electricity. However unlike the later it also provides various additional effects depending on armor part.||
				</left>
				<left>
					||<item name="industrialtest:quantum_helmet" height="{{ITEM_HEIGHT}}"> <b>Quantum Helmet</b> - when equipped refills player breath allowing for staying underwater longer.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="IV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.QuantumHelmet.capacity)
				},
				{
					name="Power per breath refill operation",
					value=string.format("%d EU",industrialtest.QuantumHelmet._breathRefillOpPower)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.QuantumHelmet:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value=tostring(industrialtest.QuantumHelmet:getReducedDamageForItem())
				}
			},S("Quantum Helmet")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:quantum_bodyarmor" height="{{ITEM_HEIGHT}}"> <b>Quantum Bodyarmor</b> - when equipped works like jetpack.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="IV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.QuantumBodyarmor.capacity)
				},
				{
					name="Power per fly operation",
					value=string.format("%d EU",industrialtest.ElectricJetpack._opPower)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.QuantumBodyarmor:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value=tostring(industrialtest.QuantumBodyarmor:getReducedDamageForItem())
				}
			},S("Quantum Bodyarmor"),2),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:quantum_leggings" height="{{ITEM_HEIGHT}}"> <b>Quantum Leggings</b> - when equipped increases player speed.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="IV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.QuantumLeggings.capacity)
				},
				{
					name="Power per distance unit",
					value=string.format("%d EU",industrialtest.QuantumLeggings._speedOpPower)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.QuantumLeggings:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value=tostring(industrialtest.QuantumLeggings:getReducedDamageForItem())
				}
			},S("Quantum Leggings"),3),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:quantum_boots" height="{{ITEM_HEIGHT}}"> <b>Quantum Boots</b> - when equipped increases player jump height and provides fall damage reduction.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="IV"
				},
				{
					name="powerCapacity",
					value=string.format("%d EU",industrialtest.QuantumBoots.capacity)
				},
				{
					name="Power per jump",
					value=string.format("%d EU",industrialtest.QuantumBoots._jumpOpPower)
				},
				{
					name="Power per fall damage unit",
					value=string.format("%d EU",industrialtest.QuantumBoots._fallDamageReductionOpPower)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.QuantumBoots:getPowerPerDamageForItem())
				},
				{
					name="damageRed",
					value=tostring(industrialtest.QuantumBoots:getReducedDamageForItem())
				}
			},S("Quantum Boots"),4)
		}
	},

	{
		name="scanners",
		title=S("Scanners"),
		icon="industrialtest:od_scanner",
		content={
			[[
				<big>||Scanners||</big>
				<left>
					||These tools provide way of detecting nearby ores in terrain. Such tool can be put to appropriate slot in <item name="industrialtest:miner" height="{{ITEM_HEIGHT}}"> <b>Miner</b> so it mines ores detected by scanner. Furthermore it can be used by player as well. There are currently 2 ore scanners which differ in detection range:||
				</left>
				<left>
					||<item name="industrialtest:od_scanner" height="{{ITEM_HEIGHT}}"> <b>OD Scanner</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="MV"
				},
				{
					name="powerStorage",
					value=string.format("%d EU",industrialtest.ODScanner.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.ODScanner:getOpPower())
				},
				{
					name="Distance",
					value=tostring(industrialtest.ODScanner.distance)
				},
				{
					name="Distance in Miner",
					value=tostring(industrialtest.ODScanner.minerDistance)
				}
			},S("OD Scanner")),
			[[
				<mono></mono>
				<left>
					||<item name="industrialtest:ov_scanner" height="{{ITEM_HEIGHT}}"> <b>OV Scanner</b>||
				</left>
			]],
			createMachineInformationTable({
				{
					name="voltage",
					value="MV"
				},
				{
					name="powerStorage",
					value=string.format("%d EU",industrialtest.OVScanner.capacity)
				},
				{
					name="opPower",
					value=string.format("%d EU",industrialtest.OVScanner:getOpPower())
				},
				{
					name="Distance",
					value=tostring(industrialtest.OVScanner.distance)
				},
				{
					name="Distance in Miner",
					value=tostring(industrialtest.OVScanner.minerDistance)
				}
			},S("OV Scanner"),2),
			[[
				<mono></mono>
				<img name="industrialtest_guide_od_scanner.png">
				<left>||Picture 1. <item name="industrialtest:od_scanner" height="{{ITEM_HEIGHT}}"> <b>OD Scanner</b> detecting some ores.||</left>
			]]
		}
	},

	{
		name="solarHelmet",
		title=S("Solar Helmet"),
		icon="industrialtest:solar_helmet",
		content={
			[[
				<big>||Solar Helmet||</big>
				<left>
					||Solar Helmet consists of <item name="{{elementKeyIronHelmet}}" height="{{ITEM_HEIGHT}}"> <b>Iron Helmet</b> with <item name="industrialtest:solar_panel" height="{{ITEM_HEIGHT}}"> <b>Solar Panel</b> attached to the top of it. It is capable of charging electric gear equipped as armor while exposed to light. However it is less efficent than regular solar panels.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="efficiency",
					value="0.5"
				}
			},S("Solar Panel"))
		}
	},

	{
		name="staticBoots",
		title=S("Static Boots"),
		icon="industrialtest:static_boots",
		content={
			[[
				<big>||Static Boots||</big>
				<left>
					||Static Boots are <item name="{{elementKeyIronBoots}}" height="{{ITEM_HEIGHT}}"> <b>Iron Boots</b> with <item name="industrialtest:insulated_copper_cable" height="{{ITEM_HEIGHT}}"> <b>Insulated Copper Cable</b> attached to the bottom. They are capable of charging electric gear equipped as armor by using static electricity which is produced while player is walking.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="Nodes per EU",
					value="5"
				}
			},S("Static Boots"))
		}
	},

	{
		name="treetap",
		title=S("Treetap"),
		icon=industrialtest.elementKeys.treetap,
		content={
			[[
				<big>||Treetap||</big>
				<left>
					||Treetap is used to extract <item name="{{elementKeyStickyResin}}" height="{{ITEM_HEIGHT}}"> <b>Sticky Resin</b> from <item name="{{elementKeyRubberWood}}" height="{{ITEM_HEIGHT}}"> <b>Rubber Wood</b>.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="uses",
					value=tostring(industrialtest.Treetap.uses)
				}
			},S("Treetap"))
		}
	},

	{
		name="wrench",
		title=S("Wrench"),
		icon="industrialtest:wrench",
		content={
			[[
				<big>||Wrench||</big>
				<left>
					||Wrench is one of the most important tools in this mod. It is used to safely remove machines with no material loss. In many cases when machine is removed without Wrench it drops <item name="industrialtest:machine_block" height="{{ITEM_HEIGHT}}"> <b>Machine Block</b> instead.||
				</left>
			]],
			createMachineInformationTable({
				{
					name="uses",
					value=tostring(industrialtest.Wrench.uses)
				}
			},S("Wrench"))
		}
	}
}

local function preprocessHypertext(content,vars)
	-- Find and replace strings which should be translated, this pattern matches any string that doesn't include | which is contained within || at both ends
	content=string.gsub(content,"||([^|]+)||",function(sourceString)
		-- Replace all sequences which should be treated as variables inside translation with format string
		local trFormatIndex=1
		local trFormatValues={}
		sourceString=string.gsub(sourceString,"{{([^{}]+)}}",function(var)
			local result=string.format("@%d",trFormatIndex)
			trFormatIndex=trFormatIndex+1
			table.insert(trFormatValues,vars[var] or "nil")
			return result
		end)
		return S(sourceString,unpack(trFormatValues))
	end)

	-- Replace all sequences which should be treated as variables with values provided in vars
	content=string.gsub(content,"{{([^{}]+)}}",function(var)
		return vars[var] or "nil"
	end)

	return content
end

local function getGuideFormspec(playerName,pageName)
	local PAGE_BUTTON_HEIGHT=0.7

	local formspec={
		"formspec_version[4]",
		"size[15,10.8]",
		"label[0.1,0.2;"..S("IndustrialTest Guide").."]",
		string.format("scrollbaroptions[max=%f]",(#pages-10.3/PAGE_BUTTON_HEIGHT)*PAGE_BUTTON_HEIGHT*10),
		"scrollbar[3.6,0.4;0.5,10.3;vertical;scrollbarList;0]",
		"scroll_container[0.1,0.4;4,10.3;scrollbarList;vertical;0.1]"
	}

	-- Contents sidebar
	for i,page in ipairs(pages) do
		table.insert(formspec,string.format("container[0,%f]",(i-1)*PAGE_BUTTON_HEIGHT))
		table.insert(formspec,string.format("item_image[0,0;%f,%f;%s]",PAGE_BUTTON_HEIGHT,PAGE_BUTTON_HEIGHT,page.icon))
		table.insert(formspec,string.format("button[%f,0;%f,%f;%s;%s]",PAGE_BUTTON_HEIGHT+0.05,3.45-PAGE_BUTTON_HEIGHT,PAGE_BUTTON_HEIGHT,page.name,page.title))
		table.insert(formspec,"container_end[]")
	end
	table.insert(formspec,"scroll_container_end[]")

	-- Introduction page is shown by default
	pageName=(pageName or "introduction")

	for _,page in ipairs(pages) do
		if page.name==pageName then
			local content=preprocessHypertext(table.concat(page.content,""),{
				IMAGE_WIDTH=550,
				ITEM_HEIGHT=16,
				elementKeyBucket=industrialtest.elementKeys.bucket,
				elementKeyCoal=industrialtest.elementKeys.coal,
				elementKeyCopperIngot=industrialtest.elementKeys.copperIngot,
				elementKeyIronBoots=industrialtest.elementKeys.ironBoots,
				elementKeyIronHelmet=industrialtest.elementKeys.ironHelmet,
				elementKeyIronIngot=industrialtest.elementKeys.ironIngot,
				elementKeyIronLump=industrialtest.elementKeys.ironLump,
				elementKeyIronPickaxe=industrialtest.elementKeys.ironPickaxe,
				elementKeyStickyResin=industrialtest.elementKeys.stickyResin,
				elementKeyRubber=industrialtest.elementKeys.rubber,
				elementKeyRubberWood=industrialtest.elementKeys.rubberWood
			})
			table.insert(formspec,string.format("hypertext[4.2,0.4;10.7,10.3;content;%s]",content))
			break
		end
	end

	return table.concat(formspec,"")
end

local function showGuide(playerName,page)
	minetest.show_formspec(playerName,"industrialtest:guide",getGuideFormspec(playerName,page))
	return true
end

local function handleGuideFields(player,formname,fields)
	if formname~="industrialtest:guide" then
		return
	end

	for _,page in ipairs(pages) do
		if fields[page.name] then
			minetest.close_formspec(player:get_player_name(),formname)
			showGuide(player:get_player_name(),page.name)
			break
		end
	end
end
minetest.register_on_player_receive_fields(handleGuideFields)

minetest.register_chatcommand("industrialtest_guide",{
	description=S("Shows graphical guide for content and features provided by IndustrialTest"),
	func=function(playerName)
		showGuide(playerName)
	end
})
