--
-- Helper function to extract the short name from a node's long name
--
-- e.g. "default:coble" -> "cobble"
--

local function get_short_name(long_name)
    local parts = {}

    for part in string.gmatch(long_name, '([^:]+)') do
        table.insert(parts, part)
    end

    if #parts == 2 then
        return parts[2]
    end

    return "unknown"
end


--
-- Create and register a new mesecore nodes (one for the off state and one for the on state)
--

function camomese.register_mesecore_nodes(original_node_name, node_description)
    -- Make 2 copies of the original node, for the off and on states respectively
    local off_state_node = table.copy(core.registered_nodes[original_node_name])
    local on_state_node = table.copy(core.registered_nodes[original_node_name])

	-- Extract the short name from the full name of the original node
    local short_name = get_short_name(original_node_name)


    --
    -- Description
    --

    -- Sets the description for the off state node
    off_state_node.description = node_description

    -- Remove the description from the on state node as it isn't needed
    on_state_node.description = nil


    ---
    --- Tiles
    ---

    -- Add overlays to the off state node
    for idx, tile in ipairs(off_state_node.tiles) do
        if type(tile) == "string" then
            off_state_node.tiles[idx] = tile .. "^camomese_mesecore_off.png"
        elseif type(tile) == "table" and tile.name then
            off_state_node.tiles[idx].name = tile.name .. "^camomese_mesecore_off.png"
        end
    end

    -- Add overlays to the on state node
    for idx, tile in ipairs(on_state_node.tiles) do
        if type(tile) == "string" then
            on_state_node.tiles[idx] = tile .. "^camomese_mesecore_on.png"
        elseif type(tile) == "table" and tile.name then
            on_state_node.tiles[idx].name = tile.name .. "^camomese_mesecore_on.png"
        end
    end


    --
    -- Mesecons rules
    --

    off_state_node.mesecons = {
        conductor = {
            state = mesecon.state.off,
            rules = mesecon.rules.alldirs,
            onstate = "camomese:mesecore_" .. short_name .. "_on"
        }
    }

    on_state_node.mesecons = {
        conductor = {
            state = mesecon.state.on,
            rules = mesecon.rules.alldirs,
            offstate = "camomese:mesecore_" .. short_name
        }
    }


    --
    -- The on state node drops the off state node
    --

    on_state_node.drop = "camomese:mesecore_" .. short_name


    --
    -- Register the new nodes
    --

	core.register_node("camomese:mesecore_" .. short_name, off_state_node)
	core.register_node("camomese:mesecore_" .. short_name .. "_on", on_state_node)


    --
    -- Register the recipe for the new node
    --

    core.register_craft({
        type = "shaped",
        output = "camomese:mesecore_" .. short_name .. " 8",
        recipe = {
            {original_node_name, original_node_name, original_node_name},
            {original_node_name, "default:mese_crystal_fragment", original_node_name},
            {original_node_name, original_node_name, original_node_name},
        }
    })
end


function camomese.register_digicore_node(original_node_name, node_description)
    -- Make a copy of the original node
    local new_node = table.copy(core.registered_nodes[original_node_name])

	-- Extract the short name from the full name of the original node
    local short_name = get_short_name(original_node_name)


    --
    -- Description
    --

    -- Set the description for the new node
    new_node.description = node_description


    ---
    --- Tiles
    ---

    -- Add overlays to the off state node
    for idx, tile in ipairs(new_node.tiles) do
        if type(tile) == "string" then
            new_node.tiles[idx] = tile .. "^camomese_digicore.png"
        elseif type(tile) == "table" and tile.name then
            new_node.tiles[idx].name = tile.name .. "^camomese_digicore.png"
        end
    end


    --
    -- Mesecons rules
    --

    new_node.digilines = {
        wire = {
            rules = {
                { x =  0, y =  0, z = -1 },
                { x =  1, y =  0, z =  0 },
                { x = -1, y =  0, z =  0 },
                { x =  0, y =  0, z =  1 },
                { x =  0, y =  1, z =  0 },
                { x =  0, y = -1, z =  0 }
            },
        },
    }


    --
    -- Register the new nodes
    --

	core.register_node("camomese:digicore_" .. short_name, new_node)


    --
    -- Register the recipe for the new node
    --

    core.register_craft({
        type = "shaped",
        output = "camomese:digicore_" .. short_name .. " 8",
        recipe = {
            {original_node_name, original_node_name, original_node_name},
            {original_node_name, "digilines:wire_std_00000000", original_node_name},
            {original_node_name, original_node_name, original_node_name},
        }
    })
end
