local utils = ...



local function get_env_table (src)
	local env = { }

	local fxn, msg = loadstring (src)

	if fxn then
		setfenv (fxn, env)

		local success, value = pcall (fxn)

		if success then
			return true, env
		end

		return false, value
	end

	return false, msg
end



local function deserializer (src, env)
	local fxn, msg = loadstring (src)

	if fxn then
		setfenv (fxn, env)

		return pcall (fxn)
	end

	return false, msg
end



local zero_code = ("0"):byte (1, 1)
local nine_code = ("9"):byte (1, 1)
local dot_code = ("."):byte (1, 1)
local minus_code = ("-"):byte (1, 1)
local plus_code = ("+"):byte (1, 1)


local std_names =
{
	["nil"] = true,
	["true"] = true,
	["false"] = true
}



local parser = { }



function parser:get_string (str, pos)
	pos = pos or 1

	if pos > 0 and pos < str:len () then
		local delim_start = str:find ("%S", pos, false)

		if not delim_start then
			return nil
		end

		local delim = str:sub (delim_start, delim_start)
		local escaped

		if delim == "\"" then
			escaped = "\\\""
		elseif delim == "'" then
			escaped = "\\'"
		else
			return nil
		end

		pos = delim_start + 1

		while true do
			local delim_end = str:find (delim, pos, true)

			if not delim_end then
				return nil
			elseif str:sub (delim_end - 1, delim_end) == escaped then
				pos = delim_end + 1
			else
				return delim_start, delim_end, str:sub (delim_start, delim_end)
			end
		end
	end

	return nil
end



function parser:get_number (str, pos)
	pos = pos or 1

	if pos > 0 and pos < str:len () then
		local next_char = str:find ("%S", pos, false)

		if not next_char then
			return nil
		end

		local number_start, number_end = str:find ("(([%+%-]?)(%d*)(%.?)(%d*))", pos, false)

		if next_char ~= number_start then
			return nil
		end

		return number_start, number_end, str:sub (number_start, number_end)
	end

	return nil
end



function parser:get_value_type (str, pos)
	pos = pos or 1

	if pos > 0 and pos < str:len () then
		local next_char = str:find ("%S", pos, false)

		if not next_char then
			return nil
		end

		local char = str:sub (next_char, next_char)
		local code = char:byte (1, 1)

		if char == "\"" or char == "'" then
			return "string"

		elseif (code >= zero_code and code <= nine_code) or
				code == dot_code or code == minus_code or code == plus_code then
			return "number"

		elseif char == "{" then
			return "table"

		else
			return "name"
		end
	end

	return nil
end



function parser:get_table (str, pos)
	pos = pos or 1

	if pos > 0 and pos < str:len () then
		local next_char = str:find ("%S", pos, false)

		if not next_char or next_char ~= str:find ("{", pos, true) then
			return nil
		end

		local delim_open = next_char

		next_char = str:find ("%S", delim_open + 1, false)

		while next_char do
			local char = str:sub (next_char, next_char)

			if char == "}" then
				return delim_open, next_char, str:sub (delim_open, next_char)

			elseif char == "[" then
				if self:get_value_type (str, next_char + 1) == "table" then
					return nil
				end

				local name_start, name_end = self:get_value (str, next_char + 1)

				if not name_start then
					return nil
				end

				local value_start, value_end = str:find ("%S", name_end + 1, false)

				if not value_start or value_start ~= (str:find ("]", name_end + 1, true)) then
					return nil
				end

				local equals = str:find ("%S", value_end + 1, false)

				if not equals or equals ~= (str:find ("=", value_end + 1, true)) then
					return nil
				end

				value_start, value_end = self:get_value (str, equals + 1)

				if not value_start then
					return nil
				end

				next_char =  str:find ("%S", value_end + 1, false)

			else
				local value_type = parser:get_value_type (str, next_char)

				if value_type == "name" then
					local name_start, name_end = str:find ("(%w+)", next_char, false)

					if not name_start then
						return nil
					end

					local equals = str:find ("%S", name_end + 1, false)

					if equals and equals == (str:find ("=", name_end + 1, true)) then
						local value_start, value_end = self:get_value (str, equals + 1)

						if not value_start then
							return nil
						end

						next_char =  str:find ("%S", value_end + 1, false)
					else
						next_char =  str:find ("%S", name_end + 1, false)
					end

				else
					local value_start, value_end = self:get_value (str, next_char)

					if not value_start then
						return nil
					end

					next_char =  str:find ("%S", value_end + 1, false)
				end
			end

			if next_char then
				local delim = str:sub (next_char, next_char)

				if delim == "}" then
					return delim_open, next_char, str:sub (delim_open, next_char)

				elseif delim == "," then
					local check_char = str:find ("%S", next_char + 1, false)

					if not check_char then
						return nil
					end

					if str:sub (check_char, check_char) == "}" then
						return delim_open, check_char, str:sub (delim_open, check_char)
					end

					next_char = str:find ("%S", next_char + 1, false)

				else
					return nil
				end
			end
		end
	end

	return nil
end



local function parser_deserialize_table (str, pos, env)
	pos = pos or 1
	local result = { }
	env = env or { }
	local err_msg = "format error"

	if pos > 0 and pos < str:len () then
		local next_char = str:find ("%S", pos, false)

		if not next_char or next_char ~= (str:find ("{", pos, true)) then
			return nil, err_msg
		end

		local delim_open = next_char

		next_char = str:find ("%S", delim_open + 1, false)

		while next_char do
			local char = str:sub (next_char, next_char)

			if char == "}" then
				return delim_open, next_char, result

			elseif char == "[" then
				local value
				local success

				if parser:get_value_type (str, next_char + 1) == "table" then
					return nil, err_msg
				end

				local name_start, name_end, name = parser:get_value (str, next_char + 1)

				if not name_start then
					return nil, err_msg
				end

				local value_start, value_end = str:find ("%S", name_end + 1, false)

				if not value_start or str:sub (value_start, value_start) ~= "]" then
					return nil, err_msg
				end

				local equals = str:find ("%S", value_end + 1, false)

				if not equals or equals ~= (str:find ("=", value_end + 1, true)) then
					return nil, err_msg
				end

				value_start, value_end, value = parser:get_value (str, equals + 1)

				if not value_start then
					return nil, err_msg
				end

				if not std_names[name] then
					success, name =  deserializer ("return "..name, env)

					if not success or not name then
						return nil, name
					end
				end

				success, value = deserializer ("return "..value, env)

				if not success then
					return nil, value
				end

				result[name] = value

				next_char =  str:find ("%S", value_end + 1, false)

			else
				local value_type = parser:get_value_type (str, next_char)

				if value_type == "name" then
					local name_start, name_end, name = str:find ("(%w+)", next_char, false)

					if not name_start then
						return nil, err_msg
					end

					local equals = str:find ("%S", name_end + 1, false)

					if equals and equals == (str:find ("=", name_end + 1, true)) then
						local success
						local value_start, value_end, value = parser:get_value (str, equals + 1)

						if not value_start then
							return nil, err_msg
						end

						success, value = deserializer ("return "..value, env)

						if not success then
							return nil, value
						end

						result[name] = value

						next_char =  str:find ("%S", value_end + 1, false)
					else
						local success

						success, name = deserializer ("return "..name, env)

						if not success then
							return nil, name
						end

						result[#result + 1] = name

						next_char =  str:find ("%S", name_end + 1, false)
					end

				elseif value_type == "table" then
					local value_start, value_end, inner = parser_deserialize_table (str, next_char, env)

					if not value_start then
						return nil, value_end
					end

					result[#result + 1] = inner

					next_char =  str:find ("%S", value_end + 1, false)

				else
					local success
					local value_start, value_end, value = parser:get_value (str, next_char)

					if not value_start then
						return nil, err_msg
					end

					success, value = deserializer ("return "..value, env)

					if not success then
						return nil, value
					end

					result[#result + 1] = value

					next_char =  str:find ("%S", value_end + 1, false)
				end
			end

			if next_char then
				local delim = str:sub (next_char, next_char)

				if delim == "}" then
					return delim_open, next_char, result

				elseif delim == "," then
					local check_char = str:find ("%S", next_char + 1, false)

					if not check_char then
						return nil, err_msg
					end

					if str:sub (check_char, check_char) == "}" then
						return delim_open, check_char, result
					end

					next_char = str:find ("%S", next_char + 1, false)

				else
					return nil, err_msg
				end

				if (utils.long_process.timer () + 0.08) < os.clock () then
					local breaker = utils.long_process.breaker ()

					if breaker then
						breaker ()
					end
				end
			end
		end
	end

	return nil, err_msg
end



function parser:deserialize_table (str, pos, env)
	local first, last, value = parser_deserialize_table (str, pos, env)

	if first then
		return value
	end

	return first, last
end



function parser:get_value (str, pos)
	pos = pos or 1

	if pos > 0 and pos < str:len () then
		local next_char = str:find ("%S", pos, false)

		if not next_char then
			return nil
		end

		local value_type = parser:get_value_type (str, next_char)

		if value_type == "string" then
			local value_start, value_end, value = self:get_string (str, next_char)

			if value_start then
				return value_start, value_end, value, "string"
			end

		elseif value_type == "number" then
			local value_start, value_end, value = self:get_number (str, next_char)

			if value_start then
				return value_start, value_end, value, "number"
			end

		elseif value_type == "table" then
			local value_start, value_end, value = self:get_table (str, next_char)

			if value_start then
				return value_start, value_end, value, "table"
			end

		elseif value_type == "name" then
			local value_start, value_end = str:find ("(%w+)", next_char, false)

			if value_start then
				return value_start, value_end, str:sub (value_start, value_end), "name"
			end
		end
	end

	return nil
end



local function get_strings_and_content (str)
	local table_start = -1
	local pos = 1
	while pos < str:len () do
		local name_start, name_end = str:find ("(%w+)", pos, false)

		if not name_start or name_start ~= (str:find ("%S", pos, false)) then
			break
		end

		local name = str:sub (name_start, name_end)

		if name == "return" then
			table_start = name_start
			break
		end

		local equals = str:find ("(=)", name_end + 1, false)

		if not equals or equals ~= (str:find ("%S", name_end + 1, false)) then
			break
		end

		local delim_start = str:find ("(\")", equals + 1, false)

		if not delim_start or delim_start ~= (str:find ("%S", equals + 1, false)) then
			break
		end

		pos = delim_start + 1

		local delim_end
		while not delim_end do
			local quote = str:find ("(\")", pos, false)

			if not quote then
				break
			elseif str:sub (quote - 1, quote) == "\\\"" then
				pos = quote + 1
			else
				delim_end = quote
			end
		end

		if not delim_end then
			break
		end

		local semi_colon = str:find ("(;)", delim_end + 1, false)

		if not semi_colon or semi_colon ~= (str:find ("%S", delim_end + 1, false)) then
			break
		end

		pos = semi_colon + 1
	end

	if table_start < 1 then
		return nil
	end

	local data = str:sub (table_start + 6):trim ()

	if data:sub (1, 1) ~= "{" or data:sub (-1) ~= "}" then
		return nil
	end

	return str:sub (1, table_start - 1), data
end



local newline_code = ("\n"):byte (1, 1)
local return_code = ("\r"):byte (1, 1)
local tab_code = ("\t"):byte (1, 1)
local space_code = (" "):byte (1, 1)
local tild_code = ("~"):byte (1, 1)

local function check_for_bytecode (src)
	local code = src:byte (1)

	if code ~= newline_code and code ~= return_code and code ~= tab_code and
			(code < space_code or code > tild_code) then
		return true
	end

	return false
end



local function deserialize_table (src)
	if check_for_bytecode (src) then
		return false, "byte code"
	end

	if not jit or src:len () < 400000 then
		return pcall (minetest.deserialize, src, true)

	else
		local string_code, content = get_strings_and_content (src)

		if not string_code then
			return false, "format error"
		end

		local env = { }
		if string_code:len () > 0 then
			local success

			success, env = get_env_table (string_code)

			if not success then
				return false, env
			end
		end

		local nodes, msg = parser:deserialize_table (content, 1, env)

		if nodes then
			return true, nodes
		end

		return false, msg
	end
end



local function deserialize (src)
	if src then
		local first = src:find ("%S", 1, false)

		if first then
			if src:sub (first, first + 6) == "return" then
				first = src:find ("%S", first + 7, false)

				if not first then
					return false, "format error"

				elseif src:sub (first, first) ~= "{" then
					if check_for_bytecode (src) then
						return false, "byte code"
					end

					return pcall (minetest.deserialize, src, true)
				end
			end

			return deserialize_table (src)
		end

		return false, "format error"
	end

	return false, "deserialising nil source"
end



function utils.deserialize (src)
	local success, value = deserialize (src)

	if success then
		return value
	end

	error (value, 2)
end
