--[[
	AND (&): Performs a bitwise AND operation.
	OR (|): Performs a bitwise OR operation.
	XOR (~): Performs a bitwise exclusive OR operation.
	NOT (~): Performs a unary bitwise NOT operation.
	Left Shift (<<): Shifts bits to the left.
	Right Shift (>>): Shifts bits to the right12.

2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 
31, 37, 41, 43, 47, 53, 59, 61, 67, 
71, 73, 79, 83, 89, 97, 101, 
103, 107, 109, 113, 127, 131, 
137, 139, 149, 151, 
157, 163, 167, 173, 179, 181, 191, 
193, 197, 199, 211, 223, 227, 229, 233, 
239, 241, 251, 257, 263, 269, 271, 277, 
281, 283, 293, 307, 311, 313, 
317, 331, 337, 347, 349, 353, 
359, 367, 373, 379, 383, 
389, 397, 401, 409, 419, 421, 431, 433, 439, 
443, 449, 457, 461, 463, 467, 479, 487, 491, 499, 
503, 509, 521, 523, 541, 547, 557, 563, 569, 571, 
577, 587, 593, 599, 601, 607, 613, 617, 619, 631, 
641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 
701, 709, 719, 727, 733, 739, 743, 751, 757, 761, 
769, 773, 787, 797, 809, 811, 821, 823, 827, 829, 
839, 853, 857, 859, 863, 877, 881, 883, 887, 907, 
911, 919, 929, 937, 941, 947, 953, 967, 971, 977, 
983, 991, 997

--]]

-- Bitwise AND
function band(a, b)
	local result = 0
	local bitval = 1
	while a > 0 and b > 0 do
		local abit = a % 2
		local bbit = b % 2
		if abit == 1 and bbit == 1 then
			result = result + bitval
		end
		a = math.floor(a / 2)
		b = math.floor(b / 2)
		bitval = bitval * 2
	end
	return result
end

-- Bitwise OR
function bor(a, b)
	local result = 0
	local bitval = 1
	while a > 0 or b > 0 do
		local abit = a % 2
		local bbit = b % 2
		if abit == 1 or bbit == 1 then
			result = result + bitval
		end
		a = math.floor(a / 2)
		b = math.floor(b / 2)
		bitval = bitval * 2
	end
	return result
end

-- Bitwise XOR
function bxor(a, b)
	local result = 0
	local bitval = 1
	while a > 0 or b > 0 do
		local abit = a % 2
		local bbit = b % 2
		if abit ~= bbit then
			result = result + bitval
		end
		a = math.floor(a / 2)
		b = math.floor(b / 2)
		bitval = bitval * 2
	end
	return result
end

-- Bitwise NOT
function bnot(a)
	local result = 0
	local bitval = 1
	while a > 0 do
		local abit = a % 2
		if abit == 0 then
			result = result + bitval
		end
		a = math.floor(a / 2)
		bitval = bitval * 2
	end
	return result
end

-- Left shift
function lshift(x, by)
	return x * 2 ^ by
end

-- Right shift
function rshift(x, by)
	return math.floor(x / 2 ^ by)
end


local gen_bitwise_assert = function(a,b)
print("AND: ", band(a, b))
print("OR: ", bor(a, b))
print("XOR: ", bxor(a, b))
print("NOT: ", bnot(271))
print("Left Shift: ", lshift(a, 1))
print("Right Shift: ", rshift(a, 1))
end
local a, b = 127, 271
--gen_bitwise_assert(a,b)
assert(band(a, b) == 15)
assert(bor(a, b) == 383)
assert(bxor(a, b) == 368)
assert(bnot(b) == 240)
assert(lshift(a, 1) == 254)
assert(rshift(a, 1) == 63)

