local PEDESTAL = {}
local REGISTERED_PEDESTALS = {}

local function get_pedestal_def(node_name)
    return REGISTERED_PEDESTALS[node_name]
end

radapi.v3.entity.register("radapi", "pedestal_item", {
    initial_properties = {
        visual = "wielditem",
        visual_size = {x=1, y=1},
        wield_item = "default:stone",
        pointable = false,
        is_visible = true,
        physical = false,
        collisionbox = {0, 0, 0, 0, 0, 0},
        selectionbox = {0, 0, 0, 0, 0, 0},
        static_save = true,
    },

    on_activate = function(self, staticdata, dtime_s, state)
        if state.itemstack then
            local item = ItemStack(state.itemstack)
            self.object:set_properties({
                wield_item = item:get_name(),
                textures = {item:get_name()}
            })
        end
    end,

    on_deactivate = function(self, removal, state)
        if removal and state.pedestal_pos then
            local meta = core.get_meta(state.pedestal_pos)
            meta:set_string("has_item", "")
            meta:set_string("entity_guid", "")
        end
    end,
})

core.register_on_placenode(function(pos, newnode, placer, oldnode, itemstack, pointed_thing)
    local pedestal_pos = {x=pos.x, y=pos.y-1, z=pos.z}
    local pedestal_node = core.get_node(pedestal_pos)
    
    if not get_pedestal_def(pedestal_node.name) then
        return
    end

    local meta = core.get_meta(pedestal_pos)

    if meta:get_string("has_item") == "true" then
        core.remove_node(pos) 
        return
    end

    core.remove_node(pos)
    local pedestal_def = get_pedestal_def(pedestal_node.name)
    local entity_name = pedestal_def.entity_name or "pedestal_api:pedestal_item"
    
    local entity = radapi.v3.entity.spawn(pos, entity_name, {
        itemstack = itemstack:to_table(),
        pedestal_pos = pedestal_pos
    })

    if entity then
        meta:set_string("has_item", "true")
        meta:set_string("entity_guid", entity:get_guid())
        
        if pedestal_def.on_display then
            pedestal_def.on_display(pos, itemstack, placer, pedestal_node, entity)
        end
    end
end)

function PEDESTAL.register(modname, name, def)
    assert(type(def) == "table", "Pedestal definition must be a table")
    assert(modname and name, "Missing modname or pedestal name")
    
    local def_id = modname .. ":" .. name
    assert(not REGISTERED_PEDESTALS[def_id], "Pedestal already registered: " .. def_id)
    
    REGISTERED_PEDESTALS[def_id] = {
        entity_name = def.entity_name,
        on_display = def.on_display,
        on_remove = def.on_remove,
        on_rotate = def.on_rotate
    }

    local node_def = table.copy(def)
    node_def.groups = node_def.groups or {}
    node_def.groups.not_in_creative_inventory = 1
    
    node_def.is_pedestal = true
    
    node_def.on_construct = function(pos)
        local meta = core.get_meta(pos)
        meta:set_string("has_item", "")
        meta:set_string("entity_guid", "")
        
        if def.on_construct then
            def.on_construct(pos)
        end
    end

    node_def.on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
        local meta = core.get_meta(pos)
        local guid = meta:get_string("entity_guid")
        
        if guid == "" then
            if def.on_rightclick then
                return def.on_rightclick(pos, node, clicker, itemstack, pointed_thing)
            end
            return
        end

        local entity = core.get_object_by_guid(guid)
        if not entity then
            meta:set_string("has_item", "")
            meta:set_string("entity_guid", "")
            return
        end

        local entity_data = entity:get_luaentity()
        local itemstack_obj = entity_data and entity_data.itemstack and ItemStack(entity_data.itemstack)
        
        if itemstack:is_empty() then
            if itemstack_obj then
                local pedestal_def = get_pedestal_def(node.name)
                local final_item = itemstack_obj
                
                if pedestal_def and pedestal_def.on_remove then
                    local result = pedestal_def.on_remove(pos, final_item, clicker, node)
                    if result then final_item = result end
                end
                
                local inv = clicker:get_inventory()
                if inv:room_for_item("main", final_item) then
                    inv:add_item("main", final_item)
                    entity:remove()
                    meta:set_string("has_item", "")
                    meta:set_string("entity_guid", "")
                end
            end
        else
            local rotation = entity:get_rotation()
            rotation.y = rotation.y + math.rad(45)
            entity:set_rotation(rotation)
            
            local pedestal_def = get_pedestal_def(node.name)
            if pedestal_def and pedestal_def.on_rotate then
                pedestal_def.on_rotate(pos, itemstack_obj, clicker, node, entity)
            end
        end
    end

    node_def.after_destruct = function(pos, oldnode)
        local meta = core.get_meta(pos)
        local guid = meta:get_string("entity_guid")
        
        if guid ~= "" then
            local entity = core.get_object_by_guid(guid)
            if entity then entity:remove() end
        end
        
        if def.after_destruct then
            def.after_destruct(pos, oldnode)
        end
    end

    core.register_node(def_id, node_def)
end

function PEDESTAL.unregister(def_id)
    assert(type(def_id) == "string", "def_id must be a string")
    
    if not REGISTERED_PEDESTALS[def_id] then
        core.log("error", "[PEDESTAL] Pedestal definition not registered: " .. def_id)
        return false
    end

    local has_active = false
    for _, node in ipairs(core.find_nodes_with_name(def_id)) do
        local meta = core.get_meta(node)
        if meta:get_string("has_item") == "true" then
            has_active = true
            break
        end
    end

    if has_active then
        core.log("error", "[PEDESTAL] Cannot unregister pedestal '" .. def_id .. "' - active instances exist")
        return false
    end

    REGISTERED_PEDESTALS[def_id] = nil
    core.log("info", "[PEDESTAL] Unregistered pedestal: " .. def_id)
    return true
end

pedestal_api = PEDESTAL
