# Pedestals API Documentation

**A professional API for creating customizable item-display pedestals in Minetest**
---

## Table of Contents
- [Introduction](#introduction)
- [Installation](#installation)
- [API Reference](#api-reference)
  - [`pedestals_api.register()`](#pedestals_apiregister)
  - [`pedestals_api.unregister()`](#pedestals_apiunregister)
- [Pedestal Definition Properties](#pedestal-definition-properties)
- [Event Handlers](#event-handlers)
- [Example Implementation](#example-implementation)
- [Dependencies](#dependencies)

---

## Introduction

The **Pedestals API** provides a robust framework for creating customizable item-holding pedestal nodes in Minetest. This API handles all entity management, rotation, and item interactions automatically, allowing mod developers to create unique pedestal experiences with minimal code.

Key features:
- Automatic entity spawning for displayed items
- Customizable rotation behavior (45° increments)
- Item retrieval via right-click
- Full event hook system for mod integration
- Safe entity cleanup on node destruction
- Support for multiple pedestal types

---

## Installation

## API Reference

### `pedestals_api.register(modname, name, def)`

Registers a new pedestal node type.

**Parameters:**
| Parameter | Type     | Description                                                                 |
|-----------|----------|-----------------------------------------------------------------------------|
| `modname` | `string` | Name of the mod registering the pedestal (e.g., `"my_mod"`)                 |
| `name`    | `string` | Unique identifier for the pedestal (e.g., `"stone_pedestal"`)               |
| `def`     | `table`  | Pedestal definition (see [Pedestal Definition Properties](#pedestal-definition-properties)) |

**Example:**
```lua
pedestals_api.register("my_mod", "stone_pedestal", {
    description = "Stone Pedestal",
    tiles = {"default_stone.png"},
    groups = {cracky = 3, oddly_breakable_by_hand = 2},
    on_display = function(pos, itemstack, placer, node, entity)
        minetest.chat_send_player(placer:get_player_name(), "Item displayed!")
    end
})
```

---

### `pedestals_api.unregister(def_id)`

Unregisters a pedestal type (only if no active instances exist).

**Parameters:**
| Parameter | Type     | Description                                                                 |
|-----------|----------|-----------------------------------------------------------------------------|
| `def_id`  | `string` | Full ID of the pedestal (e.g., `"my_mod:stone_pedestal"`)                   |

**Returns:**
- `true` if unregistered successfully
- `false` if active pedestals exist or pedestal isn't registered

**Example:**
```lua
if pedestals_api.unregister("my_mod:stone_pedestal") then
    minetest.log("info", "Successfully unregistered stone pedestal")
end
```

---

## Pedestal Definition Properties

The `def` table passed to `register()` supports these properties:

| Property             | Type     | Required | Description                                                                 |
|----------------------|----------|----------|-----------------------------------------------------------------------------|
| `description`        | `string` | Yes      | Node description (shown in inventory)                                       |
| `tiles`              | `table`  | Yes      | Node textures (array of texture paths)                                      |
| `groups`             | `table`  | No       | Node groups (automatically adds `not_in_creative_inventory = 1`)            |
| `entity_name`        | `string` | No       | Custom entity name (default: `"pedestals_api:pedestal_item"`)               |
| `on_display`         | `function`| No       | [Event handler](#event-handlers) called when item is placed                 |
| `on_remove`          | `function`| No       | [Event handler](#event-handlers) called when item is retrieved              |
| `on_rotate`          | `function`| No       | [Event handler](#event-handlers) called when item is rotated                |
| `on_construct`       | `function`| No       | Called when pedestal node is created                                        |
| `after_destruct`     | `function`| No       | Called when pedestal node is destroyed                                      |

> **Note:** All pedestal nodes automatically get `not_in_creative_inventory = 1` group.

---

## Event Handlers

### `on_display(pos, itemstack, placer, node, entity)`

**Trigger:** When an item is placed on the pedestal  
**Parameters:**
- `pos` (table): Pedestal position
- `itemstack` (ItemStack): Item being displayed
- `placer` (Player): Player who placed the item
- `node` (table): Pedestal node definition
- `entity` (Entity): Entity representing the item

---

### `on_remove(pos, itemstack, placer, node)`

**Trigger:** When item is retrieved (right-click with empty hand)  
**Parameters:**
- `pos` (table): Pedestal position
- `itemstack` (ItemStack): Item being removed
- `placer` (Player): Player retrieving the item
- `node` (table): Pedestal node definition

**Return:** Optional new itemstack (will be placed in player inventory)

---

### `on_rotate(pos, itemstack, placer, node, entity)`

**Trigger:** When item is rotated (right-click with item)  
**Parameters:**
- `pos` (table): Pedestal position
- `itemstack` (ItemStack): Item in player's hand
- `placer` (Player): Player rotating the item
- `node` (table): Pedestal node definition
- `entity` (Entity): Entity representing the item

---

### `on_construct(pos)`

**Trigger:** When pedestal node is created  
**Parameters:**
- `pos` (table): Pedestal position

---

### `after_destruct(pos, oldnode)`

**Trigger:** When pedestal node is destroyed  
**Parameters:**
- `pos` (table): Pedestal position
- `oldnode` (table): Old node definition

---

## Example Implementation

```lua
-- Register a stone pedestal
pedestals_api.register("my_mod", "stone_pedestal", {
    description = "Stone Pedestal",
    tiles = {"default_stone.png"},
    groups = {cracky = 3, oddly_breakable_by_hand = 2},
    
    on_display = function(pos, itemstack, placer, node, entity)
        minetest.chat_send_player(placer:get_player_name(), 
            "Item displayed on " .. node.name)
    end,
    
    on_rotate = function(pos, itemstack, placer, node, entity)
        minetest.chat_send_player(placer:get_player_name(), 
            "Rotated item by 45°")
    end,
    
    on_remove = function(pos, itemstack, placer, node)
        minetest.chat_send_player(placer:get_player_name(), 
            "Item retrieved: " .. itemstack:get_name())
        return itemstack -- Return original item
    end
})

-- Register a wooden pedestal with custom entity
pedestals_api.register("my_mod", "wooden_pedestal", {
    description = "Wooden Pedestal",
    tiles = {"default_wood.png"},
    groups = {choppy = 2, oddly_breakable_by_hand = 2},
    entity_name = "my_mod:wooden_pedestal_entity",
    
    on_display = function(pos, itemstack, placer, node, entity)
        -- Custom entity logic here
    end
})
```

---

## Dependencies

| Dependency | Version | Required |
|------------|---------|----------|
| `radapi`   | v3+     | Yes      |

> **Note:** The API requires `radapi` for entity management. Install it before using this API.

---
