local pedestals = {}
pedestals_api = {}

local math_rad = math.rad
local math_deg = math.deg
local vector_add = vector.add
local vector_multiply = vector.multiply

local function deg_to_rad(rot)
    if not rot then return {x=0, y=0, z=0} end
    return {x=math_rad(rot.x or 0), y=math_rad(rot.y or 0), z=math_rad(rot.z or 0)}
end

local verbose_logging = false
local function log(level, ...)
    if level == "error" or verbose_logging then
        local msg = table.concat({...}, "\t")
        core.log(level, msg)
    end
end

local function remove_entity_by_guid(guid)
    if not guid or guid == "" then return end
    local obj = core.objects_by_guid[guid]
    if obj and obj:is_valid() then
        obj:remove()
        return true
    end
    return false
end

local function get_pointed_node(player, range)
    if not player or not player:is_valid() then return nil end
    local eye_pos = player:get_pos()
    eye_pos.y = eye_pos.y + player:get_properties().eye_height
    local look_dir = player:get_look_dir()
    local end_pos = vector_add(eye_pos, vector_multiply(look_dir, range or 5))
    
    local ray = core.raycast(eye_pos, end_pos, false, false)
    for pointed in ray do
        if pointed.type == "node" then return pointed end
    end
    return nil
end

function pedestals_api.register_pedestal(name, def)
    assert(type(name) == "string", "pedestals_api.register_pedestal: name must be a string")
    assert(type(def) == "table", "pedestals_api.register_pedestal: def must be a table")

    def.description = def.description or "Pedestal"
    def.tiles = def.tiles or {"blank.png"}
    def.drawtype = "nodebox"
    def.paramtype = "light"
    def.paramtype2 = "facedir"
    
    def.visual_size = def.visual_size or 0.667
    def.glow = def.glow or 0
    def.bobbing = def.bobbing or false
    def.spin_speed = def.spin_speed or 0
    def.spin_direction = def.spin_direction or "y"
    def.rotation_offset = def.rotation_offset or {x=0,y=0,z=0}
    def.directional_offset = def.directional_offset or {x=0,y=0,z=0}
    
    local nodebox = def.nodebox or {
        type = "fixed",
        fixed = {{-0.33, -0.5, -0.33, 0.33, 0.5, 0.33}}
    }

    local groups = def.groups or {cracky = 3, oddly_breakable_by_hand = 1}

    local nodedef = {
        description      = def.description,
        tiles            = def.tiles,
        drawtype         = def.drawtype,
        mesh             = def.mesh,
        paramtype        = def.paramtype,
        paramtype2       = def.paramtype2,
        walkable         = true,
        node_box         = nodebox,
        selection_box    = nodebox,
        groups           = groups,

        on_place = function(itemstack, placer, pointed_thing)
            local pos = pointed_thing.above
            local node_above_1 = core.get_node({x=pos.x, y=pos.y+1, z=pos.z})
            local node_above_2 = core.get_node({x=pos.x, y=pos.y+2, z=pos.z})
            
            if node_above_1.name ~= "air" or node_above_2.name ~= "air" then
                if placer and placer:is_player() then
                    core.chat_send_player(placer:get_player_name(), "Not enough room!")
                end
                return itemstack
            end
            return core.item_place(itemstack, placer, pointed_thing)
        end,

        on_construct = function(pos)
            local meta = core.get_meta(pos)
            meta:set_string("infotext", def.description)
        end,

        on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
            if not (clicker and clicker:is_valid()) then return end
            local player_name = clicker:get_player_name()
            if core.is_protected(pos, player_name) then return itemstack end

            local meta = core.get_meta(pos)
            local stored_item = meta:get_string("itemstring")
            local guid = meta:get_string("entity_guid")
            local is_creative = core.is_creative_enabled(player_name)

            if stored_item == "" then
                local wielded = clicker:get_wielded_item()
                if wielded and not wielded:is_empty() then
                    local item_to_store = wielded:to_string()
                    local base_pos = vector_add(pos, {x=0, y=1.1, z=0})
                    local entity_pos = vector_add(base_pos, def.directional_offset)

                    local data = {
                        item = item_to_store,
                        visual_size = def.visual_size,
                        glow = def.glow,
                        rotation = def.rotation_offset,
                        spin_direction = def.spin_direction,
                        bobbing = def.bobbing,
                        spin_speed = def.spin_speed,
                        pedestal_pos = pos,
                        node_name = name,
                    }
                    
                    local ent = core.add_entity(entity_pos, "pedestal:display_entity", core.serialize(data))
                    if ent then
                        meta:set_string("entity_guid", ent:get_guid())
                        meta:set_string("itemstring", item_to_store)
                        if not is_creative then
                            wielded:take_item(1)
                            clicker:set_wielded_item(wielded)
                        end
                        if def._after_placed then def._after_placed(pos, item_to_store, clicker) end
                    end
                end
            else
                remove_entity_by_guid(guid)
                if not is_creative then
                    local stack = ItemStack(stored_item)
                    local inv = clicker:get_inventory()
                    if inv:room_for_item("main", stack) then inv:add_item("main", stack)
                    else core.add_item(pos, stack) end
                end
                meta:set_string("itemstring", "")
                meta:set_string("entity_guid", "")
                if def._after_takeout then def._after_takeout(pos, stored_item, clicker) end
            end
            return itemstack
        end,

        on_dig = function(pos, node, digger)
            local meta = core.get_meta(pos)
            remove_entity_by_guid(meta:get_string("entity_guid"))
            local item_str = meta:get_string("itemstring")
            if item_str ~= "" then core.add_item(pos, ItemStack(item_str)) end
            core.node_dig(pos, node, digger)
        end,
    }

    core.register_node(name, nodedef)
    pedestals[name] = def
    if def.recipe then
        core.register_craft({output = name, recipe = def.recipe})
    end
end

core.register_entity("pedestals_api:display_entity", {
    initial_properties = {
        visual = "wielditem",
        wield_item = "air",
        pointable = false,
        physical = false,
        static_save = true,
    },

    on_activate = function(self, staticdata)
        local data = core.deserialize(staticdata or "") or {}
        self.base_pos = self.object:get_pos()
        self.pedestal_pos = data.pedestal_pos or self.base_pos
        self.node_name = data.node_name or ""
        self.bobbing = data.bobbing or false
        self.spin_speed = data.spin_speed or 0
        self.spin_axis = data.spin_direction or "y"
        self._stored_data = data

        if data.item then self.object:set_properties({wield_item = data.item}) end
        if data.visual_size then
            local s = data.visual_size
            self.object:set_properties({visual_size = {x=s, y=s, z=s}})
        end
        if data.glow then self.object:set_properties({glow=data.glow}) end
        if data.rotation then self.object:set_rotation(deg_to_rad(data.rotation)) end
        self.check_timer = 0
    end,

    get_staticdata = function(self) return core.serialize(self._stored_data or {}) end,

    on_step = function(self, dtime)
        if self.bobbing then
            self.bob_time = (self.bob_time or 0) + dtime
            local new_pos = vector.new(self.base_pos)
            new_pos.y = new_pos.y + (0.05 * math.sin(2 * math.pi * self.bob_time))
            self.object:set_pos(new_pos)
        end

        if self.spin_speed ~= 0 then
            local rot = self.object:get_rotation()
            local amt = self.spin_speed * dtime
            if self.spin_axis:find("x") then rot.x = rot.x + amt end
            if self.spin_axis:find("y") then rot.y = rot.y + amt end
            if self.spin_axis:find("z") then rot.z = rot.z + amt end
            self.object:set_rotation(rot)
        end

        self.check_timer = self.check_timer + dtime
        if self.check_timer >= 5 then
            self.check_timer = 0
            local node = core.get_node(self.pedestal_pos)
            if not node or node.name ~= self.node_name then self.object:remove() end
        end
    end,
})

