pedestals_api = {}

-- ------------------------------------------------------------------
-- 1. Light API Implementation (Internal)
-- ------------------------------------------------------------------
local modname = core.get_current_modname()
-- [pos_str] = {orig = node_name, src = {id = level}}
-- Tracks which light sources are affecting each position
local positions = {}
-- [source_id] = {pos, radius, level, cells}
-- Stores information about each active light source
local sources   = {}
-- Counter for generating unique light source IDs
local source_id_counter = 1

-- Register the 14 transparent light-emitting nodes
-- These nodes are used internally to create light effects at different intensities
do
    -- Register a single light node with the given intensity level
    local function regnode(i)
        local name = modname .. ":light_" .. i
        if not core.registered_nodes[name] then
            core.register_node(name, {
                description = "Light level " .. i,
                tiles = {"air.png"},
                drawtype = "airlike",
                paramtype = "light",
                light_source = i,
                sunlight_propagates = true,
                walkable = false, pointable = false,
                diggable = false, buildable_to = true,
                groups = {not_in_creative_inventory = 1, is_mv_sandbox_node = 1}
            })
        end
    end
    for i = 1, 14 do regnode(i) end
end

-- Update the light node at a position based on the highest light level affecting it
-- If no light sources are affecting the position, restore the original node
local function refresh_position(pstr)
    local data = positions[pstr]
    if not data then return end
    local highest = 0
    for _, l in pairs(data.src) do
        if l > highest then highest = l end
    end
    
    local node
    if highest == 0 then
        -- No light sources affecting this position, restore original node
        node = {name = data.orig}
        positions[pstr] = nil
    else
        -- Set light node with the highest light level affecting this position
        node = {name = modname .. ":light_" .. highest}
    end
    core.swap_node(core.string_to_pos(pstr), node)
end

-- Add a light source in a spherical area around a center position
-- center: position table {x, y, z}
-- radius: integer, distance from center to affect
-- level: integer 0-14, light intensity
-- Returns: light source ID for later removal
function pedestals_api.add_area_light(center, radius, level)
    center = vector.new(center)
    radius = radius or 0
    level  = level  or 0
    local id = source_id_counter
    source_id_counter = source_id_counter + 1

    -- Create light source record
    local src = {pos = center, radius = radius, level = level, cells = {}}
    sources[id] = src

    -- Apply light to all positions within the radius
    for x = center.x-radius, center.x+radius do
        for y = center.y-radius, center.y+radius do
            for z = center.z-radius, center.z+radius do
                local p = {x=x, y=y, z=z}
                local s = core.pos_to_string(p)
                if not positions[s] then
                    positions[s] = {orig = core.get_node(p).name, src = {}}
                end
                positions[s].src[id] = level
                table.insert(src.cells, s)
                refresh_position(s)
            end
        end
    end
    return id
end

-- Remove a previously added light source by its ID
-- id: integer, the ID returned by add_area_light
function pedestals_api.remove_area_light(id)
    local src = sources[id]
    if not src then return end
    for _, s in ipairs(src.cells) do
        if positions[s] then
            positions[s].src[id] = nil
            refresh_position(s)
        end
    end
    sources[id] = nil
end

-- ------------------------------------------------------------------
-- 2. Pedestal API Logic
-- ------------------------------------------------------------------

-- Convert degrees to radians for each axis in a rotation table
-- rot: table with optional x, y, z keys in degrees
-- Returns: table with x, y, z in radians
local function deg_to_rad(rot)
    if not rot then return {x=0, y=0, z=0} end
    return {x=math.rad(rot.x or 0), y=math.rad(rot.y or 0), z=math.rad(rot.z or 0)}
end

-- Remove an entity by its GUID if it exists and is valid
-- guid: string, the global unique identifier of the entity
local function remove_entity_by_guid(guid)
    if not guid or guid == "" then return end
    local obj = core.objects_by_guid[guid]
    if obj and obj:is_valid() then obj:remove() end
end

-- Register a new pedestal node type
-- name: string, the node name (e.g. "mymod:pedestal")
-- def: table, pedestal definition with the following optional fields:
--   description: string, shown in inventory and on hover
--   tiles: table of texture names
--   nodebox: node box definition for the pedestal shape
--   groups: node groups table
--   directional_offset: vector, additional offset for displayed item
--   visual_size: number, size multiplier for displayed item (default: 0.667)
--   glow: number, additional lighting for displayed item (default: 0)
--   rotation_offset: table, additional rotation for displayed item
--   spin_direction: string, axis to spin item on (default: "y")
--   bobbing: boolean, whether item should bob up and down (default: false)
--   spin_speed: number, rotation speed in radians per second (default: 0)
--   light_items: table mapping item names to light configs
--     Each light config is a table with:
--       radius: integer, light radius
--       level: integer 0-14, light intensity
function pedestals_api.register_pedestal(name, def)
    -- Default light_items if not provided
    def.light_items = def.light_items or {}
    if def.recipe then
        core.register_craft({
            output = name,
            recipe = def.recipe,
            type = def.recipe_type or "shaped",
        })
    end
    core.register_node(name, {
        description = def.description or "Pedestal",
        tiles = def.tiles or {"blank.png"},
        drawtype = "nodebox",
        paramtype = "light",
        paramtype2 = "facedir",
        node_box = def.nodebox or {
            type = "fixed",
            fixed = {{-0.33, -0.5, -0.33, 0.33, 0.5, 0.33}}
        },
        groups = def.groups or {cracky = 3, oddly_breakable_by_hand = 1},
        
        -- Set infotext when pedestal is placed
        on_construct = function(pos)
            local meta = core.get_meta(pos)
            meta:set_string("infotext", def.description)
        end,
        
        -- Handle right-click interaction (placing/removing items)
        on_rightclick = function(pos, node, clicker, itemstack)
            if not (clicker and clicker:is_valid()) then return end
            local player_name = clicker:get_player_name()
            if core.is_protected(pos, player_name) then return itemstack end

            local meta = core.get_meta(pos)
            local stored_item = meta:get_string("itemstring")
            local is_creative = core.is_creative_enabled(player_name)

            -- TAKE ITEM OUT
            if stored_item ~= "" then
                -- Remove Light if exists
                local light_id = meta:get_int("light_id")
                if light_id > 0 then
                    pedestals_api.remove_area_light(light_id)
                    meta:set_int("light_id", 0)
                end

                -- Remove Entity
                remove_entity_by_guid(meta:get_string("entity_guid"))
                
                -- Give item back
                if not is_creative then
                    local stack = ItemStack(stored_item)
                    local inv = clicker:get_inventory()
                    if inv:room_for_item("main", stack) then 
                        inv:add_item("main", stack)
                    else 
                        core.add_item(pos, stack) 
                    end
                end

                meta:set_string("itemstring", "")
                meta:set_string("entity_guid", "")
                return itemstack
            end

            -- PLACE ITEM IN
            local wielded = clicker:get_wielded_item()
            if not wielded:is_empty() then
                local item_to_store = wielded:get_name() -- use name for light lookup
                local ent_pos = vector.add(pos, {x=0, y=1.1, z=0})
                ent_pos = vector.add(ent_pos, def.directional_offset or {x=0,y=0,z=0})

                -- Prepare data for the display entity
                local data = {
                    item = item_to_store,
                    visual_size = def.visual_size or 0.667,
                    glow = def.glow or 0,
                    rotation = def.rotation_offset or {x=0,y=0,z=0},
                    spin_direction = def.spin_direction or "y",
                    bobbing = def.bobbing or false,
                    spin_speed = def.spin_speed or 0,
                    pedestal_pos = pos,
                    node_name = name,
                }

                -- Create the display entity
                local ent = core.add_entity(ent_pos, "pedestals_api:display_entity", core.serialize(data))
                if ent then
                    meta:set_string("itemstring", item_to_store)
                    meta:set_string("entity_guid", ent:get_guid())

                    -- Handle Light Logic
                    local light_cfg = def.light_items[item_to_store]
                    if light_cfg then
                        local lid = pedestals_api.add_area_light(pos, light_cfg.radius, light_cfg.level)
                        meta:set_int("light_id", lid)
                    end

                    if not is_creative then
                        wielded:take_item(1)
                        clicker:set_wielded_item(wielded)
                    end
                end
            end
            return itemstack
        end,

        -- Clean up when the pedestal is dug
        on_dig = function(pos, node, digger)
            local meta = core.get_meta(pos)
            
            -- Cleanup Light
            local light_id = meta:get_int("light_id")
            if light_id > 0 then pedestals_api.remove_area_light(light_id) end
            
            -- Cleanup Entity
            remove_entity_by_guid(meta:get_string("entity_guid"))
            
            -- Drop Item
            local item_str = meta:get_string("itemstring")
            if item_str ~= "" then core.add_item(pos, ItemStack(item_str)) end
            
            core.node_dig(pos, node, digger)
        end,
    })
end

-- ------------------------------------------------------------------
-- 3. Display Entity
-- ------------------------------------------------------------------

-- Entity that displays items above pedestals with optional animations
core.register_entity("pedestals_api:display_entity", {
    initial_properties = {
        visual = "wielditem",
        wield_item = "air",
        pointable = false,
        physical = false,
        static_save = true,
    },

    -- Initialize entity from saved data or defaults
    on_activate = function(self, staticdata)
        local data = core.deserialize(staticdata or "") or {}
        self.base_pos = self.object:get_pos()
        self.pedestal_pos = data.pedestal_pos or self.base_pos
        self.node_name = data.node_name or ""
        self.bobbing = data.bobbing or false
        self.spin_speed = data.spin_speed or 0
        self.spin_axis = data.spin_direction or "y"
        self._stored_data = data
        self.timer = 0

        -- Apply visual properties from data
        if data.item then self.object:set_properties({wield_item = data.item}) end
        if data.visual_size then
            local s = data.visual_size
            self.object:set_properties({visual_size = {x=s, y=s, z=s}})
        end
        if data.glow then self.object:set_properties({glow=data.glow}) end
        if data.rotation then self.object:set_rotation(deg_to_rad(data.rotation)) end
    end,

    -- Save entity state for later reactivation
    get_staticdata = function(self) return core.serialize(self._stored_data or {}) end,

    -- Handle entity animations and validity checks
    on_step = function(self, dtime)
        self.timer = self.timer + dtime

        -- Bobbing animation
        if self.bobbing then
            local new_pos = vector.new(self.base_pos)
            new_pos.y = new_pos.y + (0.05 * math.sin(2 * math.pi * self.timer * 0.5))
            self.object:set_pos(new_pos)
        end

        -- Spinning animation
        if self.spin_speed ~= 0 then
            local rot = self.object:get_rotation()
            local amt = self.spin_speed * dtime
            if self.spin_axis:find("x") then rot.x = rot.x + amt end
            if self.spin_axis:find("y") then rot.y = rot.y + amt end
            if self.spin_axis:find("z") then rot.z = rot.z + amt end
            self.object:set_rotation(rot)
        end

        -- Validity check - remove if pedestal no longer exists
        if self.timer % 5 < dtime then
            local node = core.get_node(self.pedestal_pos)
            if node.name ~= self.node_name then self.object:remove() end
        end
    end,
})
