local ITEM_DATA = {}
local ENTITIES = {}
local RADAPI = {}
local function deepcopy(orig)
    local orig_type = type(orig)
    if orig_type == 'table' then
        local copy = {}
        for orig_key, orig_value in next, orig, nil do
            copy[orig_key] = deepcopy(orig_value)
        end
        local mt = getmetatable(orig)
        if mt then
            setmetatable(copy, deepcopy(mt))
        end
        return copy
    else
        return orig
    end
end

local function merge_properties(base, override)
    local props = deepcopy(base)
    if override then
        for k, v in pairs(override) do
            if type(v) == "table" and type(props[k]) == "table" then
                props[k] = merge_properties(props[k], v)
            else
                props[k] = deepcopy(v)
            end
        end
    end
    return props
end

function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name

    if ITEM_DATA[full_name] then
        core.log("warning", "[radapi] Duplicate registration attempt for " .. full_name)
        return false
    end

    if def.type == "tool" then
        core.register_tool(full_name, def)
    elseif def.type == "node" then
        core.register_node(full_name, def)
    elseif def.type == "craftitem" then
        core.register_craftitem(full_name, def)
    else
        return false
    end

    if def.craft then
        core.register_craft(def.craft)
    end

    ITEM_DATA[full_name] = {
        properties = deepcopy(def.properties),
        attach     = deepcopy(def.attach),
        on_attach  = def.on_attach,
        on_reload  = def.on_reload,
        wieldview  = def.wieldview,
    }

    return true
end

function RADAPI.get_extras(full_name)
    local extras = ITEM_DATA[full_name]
    return extras and deepcopy(extras) or nil
end

function RADAPI.update_extras(full_name, fields)
    local extras = ITEM_DATA[full_name]
    if not extras then return false end
    for k, v in pairs(fields) do
        if type(v) == "table" and type(extras[k]) == "table" then
            extras[k] = merge_properties(extras[k], v)
        else
            extras[k] = deepcopy(v)
        end
    end
    return true
end

function RADAPI.get_registered_item_names()
    local names = {}
    for name, _ in pairs(ITEM_DATA) do
        table.insert(names, name)
    end
    return table.copy(names)
end

function RADAPI.get_registered_items()
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        table.insert(items, { name = name, def = deepcopy(extras) })
    end
    return table.copy(items)
end

function RADAPI.get_registered_items_by_type(item_type)
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        local base = core.registered_items[name]
        if base and base.type == item_type then
            table.insert(items, { name = name, def = deepcopy(extras) })
        end
    end
    return table.copy(items)
end

function RADAPI.attach_entity(player, itemstack, opts)
    opts = opts or {}
    if not player or not itemstack or itemstack:is_empty() then return false end

    local item_name = itemstack:get_name()
    local extras = ITEM_DATA[item_name]
    if not extras then return false end

    local ent
    if extras.wieldview == "wielditem" then
        ent = core.add_entity(player:get_pos(), "radapi:wield_entity_item")
    elseif extras.wieldview == "itemframe" then
        ent = core.add_entity(player:get_pos(), "radapi:wield_entity_item")
    else
        ent = core.add_entity(player:get_pos(), "radapi:wield_entity")
    end
    if not ent then return false end

    local current = ent:get_properties()
    local props = merge_properties(current, extras.properties or {})

    if extras.wieldview == "wielditem" or extras.wieldview == "itemframe" then
        props.visual = "wielditem"
        props.wield_item = item_name
    else
        props.visual = "mesh"
        props.mesh = "blank.glb"
        props.textures = {"blank.png"}
    end

    ent:set_properties(props)

    local attach = extras.attach or {}
    ent:set_attach(player,
        attach.bone or "",
        attach.pos or {x=0,y=0,z=0},
        attach.rot or {x=0,y=0,z=0},
        attach.force_visible or false
    )

    local name = player:get_player_name()
    ENTITIES[name] = ENTITIES[name] or {}

    if opts.id then
        for i, e in ipairs(ENTITIES[name]) do
            if e.id == opts.id then
                if e.entity and e.entity:get_luaentity() then
                    e.entity:remove()
                end
                table.remove(ENTITIES[name], i)
                break
            end
        end
    end

    table.insert(ENTITIES[name], {
        entity    = ent,
        item_name = item_name,
        stack     = ItemStack(itemstack),
        id        = opts.id,
    })

    if extras.on_attach then extras.on_attach(player, ent) end
    return true
end

function RADAPI.detach_entity(player, id)
    local name = player:get_player_name()
    local list = ENTITIES[name]
    if not list then return false end

    for i, e in ipairs(list) do
        if e.id == id then
            if e.entity and e.entity:get_luaentity() then
                e.entity:remove()
            end
            table.remove(list, i)
            return true
        end
    end
    return false
end

function RADAPI.detach_all(player)
    local name = player:get_player_name()
    local list = ENTITIES[name]
    if not list then return false end

    for _, e in ipairs(list) do
        if e.entity and e.entity:get_luaentity() then
            e.entity:remove()
        end
    end
    ENTITIES[name] = nil
    return true
end

function RADAPI.get_entities(player)
    local list = ENTITIES[player:get_player_name()] or {}
    local copy = {}
    for i, e in ipairs(list) do
        copy[i] = {
            entity    = e.entity,
            item_name = e.item_name,
            stack     = ItemStack(e.stack),
            id        = e.id,
        }
    end
    return copy
end

function RADAPI.get_attached_items(player)
    local entries = RADAPI.get_entities(player)
    local list = {}
    for _, entry in ipairs(entries) do
        table.insert(list, entry.item_name)
    end
    return list
end

function RADAPI.get_attached_entries(player)
    local entries = RADAPI.get_entities(player)
    local out = {}
    for i, entry in ipairs(entries) do
        out[i] = {
            item_name = entry.item_name,
            id        = entry.id,
            stack     = ItemStack(entry.stack),
        }
    end
    return out
end

function RADAPI.reapply_attachment(player, entry)
    local extras = ITEM_DATA[entry.item_name]
    if not extras then return false end

    local attach = extras.attach or {}
    entry.entity:set_attach(player,
        attach.bone or "",
        attach.pos or {x=0,y=0,z=0},
        attach.rot or {x=0,y=0,z=0},
        attach.force_visible or false
    )
    return true
end

function RADAPI.reload_attached_items(player, item_list)
    item_list = item_list or RADAPI.get_attached_entries(player)
    if not item_list or #item_list == 0 then return false end

    for _, entry in ipairs(item_list) do
        local extras = ITEM_DATA[entry.item_name]
        local entries = ENTITIES[player:get_player_name()] or {}

        local attached
        for _, e in ipairs(entries) do
            if e.id == entry.id then
                attached = e
                break
            end
        end

        if attached then
            RADAPI.reapply_attachment(player, attached)
            if extras and extras.on_reload then
                extras.on_reload(player, attached.entity, attached)
            end
        else
            RADAPI.attach_entity(player, entry.stack, { id = entry.id })
        end
    end
    return true
end

function RADAPI.has_item(name)
    return ITEM_DATA[name] ~= nil
end

function RADAPI.debug_dump(player)
    local entries = RADAPI.get_entities(player)
    for _, e in ipairs(entries) do
        core.log("action", "[radapi] Attached: " .. e.item_name .. " (id=" .. tostring(e.id) .. ")")
    end
end

core.register_entity("radapi:wield_entity", {
    initial_properties = {
        visual = "mesh",
        mesh = "blank.glb",
        textures = {"blank.png"},
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical = false,
        collide_with_objects = false,
    },
})

core.register_entity("radapi:wield_entity_item", {
    initial_properties = {
        visual = "wielditem",
        wield_item = "",
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical = false,
        collide_with_objects = false,
    },
})

local function spawn_display_entity(pos, item_name, param2)
    local ent = core.add_entity(
        vector.add(pos, {x = 0, y = 0.5, z = 0}),
        "radapi:wield_entity_item"
    )
    if ent then
        ent:set_properties({
            visual = "wielditem",
            wield_item = item_name,
            visual_size = {x = 0.75, y = 0.75},
        })
        local dir = core.facedir_to_dir(param2)
        local yaw = core.dir_to_yaw(dir)
        ent:set_yaw(yaw)
    end
end

local function remove_display_entity(pos)
    local objs = core.get_objects_inside_radius(pos, 1)
    for _, obj in ipairs(objs) do
        local luaent = obj:get_luaentity()
        if luaent and luaent.name == "radapi:wield_entity_item" then
            obj:remove()
        end
    end
end

function RADAPI.register_item_frame(modname, name, def)
    local full_name = modname .. ":" .. name

    local node_def = {
        description = def.description or "Item Frame",
        drawtype = def.drawtype or "mesh",
        mesh = def.mesh or "blank.glb",
        tiles = def.tiles or {"blank.png"},
        paramtype2 = "facedir",
        groups = def.groups or {choppy = 2, oddly_breakable_by_hand = 2},
        type = "node",
        wieldview = "itemframe",

        on_construct = function(pos)
            local meta = core.get_meta(pos)
            meta:set_string("item", "")
        end,

        on_rightclick = function(pos, node, clicker, itemstack)
            local meta = core.get_meta(pos)
            local stored = meta:get_string("item")

            if itemstack:is_empty() then
                if stored ~= "" then
                    local inv = clicker:get_inventory()
                    if inv then
                        inv:add_item("main", stored)
                    end
                    meta:set_string("item", "")
                    remove_display_entity(pos)
                end
            else
                meta:set_string("item", itemstack:get_name())
                itemstack:take_item()
                spawn_display_entity(pos, meta:get_string("item"), node.param2)
                return itemstack
            end
        end,

        on_destruct = function(pos)
            local meta = core.get_meta(pos)
            local item = meta:get_string("item")
            if item and item ~= "" then
                core.add_item(pos, item)
            end
            remove_display_entity(pos)
        end,
    }

    RADAPI.register(modname, name, node_def)
end

radapi = RADAPI