# Register, Attach and Detach API

## Index
1. [Register Items](#register-items)  
2. [Attach & Detach](#attach-detach)  
3. [Attachment Management](#attachment-management)  
4. [Item Frames](#item-frames)  
5. [API Reference](#api-reference)  
6. [Examples](#examples)  
   - [Tool with Custom Mesh](#tool-with-custom-mesh)  
   - [Craftitem With Default Image](#craftitem-with-default-image)  
   - [Craftitem Without Wield Entity](#craftitem-without-any-wield-entity)  
   - [Wielditem Visual Example](#wielditem-visual-example)  
   - [Saving and Reloading Attachments](#saving-and-reloading-attachments)  
   - [Item Frame Example](#item-frame-example)  
7. [Summary](#summary)

---

## Register Items
```lua
radapi.register(modname, name, def)
```
Registers a tool, node, craftitem, or item frame under the name `modname:name`.

### Definition Fields
- **Standard fields**: `type`, `description`, `inventory_image`, etc. → handled by Minetest’s own registry (`minetest.registered_items`).
- **RADAPI extras**:
  - `properties`: Entity properties (mesh, textures, size)
  - `attach`: Attach position/rotation/bone
  - `on_attach`: Callback when entity is attached
  - `on_reload`: Callback when entity is reloaded after persistence
  - `wieldview`: `"mesh"`, `"wielditem"`, or `"itemframe"` visual mode
- **Crafting**:
  - `craft`: Full craft definition (shapeless, cooking, fuel, etc.)

⚠️ **Duplicate protection**: If you try to register the same item twice, RADAPI logs a warning and ignores the second attempt.

---

## Attach & Detach
- `radapi.attach_entity(player, itemstack, opts)` → attach an item’s wield entity to a player  
  - `opts.id` → optional identifier for duplicate protection and slot management
- `radapi.detach_entity(player, id)` → detach a specific item’s wield entity by identifier
- `radapi.detach_all(player)` → detach all wield entities from a player

Multiple items can be attached per player, each tracked by `id`.

---

## Attachment Management
- `radapi.get_entities(player)` → returns a **safe copy** of all attached entries `{entity, item_name, stack, id}`
- `radapi.get_attached_items(player)` → returns a list of item names currently attached
- `radapi.get_attached_entries(player)` → returns detailed entries `{item_name, id, stack}`
- `radapi.reload_attached_items(player, item_list)` → re‑attaches items from a saved list (calls `on_reload` if defined)
- `radapi.get_extras(item_name)` → returns the RADAPI‑specific metadata for a registered item
- `radapi.has_item(name)` → check if an item is registered
- `radapi.debug_dump(player)` → log all attached items for debugging

---

## Item Frames
RADAPI supports registering item frames that display items in-world.

- Use `radapi.register(...)` with `wieldview = "itemframe"` to define a frame node.  
- Item frames store their item in node metadata and spawn/remove display entities automatically.  
- Display entities use the `"wielditem"` visual mode and rotate based on the node’s facedir.  
- Crafting recipes can be included in the definition.

---

## API Reference

| Function | Description |
|----------|-------------|
| `radapi.register(modname, name, def)` | Register a tool, node, craftitem, or item frame with extras |
| `radapi.attach_entity(player, itemstack, opts)` | Attach an item’s wield entity to a player |
| `radapi.detach_entity(player, id)` | Detach a specific item’s wield entity by identifier |
| `radapi.detach_all(player)` | Detach all wield entities from a player |
| `radapi.get_entities(player)` | Get a safe copy of attached entities |
| `radapi.get_attached_items(player)` | Get a list of attached item names |
| `radapi.get_attached_entries(player)` | Get detailed attached entries (name, id, stack) |
| `radapi.reload_attached_items(player, item_list)` | Reattach items from a saved list |
| `radapi.get_extras(item_name)` | Get RADAPI‑specific metadata for an item |
| `radapi.has_item(name)` | Check if an item is registered |
| `radapi.debug_dump(player)` | Log attached items for debugging |

---

## Examples

### Tool with Custom Mesh
```lua
radapi.register("mymod", "sword", {
    type = "tool",
    description = "Forged Sword",
    inventory_image = "sword.png",
    craft = {
        output = "mymod:sword",
        recipe = {
            {"default:steel_ingot", "default:steel_ingot", "default:steel_ingot"},
            {"", "default:stick", ""},
            {"", "default:stick", ""}
        }
    },
    wieldview = "mesh",
    properties = {
        mesh = "sword.glb",
        textures = {"sword_texture.png"},
        visual_size = {x=1, y=1}
    },
    attach = {
        bone = "Arm_Right",
        pos = {x=0, y=5, z=0},
        rot = {x=0, y=90, z=0}
    },
    on_attach = function(player, ent)
        minetest.chat_send_player(player:get_player_name(), "Sword attached!")
    end,
    on_reload = function(player, ent, entry)
        minetest.chat_send_player(player:get_player_name(), "Sword reloaded!")
    end,
})
```

### Craftitem With Default Image
```lua
radapi.register("mymod", "pickaxe", {
    type = "tool",
    description = "Pickaxe",
    inventory_image = "pickaxe.png",
})
```

### Craftitem Without Any Wield Entity
```lua
radapi.register("mymod", "potion", {
    type = "craftitem",
    description = "Healing Potion",
    inventory_image = "potion.png",
    craft = {
        output = "mymod:potion",
        recipe = {
            {"default:glass_bottle", "default:apple", "default:glass_bottle"},
        }
    },
})
```

### Wielditem Visual Example
```lua
radapi.register("mymod", "apple", {
    type = "craftitem",
    description = "Shiny Apple",
    inventory_image = "apple.png",
    wieldview = "wielditem",
    attach = {
        bone = "Arm_Right",
        pos = {x=0, y=4, z=0},
        rot = {x=0, y=0, z=0},
    },
    on_attach = function(player, ent)
        minetest.chat_send_player(player:get_player_name(), "Apple attached with wielditem view!")
    end,
})
```

### Saving and Reloading Attachments
```lua
local saved = radapi.get_attached_entries(player)
radapi.detach_all(player)
radapi.reload_attached_items(player, saved)
```

### Item Frame Example
```lua
radapi.register("mymod", "wooden_frame", {
    type = "node",
    description = "Wooden Item Frame",
    mesh = "wooden_frame.obj",
    tiles = {"wood_texture.png"},
    groups = {choppy = 2, oddly_breakable_by_hand = 2},
    wieldview = "itemframe",

    craft = {
        output = "mymod:wooden_frame",
        recipe = {
            {"default:stick", "default:slab_wood", "default:stick"},
            {"default:stick", "",                  "default:stick"},
        }
    },
})
```

---

## Summary
- **Base item info** (type, description, inventory image, etc.) comes from Minetest’s built‑in registry.  
- **RADAPI stores only extras**: `properties`, `attach`, `on_attach`, `on_reload`, `wieldview`.  
- Attach visuals with `radapi.attach_entity`.  
- Detach visuals with `radapi.detach_entity` or `detach_all`.  
- Multiple items can be attached per player, tracked by `id`.  
- Use `get_attached_items` or `get_attached_entries` plus `reload_attached_items` to snapshot and restore attachments.  
- Use `get_extras(item_name)` to inspect RADAPI‑specific metadata.  
- Item frames are registered with `radapi.register` using `wieldview = "itemframe"`, with optional crafting recipes.  
- Duplicate protection prevents silent overrides; conflicts can be inspected with helper utilities.  

---
