# Register, Attach and Detach API

## Index
1. [Register Items](#register-items)
2. [Attach & Detach](#attach-detach)
3. [Attachment Management](#attachment-management)
4. [Global Events](#global-events)
5. [Item Frames](#item-frames)
6. [API Reference](#api-reference)
7. [Examples](#examples)
   - [Tool with Custom Mesh](#tool-with-custom-mesh)
   - [Craftitem With Default Image](#craftitem-with-default-image)  
   - [Craftitem Without Wield Entity](#craftitem-without-any-wield-entity)  
   - [Wielditem Visual Example](#wielditem-visual-example)  
   - [Saving and Reloading Attachments](#saving-and-reloading-attachments)  
   - [Item Frame Example](#item-frame-example)
   - [Item Frame with Custom Display Positioning](#item-frame-with-custom-display-positioning)
   - [Item Frame with Multiple Materials](#item-frame-with-multiple-materials)
7. [Summary](#summary)

---

## Register Items
```lua
radapi.register(modname, name, def)
```
Registers a tool, node, craftitem, or item frame under the name `modname:name`.

### Definition Fields
- **Standard fields**: `type`, `description`, `inventory_image`, etc. → handled by Minetest's own registry (`minetest.registered_items`).
- **RADAPI extras**:
  - `properties`: Entity properties (mesh, textures, size)
  - `attach`: Attach position/rotation/bone
  - `on_attach`: Callback when entity is attached
  - `on_reload`: Callback when entity is reloaded after persistence
  - `wieldview`: `"mesh"`, `"wielditem"`, or `"itemframe"` visual mode
- **Crafting**:
  - `craft`: Full craft definition (shapeless, cooking, fuel, etc.)

⚠️ **Duplicate protection**: If you try to register the same item twice, RADAPI logs a warning and ignores the second attempt.

---

## Attach & Detach
- `radapi.attach_entity(target, itemstack, opts)` → attach an item's wield entity to a target
  - `target` can be a `player` object or any other entity object (e.g., a mob).
  - `opts.id` → optional identifier for duplicate protection and slot management
- `radapi.detach_entity(target, id)` → detach a specific item's wield entity by identifier
- `radapi.detach_all(target)` → detach all wield entities from a target

Multiple items can be attached per target, each tracked by `id`.

---

## Attachment Management
- `radapi.get_entities(target)` → returns a **safe copy** of all attached entries `{entity, item_name, stack, id}`
- `radapi.get_attached_items(target)` → returns a list of item names currently attached
- `radapi.get_attached_entries(target)` → returns detailed entries `{item_name, id, stack}`
- `radapi.reload_attached_items(player, item_list)` → re-attaches items from a saved list for a player (calls `on_reload` if defined)
- `radapi.get_extras(item_name)` → returns the RADAPI-specific metadata for a registered item
- `radapi.update_extras(full_name, fields)` → updates RADAPI-specific metadata for a registered item
- `radapi.get_registered_item_names()` → returns all registered item names
- `radapi.get_registered_items()` → returns all registered items with their extras
- `radapi.get_registered_items_by_type(item_type)` → returns registered items filtered by type
- `radapi.has_item(name)` → check if an item is registered
- `radapi.debug_dump(player)` → log all attached items for debugging

---

## Global Events
Other mods can subscribe to RADAPI events to react to its lifecycle.

```lua
radapi.register_callback(event_name, function(...) end)
```

### Available Events
- `on_radapi_item_registered(full_name, def)`: Fires after an item is registered.
- `on_radapi_entity_attached(target, entity, entry)`: Fires after an entity is attached to a target.
- `on_radapi_entity_detached(target, entry)`: Fires just before an entity is detached from a target.

---

## Item Frames
RADAPI provides a powerful helper function to easily create item frames.

```lua
radapi.register_item_frame(modname, name, def)
```

### Features
- **Easy Registration**: Creates a node that functions as an item frame.
- **Automatic Display**: Spawns and removes a display entity to show the contained item.
- **Rotation Control**: Players can sneak and right-click with an empty hand to rotate the displayed item by 90 degrees.
- **Custom Display Positioning**: Control where items appear within the frame using `display_offset` and `display_visual_size`.
- **Multiple Materials**: Support for multi-material 3D models with multiple textures.
- **Customization**: The `def` table can override default properties like `mesh`, `tiles`, and `groups` to create unique frame styles.
- **Permission Control**: You can define an `on_place(pos, clicker, itemstack)` function in the `def` table. If it returns `false`, the item will not be placed, allowing for protection checks.

### New Item Frame Features (v1.2.0)
- **Custom Display Positioning**: Use `display_offset` and `display_visual_size` to control how items are displayed within frames.
- **Multi-Material Support**: Specify multiple textures in the `tiles` array for complex 3D models.

---

## API Reference

| Function | Description | Returns |
|----------|-------------|---------|
| `radapi.register(modname, name, def)` | Register a tool, node, or craftitem with extras | `true` on success, `false` on failure or duplicate |
| `radapi.register_item_frame(modname, name, def)` | Register a feature-rich item frame node | `true` on success, `false` on failure |
| `radapi.register_callback(event, func)` | Subscribe to a global RADAPI event | `nil` |
| `radapi.attach_entity(target, itemstack, opts)` | Attach an item's wield entity to a target (player or entity) | `true` on success, `false` on failure |
| `radapi.detach_entity(target, id)` | Detach a specific item's wield entity by identifier | `true` on success, `false` if not found or on failure |
| `radapi.detach_all(target)` | Detach all wield entities from a target | `true` on success, `false` if no entities found |
| `radapi.get_entities(target)` | Get a safe copy of attached entities from a target | Table of attached entity entries |
| `radapi.get_attached_items(target)` | Get a list of attached item names from a target | Table of item names (strings) |
| `radapi.get_attached_entries(target)` | Get detailed attached entries from a target | Table of detailed attachment entries |
| `radapi.reload_attached_items(player, item_list)` | Reattach items from a saved list to a player | `true` on success, `false` if no items to reload |
| `radapi.get_extras(item_name)` | Get RADAPI-specific metadata for an item | Table of RADAPI extras or `nil` if not found |
| `radapi.update_extras(full_name, fields)` | Update RADAPI-specific metadata for an item | `true` on success, `false` if item not found |
| `radapi.get_registered_item_names()` | Get all registered item names | Table of registered item names |
| `radapi.get_registered_items()` | Get all registered items with their extras | Table of items `{name, def}` |
| `radapi.get_registered_items_by_type(item_type)` | Get registered items filtered by type | Table of items `{name, def}` of specified type |
| `radapi.has_item(name)` | Check if an item is registered | `true` if registered, `false` otherwise |
| `radapi.debug_dump(target)` | Log attached items for debugging | `nil` |

---

## Examples

### Tool with Custom Mesh
```lua
radapi.register("mymod", "sword", {
    type = "tool",
    description = "Forged Sword",
    inventory_image = "sword.png",
    craft = {
        output = "mymod:sword",
        recipe = {
            {"default:steel_ingot", "default:steel_ingot", "default:steel_ingot"},
            {"", "default:stick", ""},
            {"", "default:stick", ""}
        }
    },
    wieldview = "mesh",
    properties = {
        mesh = "sword.glb",
        textures = {"sword_texture.png"},
        visual_size = {x=1, y=1}
    },
    attach = {
        bone = "Arm_Right",
        pos = {x=0, y=5, z=0},
        rot = {x=0, y=90, z=0}
    },
    on_attach = function(player, ent)
        minetest.chat_send_player(player:get_player_name(), "Sword attached!")
    end,
    on_reload = function(player, ent, entry)
        minetest.chat_send_player(player:get_player_name(), "Sword reloaded!")
    end,
})
```

### Craftitem With Default Image
```lua
radapi.register("mymod", "pickaxe", {
    type = "tool",
    description = "Pickaxe",
    inventory_image = "pickaxe.png",
})
```

### Craftitem Without Any Wield Entity
```lua
radapi.register("mymod", "potion", {
    type = "craftitem",
    description = "Healing Potion",
    inventory_image = "potion.png",
    craft = {
        output = "mymod:potion",
        recipe = {
            {"default:glass_bottle", "default:apple", "default:glass_bottle"},
        }
    },
})
```

### Wielditem Visual Example
```lua
radapi.register("mymod", "apple", {
    type = "craftitem",
    description = "Shiny Apple",
    inventory_image = "apple.png",
    wieldview = "wielditem",
    attach = {
        bone = "Arm_Right",
        pos = {x=0, y=4, z=0},
        rot = {x=0, y=0, z=0},
    },
    on_attach = function(player, ent)
        minetest.chat_send_player(player:get_player_name(), "Apple attached with wielditem view!")
    end,
})
```

### Saving and Reloading Attachments
```lua
local saved = radapi.get_attached_entries(player)
radapi.detach_all(player)
radapi.reload_attached_items(player, saved)
```

### Item Frame Example
```lua
radapi.register_item_frame("mymod", "wooden_frame", {
    description = "Wooden Item Frame",
    mesh = "wooden_frame.obj",
    tiles = {"wood_texture.png"},
    groups = {choppy = 2, oddly_breakable_by_hand = 2},
    craft = {
        output = "mymod:wooden_frame",
        recipe = {
            {"default:stick", "default:slab_wood", "default:stick"},
            {"default:stick", "",                  "default:stick"},
        }
    },
    -- Optional permission check
    on_place = function(pos, clicker, itemstack)
        -- Return false to prevent placement
        return minetest.is_protected(pos, clicker:get_player_name()) == false
    end
})
```

### Item Frame with Custom Display Positioning
```lua
radapi.register_item_frame("mymod", "wall_mount_frame", {
    description = "Wall Mount Frame",
    mesh = "wall_frame.obj",
    tiles = {"wall_frame_texture.png"},
    groups = {choppy = 2, oddly_breakable_by_hand = 2},
    
    -- Custom display positioning for items within the frame
    display_offset = {x = 0, y = 0.3, z = 0.1},    -- Position closer to wall
    display_visual_size = {x = 0.6, y = 0.6},      -- Smaller displayed items
    
    craft = {
        output = "mymod:wall_mount_frame",
        recipe = {
            {"", "default:stick", ""},
            {"default:stick", "", "default:stick"},
            {"", "default:stick", ""}
        }
    }
})
```

### Item Frame with Multiple Materials
```lua
radapi.register_item_frame("mymod", "ornate_frame", {
    description = "Ornate Item Frame",
    drawtype = "mesh",
    mesh = "ornate_frame.obj",                     -- 3D model with multiple materials
    tiles = {
        "frame_gold.png",                          -- First material group
        "frame_silver.png",                        -- Second material group
        "frame_glass.png"                          -- Third material group
    },
    groups = {cracky = 3, level = 2},
    
    display_offset = {x = 0, y = 0.5, z = 0},
    display_visual_size = {x = 0.8, y = 0.8},
    
    craft = {
        output = "mymod:ornate_frame",
        recipe = {
            {"default:gold_ingot", "default:glass", "default:gold_ingot"},
            {"default:gold_ingot", "",              "default:gold_ingot"},
            {"default:gold_ingot", "default:glass", "default:gold_ingot"}
        }
    }
})
```

---

## Summary
- **Base item info** (type, description, inventory image, etc.) comes from Minetest's built-in registry.  
- **RADAPI stores only extras**: `properties`, `attach`, `on_attach`, `on_reload`, `wieldview`.  
- Attach visuals with `radapi.attach_entity`.  
- Detach visuals with `radapi.detach_entity` or `detach_all`.  
- Multiple items can be attached per player, tracked by `id`.  
- Use `get_attached_items` or `get_attached_entries` plus `reload_attached_items` to snapshot and restore attachments.  
- Use `get_extras(item_name)` to inspect RADAPI-specific metadata.  
- Item frames are registered with `radapi.register` using `wieldview = "itemframe"`, with optional crafting recipes.  
- **New in v1.2.0**: Item frames support custom display positioning and multi-material models.  
- Duplicate protection prevents silent overrides; conflicts can be inspected with helper utilities.  

---