local ITEM_DATA = {}
local ENTITIES = {}
local CALLBACKS = {}
local RADAPI = {}
local function deepcopy(orig)
    local orig_type = type(orig)
    if orig_type == 'table' then
        local copy = {}
        for orig_key, orig_value in next, orig, nil do
            copy[orig_key] = deepcopy(orig_value)
        end
        local mt = getmetatable(orig)
        if mt then
            setmetatable(copy, deepcopy(mt))
        end
        return copy
    else
        return orig
    end
end

local function merge_properties(base, override)
    if not override then
        local copy = {}
        for k, v in pairs(base) do
            if type(v) == "table" then
                copy[k] = merge_properties(v, nil)  -- Recursive copy for nested tables
            else
                copy[k] = v
            end
        end
        return copy
    end

    local props = {}
    for k, v in pairs(base) do
        if type(v) == "table" then
            props[k] = merge_properties(v, nil)  -- Recursive copy for nested tables
        else
            props[k] = v
        end
    end

    for k, v in pairs(override) do
        if type(v) == "table" and type(props[k]) == "table" then
            props[k] = merge_properties(props[k], v)
        else
            props[k] = v  
        end
    end

    return props
end


local function get_target_key(target)
    if not target then return nil end
    
    if target._radapi_key then
        return target._radapi_key
    end
    
    local key
    if target.is_player and target:is_player() then
        key = "player_" .. target:get_player_name()
    elseif target.get_luaentity then
        key = "object_" .. tostring(target)
    elseif target.object and target.name then
        key = "object_" .. tostring(target.object)
    else
        return nil
    end
    
    target._radapi_key = key
    return key
end


function RADAPI.register_callback(event, func)
    CALLBACKS[event] = CALLBACKS[event] or {}
    table.insert(CALLBACKS[event], func)
end

function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name

    if ITEM_DATA[full_name] then
        core.log("warning", "[radapi] Duplicate registration attempt for " .. full_name)
        return false
    end

    if def.type == "tool" then
        core.register_tool(full_name, def)
    elseif def.type == "node" then
        core.register_node(full_name, def)
    elseif def.type == "craftitem" then
        core.register_craftitem(full_name, def)
    else
        return false
    end

    if def.craft then
        core.register_craft(def.craft)
    end

    ITEM_DATA[full_name] = {
        properties = deepcopy(def.properties),
        attach     = deepcopy(def.attach),
        on_attach  = def.on_attach,
        on_reload  = def.on_reload,
        wieldview  = def.wieldview,
    }

    if CALLBACKS.on_radapi_item_registered then
        for _, func in ipairs(CALLBACKS.on_radapi_item_registered) do
            func(full_name, def)
        end
    end

    return true
end

function RADAPI.get_extras(full_name)
    local extras = ITEM_DATA[full_name]
    return extras and deepcopy(extras) or nil
end

function RADAPI.update_extras(full_name, fields)
    local extras = ITEM_DATA[full_name]
    if not extras then return false end
    for k, v in pairs(fields) do
        if type(v) == "table" and type(extras[k]) == "table" then
            extras[k] = merge_properties(extras[k], v)
        else
            extras[k] = deepcopy(v)
        end
    end
    return true
end

function RADAPI.get_registered_item_names()
    local names = {}
    for name, _ in pairs(ITEM_DATA) do
        table.insert(names, name)
    end
    return table.copy(names)
end

function RADAPI.get_registered_items()
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        table.insert(items, { name = name, def = deepcopy(extras) })
    end
    return table.copy(items)
end

function RADAPI.get_registered_items_by_type(item_type)
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        local base = core.registered_items[name]
        if base and base.type == item_type then
            table.insert(items, { name = name, def = deepcopy(extras) })
        end
    end
    return table.copy(items)
end

function RADAPI.attach_entity(target, itemstack, opts)
    opts = opts or {}
    if not target or not itemstack or itemstack:is_empty() then return false end

    local item_name = itemstack:get_name()
    local extras = ITEM_DATA[item_name]
    if not extras then return false end

    local ent
    local pos = target.get_pos and target:get_pos() or {x=0, y=0, z=0}
    if extras.wieldview == "wielditem" then
        ent = core.add_entity(pos, "radapi:wield_entity_item")
    elseif extras.wieldview == "itemframe" then
        ent = core.add_entity(pos, "radapi:wield_entity_item")
    else
        ent = core.add_entity(pos, "radapi:wield_entity")
    end
    if not ent then return false end

    local current = ent:get_properties()
    local props = merge_properties(current, extras.properties or {})

    if extras.wieldview == "wielditem" or extras.wieldview == "itemframe" then
        props.visual = "wielditem"
        props.wield_item = item_name
    else
        props.visual = "mesh"
        props.mesh = "blank.glb"
        props.textures = {"blank.png"}
    end

    ent:set_properties(props)

    local attach = extras.attach or {}
    ent:set_attach(target,
        attach.bone or "",
        attach.pos or {x=0,y=0,z=0},
        attach.rot or {x=0,y=0,z=0},
        attach.force_visible or false
    )

    local key = get_target_key(target)
    ENTITIES[key] = ENTITIES[key] or {}

    if opts.id then
        for i, e in ipairs(ENTITIES[key]) do
            if e.id == opts.id then
                if e.entity and e.entity:get_luaentity() then
                    e.entity:remove()
                end
                table.remove(ENTITIES[key], i)
                break
            end
        end
    end
    local entry = {
        entity    = ent,
        item_name = item_name,
        stack     = ItemStack(itemstack),
        id        = opts.id,
    }
    table.insert(ENTITIES[key], entry)

    if extras.on_attach then extras.on_attach(target, ent) end

    if CALLBACKS.on_radapi_entity_attached then
        for _, func in ipairs(CALLBACKS.on_radapi_entity_attached) do
            func(target, ent, entry)
        end
    end

    return true
end

function RADAPI.detach_entity(target, id)
    local key = get_target_key(target)
    local list = ENTITIES[key]
    if not list then return false end

    for i, e in ipairs(list) do
        if e.id == id then
            if e.entity and e.entity:get_luaentity() then
                e.entity:remove()
            end
            if CALLBACKS.on_radapi_entity_detached then
                for _, func in ipairs(CALLBACKS.on_radapi_entity_detached) do
                    func(target, e)
                end
            end
            table.remove(list, i)
            return true
        end
    end
    return false
end

function RADAPI.detach_all(target)
    local key = get_target_key(target)
    local list = ENTITIES[key]
    if not list then return false end

    for _, e in ipairs(list) do
        if e.entity and e.entity:get_luaentity() then
            e.entity:remove()
        end
        if CALLBACKS.on_radapi_entity_detached then
            for _, func in ipairs(CALLBACKS.on_radapi_entity_detached) do
                func(target, e)
            end
        end
    end
    ENTITIES[key] = nil
    return true
end

function RADAPI.get_entities(player)
    local list = ENTITIES[get_target_key(player)] or {}
    local copy = {}
    for i, e in ipairs(list) do
        copy[i] = {
            entity    = e.entity,
            item_name = e.item_name,
            stack     = ItemStack(e.stack),
            id        = e.id,
        }
    end
    return copy
end

function RADAPI.get_attached_items(player)
    local entries = RADAPI.get_entities(player)
    local list = {}
    for _, entry in ipairs(entries) do
        table.insert(list, entry.item_name)
    end
    return list
end

function RADAPI.get_attached_entries(player)
    local entries = RADAPI.get_entities(player)
    local out = {}
    for i, entry in ipairs(entries) do
        out[i] = {
            item_name = entry.item_name,
            id        = entry.id,
            stack     = ItemStack(entry.stack),
        }
    end
    return out
end

function RADAPI.reapply_attachment(player, entry)
    local extras = ITEM_DATA[entry.item_name]
    if not extras then return false end

    local attach = extras.attach or {}
    entry.entity:set_attach(player,
        attach.bone or "",
        attach.pos or {x=0,y=0,z=0},
        attach.rot or {x=0,y=0,z=0},
        attach.force_visible or false
    )
    return true
end

function RADAPI.reload_attached_items(player, item_list)
    item_list = item_list or RADAPI.get_attached_entries(player)
    if not player or not item_list or #item_list == 0 then return false end

    for _, entry in ipairs(item_list) do
        local extras = ITEM_DATA[entry.item_name]
        local entries = ENTITIES[player:get_player_name()] or {}

        local attached
        for _, e in ipairs(entries) do
            if e.id == entry.id then
                attached = e
                break
            end
        end

        if attached then
            RADAPI.reapply_attachment(player, attached)
            if extras and extras.on_reload then
                extras.on_reload(player, attached.entity, attached)
            end
        else
            RADAPI.attach_entity(player, entry.stack, { id = entry.id })
        end
    end
    return true
end

function RADAPI.has_item(name)
    return ITEM_DATA[name] ~= nil
end

function RADAPI.debug_dump(player)
    local entries = RADAPI.get_entities(player)
    for _, e in ipairs(entries) do
        core.log("action", "[radapi] Attached: " .. e.item_name .. " (id=" .. tostring(e.id) .. ")")
    end
end

core.register_entity("radapi:wield_entity", {
    initial_properties = {
        visual = "mesh",
        mesh = "blank.glb",
        textures = {"blank.png"},
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical = false,
        collide_with_objects = false,
    },
})

core.register_entity("radapi:wield_entity_item", {
    initial_properties = {
        visual = "wielditem",
        wield_item = "",
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical = false,
        collide_with_objects = false,
    },
})

local function spawn_display_entity(pos, item_name, param2, display_props)
    -- Use custom offset or default
    local offset = display_props and display_props.offset or {x = 0, y = 0.5, z = 0}
    local ent_pos = vector.add(pos, offset)
    
    local ent = core.add_entity(ent_pos, "radapi:wield_entity_item")
    if ent then
        -- Use custom visual size or default
        local visual_size = display_props and display_props.visual_size or {x = 0.75, y = 0.75}
        
        ent:set_properties({
            visual = "wielditem",
            wield_item = item_name,
            visual_size = visual_size,
        })
        local dir = core.facedir_to_dir(param2)
        local yaw = core.dir_to_yaw(dir)
        ent:set_yaw(yaw)
    end
end

local function remove_display_entity(pos)
    local objs = core.get_objects_inside_radius(pos, 1)
    for _, obj in ipairs(objs) do
        local luaent = obj:get_luaentity()
        if luaent and luaent.name == "radapi:wield_entity_item" then
            obj:remove()
        end
    end
end

function RADAPI.register_item_frame(modname, name, def)
    local full_name = modname .. ":" .. name

    -- Store display properties for later use
    local display_props = {
        offset = def.display_offset or {x = 0, y = 0.5, z = 0},
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75},
    }

    local node_def = {
        description = def.description or "Item Frame",
        drawtype = def.drawtype or "mesh",
        mesh = def.mesh or "blank.glb",
        tiles = def.tiles or {"blank.png"},
        paramtype2 = "facedir",
        groups = def.groups or {choppy = 2, oddly_breakable_by_hand = 2},
        type = "node",
        wieldview = "itemframe",
        
        -- Store display properties in node definition
        _radapi_display_props = display_props,

        on_construct = function(pos)
            local meta = core.get_meta(pos)
            meta:set_string("item", "")
            meta:set_int("yaw_step", 0)
            -- Store display properties in metadata
            meta:set_string("_radapi_display_props", core.write_json(display_props))
        end,

        on_rightclick = function(pos, node, clicker, itemstack)
            local meta = core.get_meta(pos)
            local stored = meta:get_string("item")

            if itemstack:is_empty() then
                if clicker:get_player_control().sneak and stored ~= "" then
                    -- Sneak-click with empty hand to rotate
                    local yaw_step = (meta:get_int("yaw_step") + 1) % 4
                    meta:set_int("yaw_step", yaw_step)

                    local objs = core.get_objects_inside_radius(vector.add(pos, {x=0, y=0.5, z=0}), 0.5)
                    for _, obj in ipairs(objs) do
                        local luaent = obj:get_luaentity()
                        if luaent and luaent.name == "radapi:wield_entity_item" then
                            local dir = core.facedir_to_dir(node.param2)
                            local base_yaw = core.dir_to_yaw(dir)
                            obj:set_yaw(base_yaw + (yaw_step * (math.pi / 2)))
                            break
                        end
                    end
                elseif stored ~= "" then
                    -- Regular click with empty hand to take item
                    local inv = clicker:get_inventory()
                    if not inv or not inv:room_for_item("main", stored) then return end

                    inv:add_item("main", stored)
                    meta:set_string("item", "")
                    meta:set_int("yaw_step", 0)
                    remove_display_entity(pos)
                end
            else
                -- Place item in frame
                if stored ~= "" then return end -- Frame is already full
                if def.on_place and not def.on_place(pos, clicker, itemstack) then return end

                meta:set_string("item", itemstack:get_name())
                itemstack:take_item()
                
                -- Use custom display properties
                local display_props_json = meta:get_string("_radapi_display_props")
                local display_props = display_props_json ~= "" and core.parse_json(display_props_json) or nil
                
                spawn_display_entity(pos, meta:get_string("item"), node.param2, display_props)
                return itemstack
            end
        end,

        on_destruct = function(pos)
            local meta = core.get_meta(pos)
            local item = meta:get_string("item")
            if item and item ~= "" then
                core.add_item(pos, item)
            end
            remove_display_entity(pos)
        end,
    }

    -- Merge user definition over the frame defaults
    for k, v in pairs(def) do
        if node_def[k] == nil or type(v) ~= "function" then
            node_def[k] = v
        end
    end

    RADAPI.register(modname, name, node_def)
end

radapi = RADAPI
