local FUNCTIONS = dofile(core.get_modpath(core.get_current_modname()) .. "/functions.lua")

local ITEM_DATA          = {}
local ENTITIES           = {}
local CALLBACKS          = {}
local RADAPI             = {}
local ATTACHED_ENTITIES  = {}

local mod_storage = core.get_mod_storage()

function RADAPI.register_callback(event, func)
    CALLBACKS[event] = CALLBACKS[event] or {}
    table.insert(CALLBACKS[event], func)
end

function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name

    if ITEM_DATA[full_name] then
        core.log("warning", "[radapi] Duplicate registration attempt for " .. full_name)
        return false
    end

    if def.type == "tool" then
        core.register_tool(full_name, def)
    elseif def.type == "node" then
        core.register_node(full_name, def)
    elseif def.type == "craftitem" then
        core.register_craftitem(full_name, def)
    else
        return false
    end

    if def.craft then
        core.register_craft(def.craft)
    end

    ITEM_DATA[full_name] = {
        properties = def.properties,
        attach     = def.attach,
        on_attach  = def.on_attach,
        on_reload  = def.on_reload,
        wieldview  = def.wieldview,
    }

    if CALLBACKS.on_radapi_item_registered then
        for _, func in ipairs(CALLBACKS.on_radapi_item_registered) do
            func(full_name, def)
        end
    end

    return true
end

function RADAPI.get_extras(full_name)
    local extras = ITEM_DATA[full_name]
    if not extras then return nil end
    local is_flat = true
    for _, v in pairs(extras) do
        if type(v) == "table" then
            is_flat = false
            break
        end
    end

    return is_flat and table.copy(extras) or FUNCTIONS.deepcopy(extras)
end

function RADAPI.update_extras(full_name, fields)
    local extras = ITEM_DATA[full_name]
    if not extras then return false end
    for k, v in pairs(fields) do
        if type(v) == "table" and type(extras[k]) == "table" then
            extras[k] = FUNCTIONS.merge_properties(extras[k], v)
        else
            extras[k] = FUNCTIONS.deepcopy(v)
        end
    end
    return true
end

function RADAPI.get_registered_item_names()
    local names = {}
    for name, _ in pairs(ITEM_DATA) do
        table.insert(names, name)
    end
    return names
end

function RADAPI.get_registered_items()
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        table.insert(items, { name = name, def = extras })
    end
    return items
end

function RADAPI.get_registered_items_by_type(item_type)
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        local base = core.registered_items[name]
        if base and base.type == item_type then
            table.insert(items, { name = name, def = extras })
        end
    end
    return items
end

function RADAPI.attach_entity(target, itemstack, opts)
    opts = opts or {}
    if not target or not itemstack or itemstack:is_empty() then return false end

    local item_name = itemstack:get_name()
    local extras = ITEM_DATA[item_name]
    if not extras then return false end

    local ent
    local pos = target.get_pos and target:get_pos() or {x = 0, y = 0, z = 0}
    if extras.wieldview == "wielditem" then
        ent = core.add_entity(pos, "radapi:wield_entity_item")
    elseif extras.wieldview == "itemframe" then
        ent = core.add_entity(pos, "radapi:wield_entity_item")
    else
        ent = core.add_entity(pos, "radapi:wield_entity")
    end
    if not ent then return false end

    local current = ent:get_properties()
    local props = FUNCTIONS.merge_properties(current, extras.properties or {})

    if extras.wieldview == "wielditem" or extras.wieldview == "itemframe" then
        props.visual = "wielditem"
        props.wield_item = item_name
    else
        props.visual = "mesh"
        props.mesh = "blank.glb"
        props.textures = {"blank.png"}
    end

    ent:set_properties(props)

    local attach = extras.attach or {}
    ent:set_attach(target,
        attach.bone or "",
        attach.pos or {x = 0, y = 0, z = 0},
        attach.rot or {x = 0, y = 0, z = 0},
        attach.force_visible or false
    )

    local key = FUNCTIONS.get_target_key(target)
    ENTITIES[key] = ENTITIES[key] or {}

    if opts.id then
        for i, e in ipairs(ENTITIES[key]) do
            if e.id == opts.id then
                if e.entity and e.entity:get_luaentity() then
                    e.entity:remove()
                end
                table.remove(ENTITIES[key], i)
                break
            end
        end
    end
    local entry = {
        entity    = ent,
        item_name = item_name,
        stack     = ItemStack(itemstack),
        id        = opts.id,
    }
    table.insert(ENTITIES[key], entry)

    local player = target:is_player() and target or (target.get_attach and target:get_attach())
    if player and player:is_player() then
        local player_name = player:get_player_name()
        if not ATTACHED_ENTITIES[player_name] then
            ATTACHED_ENTITIES[player_name] = {}
        end
        table.insert(ATTACHED_ENTITIES[player_name], ent)
    end

    if extras.on_attach then extras.on_attach(target, ent) end

    if CALLBACKS.on_radapi_entity_attached then
        for _, func in ipairs(CALLBACKS.on_radapi_entity_attached) do
            func(target, ent, entry)
        end
    end

    return true
end

function RADAPI.detach_entity(target, id)
    local key = FUNCTIONS.get_target_key(target)
    local list = ENTITIES[key]
    if not list then return false end

    for i, e in ipairs(list) do
        if e.id == id then
            local player = target:is_player() and target or (target.get_attach and target:get_attach())
            if player and player:is_player() then
                local player_name = player:get_player_name()
                if ATTACHED_ENTITIES[player_name] then
                    for j, attached_ent in ipairs(ATTACHED_ENTITIES[player_name]) do
                        if attached_ent == e.entity then
                            table.remove(ATTACHED_ENTITIES[player_name], j)
                            break
                        end
                    end
                end
            end

            if e.entity and e.entity:get_luaentity() then
                e.entity:remove()
            end
            if CALLBACKS.on_radapi_entity_detached then
                for _, func in ipairs(CALLBACKS.on_radapi_entity_detached) do
                    func(target, e)
                end
            end
            table.remove(list, i)
            return true
        end
    end
    return false
end

function RADAPI.detach_all(target)
    local key = FUNCTIONS.get_target_key(target)
    local list = ENTITIES[key]
    if not list then return false end

    local player = target:is_player() and target or (target.get_attach and target:get_attach())
    if player and player:is_player() then
        local player_name = player:get_player_name()
        ATTACHED_ENTITIES[player_name] = nil
    end

    for _, e in ipairs(list) do
        if e.entity and e.entity:get_luaentity() then
            e.entity:remove()
        end
        if CALLBACKS.on_radapi_entity_detached then
            for _, func in ipairs(CALLBACKS.on_radapi_entity_detached) do
                func(target, e)
            end
        end
    end
    ENTITIES[key] = nil
    return true
end

function RADAPI.get_entities(player)
    local list = ENTITIES[FUNCTIONS.get_target_key(player)] or {}
    local copy = {}
    for i, e in ipairs(list) do
        copy[i] = {
            entity    = e.entity,
            item_name = e.item_name,
            stack     = ItemStack(e.stack),
            id        = e.id,
        }
    end
    return copy
end

function RADAPI.get_attached_items(player)
    local entries = RADAPI.get_entities(player)
    local list = {}
    for _, entry in ipairs(entries) do
        table.insert(list, entry.item_name)
    end
    return list
end

function RADAPI.get_attached_entries(player)
    local entries = RADAPI.get_entities(player)
    local out = {}
    for i, entry in ipairs(entries) do
        out[i] = {
            item_name = entry.item_name,
            id        = entry.id,
            stack     = ItemStack(entry.stack),
        }
    end
    return out
end

function RADAPI.reapply_attachment(player, entry)
    local extras = ITEM_DATA[entry.item_name]
    if not extras then return false end

    local attach = extras.attach or {}
    entry.entity:set_attach(player,
        attach.bone or "",
        attach.pos or {x = 0, y = 0, z = 0},
        attach.rot or {x = 0, y = 0, z = 0},
        attach.force_visible or false
    )
    return true
end

function RADAPI.reload_attached_items(player, item_list)
    if not player then return false end

    local player_name = player:get_player_name()
    local attached_list = ATTACHED_ENTITIES[player_name]

    if not item_list then
        if not attached_list or #attached_list == 0 then return true end

        for _, entity in ipairs(attached_list) do
            if entity and entity:get_luaentity() then
                local ent_obj = entity:get_luaentity()
                local player_key = FUNCTIONS.get_target_key(player)
                local player_entities = ENTITIES[player_key] or {}
                for _, entry in ipairs(player_entities) do
                    if entry.entity == entity and entry.entity:get_luaentity() then
                        local extras = ITEM_DATA[entry.item_name]
                        if extras then
                            RADAPI.reapply_attachment(player, entry)
                            if extras.on_reload then
                                extras.on_reload(player, entry.entity, entry)
                            end
                        end
                        break
                    end
                end
            else
                for i, ent in ipairs(attached_list) do
                    if ent == entity then
                        table.remove(attached_list, i)
                        break
                    end
                end
            end
        end
        return true
    end

    if item_list and #item_list > 0 then
        for _, entry in ipairs(item_list) do
            local extras = ITEM_DATA[entry.item_name]
            local entries = ENTITIES[FUNCTIONS.get_target_key(player)] or {}

            local attached
            for _, e in ipairs(entries) do
                if e.id == entry.id then
                    attached = e
                    break
                end
            end

            if attached then
                if extras then
                    RADAPI.reapply_attachment(player, attached)
                    if extras.on_reload then
                        extras.on_reload(player, attached.entity, attached)
                    end
                end
            else
                RADAPI.attach_entity(player, entry.stack, { id = entry.id })
            end
        end
        return true
    end

    return false
end

function RADAPI.has_item(name)
    return ITEM_DATA[name] ~= nil
end

function RADAPI.debug_dump(player)
    local entries = RADAPI.get_entities(player)
    for _, e in ipairs(entries) do
        core.log("action", "[radapi] Attached: " .. e.item_name .. " (id=" .. tostring(e.id) .. ")")
    end
    local player_name = player:get_player_name()
    local attached = ATTACHED_ENTITIES[player_name] or {}
    core.log("action", "[radapi] Player " .. player_name .. " has " .. #attached .. " attached entities tracked.")
end

core.register_entity("radapi:wield_entity", {
    initial_properties = {
        visual = "mesh",
        mesh   = "blank.glb",
        textures = {"blank.png"},
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical  = false,
        collide_with_objects = false,
    },
})

core.register_entity("radapi:wield_entity_item", {
    initial_properties = {
        visual     = "wielditem",
        wield_item = "",
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical  = false,
        collide_with_objects = false,
    },
})

local function get_pos_key(pos)
    return string.format("%d,%d,%d", pos.x, pos.y, pos.z)
end

local function spawn_display_entity(pos, item_name, param2, display_props)
    local offset     = display_props and display_props.offset or {x = 0, y = 0.5, z = 0}
    local ent_pos    = vector.add(pos, offset)

    local ent = core.add_entity(ent_pos, "radapi:wield_entity_item")
    if ent then
        local visual_size = display_props and display_props.visual_size or {x = 0.75, y = 0.75}

        ent:set_properties({
            visual      = "wielditem",
            wield_item  = item_name,
            visual_size = visual_size,
        })
        local dir  = core.facedir_to_dir(param2)
        local yaw  = core.dir_to_yaw(dir)
        ent:set_yaw(yaw)
    end
    return ent
end

local function remove_display_entity_by_key(pos)
    local pos_key = get_pos_key(pos)
    local entity_key = mod_storage:get_string("entity_" .. pos_key)
    if entity_key ~= "" then
        FUNCTIONS.remove_entity_reference(entity_key)
        mod_storage:set_string("entity_" .. pos_key, "")
    end
end

local function cleanup_storage(pos)
    local pos_key = get_pos_key(pos)
    mod_storage:set_string("entity_" .. pos_key, "")
    mod_storage:set_string("item_" .. pos_key, "")
    mod_storage:set_int("yaw_step_" .. pos_key, 0)
    mod_storage:set_string("display_props_" .. pos_key, "")
end

function RADAPI.register_item_frame(modname, name, def)
    local full_name = modname .. ":" .. name

    local display_props = {
        offset       = def.display_offset or {x = 0, y = 0.5, z = 0},
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75},
    }

    local node_def = {
        description = def.description or "Item Frame",
        drawtype   = def.drawtype or "mesh",
        mesh       = def.mesh or "blank.glb",
        tiles      = def.tiles or {"blank.png"},
        paramtype2 = "facedir",
        groups     = def.groups or {choppy = 2, oddly_breakable_by_hand = 2},
        type       = "node",
        wieldview  = "itemframe",
        _radapi_display_props = display_props,

        on_construct = function(pos)
            local pos_key = get_pos_key(pos)
            mod_storage:set_string("item_" .. pos_key, "")
            mod_storage:set_int("yaw_step_" .. pos_key, 0)
            mod_storage:set_string("entity_" .. pos_key, "")
            mod_storage:set_string("display_props_" .. pos_key, core.serialize(display_props))
        end,

        on_rightclick = function(pos, node, clicker, itemstack)
            local pos_key = get_pos_key(pos)
            local stored = mod_storage:get_string("item_" .. pos_key)

            if itemstack:is_empty() then
                if clicker:get_player_control().sneak and stored ~= "" then
                    local yaw_step = (mod_storage:get_int("yaw_step_" .. pos_key) + 1) % 4
                    mod_storage:set_int("yaw_step_" .. pos_key, yaw_step)

                    local entity_key = mod_storage:get_string("entity_" .. pos_key)
                    if entity_key ~= "" then
                        local entity = FUNCTIONS.get_entity_reference(entity_key)
                        if entity and entity:get_luaentity() then
                            local dir = core.facedir_to_dir(node.param2)
                            local base_yaw = core.dir_to_yaw(dir)
                            entity:set_yaw(base_yaw + (yaw_step * (math.pi / 2)))
                        end
                    end
                elseif stored ~= "" then
                    local inv = clicker:get_inventory()
                    if not inv or not inv:room_for_item("main", stored) then return end

                    inv:add_item("main", stored)
                    
                    remove_display_entity_by_key(pos)
                    
                    mod_storage:set_string("item_" .. pos_key, "")
                    mod_storage:set_int("yaw_step_" .. pos_key, 0)
                end
            else
                if stored ~= "" then return end
                if def.on_place and not def.on_place(pos, clicker, itemstack) then return end

                mod_storage:set_string("item_" .. pos_key, itemstack:get_name())
                itemstack:take_item()

                remove_display_entity_by_key(pos)

                local display_props_serialized = mod_storage:get_string("display_props_" .. pos_key)
                local display_props = display_props_serialized ~= "" and core.deserialize(display_props_serialized) or nil
                
                local ent = spawn_display_entity(pos, mod_storage:get_string("item_" .. pos_key), node.param2, display_props)
                if ent then
                    local entity_key = FUNCTIONS.gen_uuid()
                    FUNCTIONS.store_entity_reference(entity_key, ent)
                    mod_storage:set_string("entity_" .. pos_key, entity_key)
                end
                
                return itemstack
            end
        end,

        on_destruct = function(pos)
            local pos_key = get_pos_key(pos)
            local item = mod_storage:get_string("item_" .. pos_key)
            
            remove_display_entity_by_key(pos)
            
            cleanup_storage(pos)
            
            if item and item ~= "" then
                core.add_item(pos, item)
            end
        end,
    }

    for k, v in pairs(def) do
        if node_def[k] == nil or type(v) ~= "function" then
            node_def[k] = v
        end
    end

    RADAPI.register(modname, name, node_def)
end

function RADAPI.register_pedestal(modname, name, def)
    local full_name = modname .. ":" .. name

    local display_props = {
        offset = def.display_offset or {x = 0, y = 0.5, z = 0},
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75},
    }

    local node_def = {
        description = def.description or "Pedestal",
        drawtype = def.drawtype or "mesh",
        mesh = def.mesh or "blank.glb",
        tiles = def.tiles or {"blank.png"},
        paramtype2 = "facedir",
        groups = def.groups or {choppy = 2, oddly_breakable_by_hand = 2},
        type = "node",
        wieldview = "itemframe",
        _radapi_display_props = display_props,

        on_construct = function(pos)
            local pos_key = get_pos_key(pos)
            mod_storage:set_string("item_" .. pos_key, "")
            mod_storage:set_int("yaw_step_" .. pos_key, 0)
            mod_storage:set_string("infotext_" .. pos_key, "Empty Pedestal")
            mod_storage:set_string("entity_" .. pos_key, "")
            mod_storage:set_string("display_props_" .. pos_key, core.serialize(display_props))
        end,

        on_rightclick = function(pos, node, clicker, itemstack)
            local pos_key = get_pos_key(pos)
            local stored = mod_storage:get_string("item_" .. pos_key)

            if itemstack:is_empty() then
                if stored ~= "" then
                    local inv = clicker:get_inventory()
                    if inv and inv:room_for_item("main", stored) then
                        inv:add_item("main", stored)
                        
                        remove_display_entity_by_key(pos)
                        
                        mod_storage:set_string("item_" .. pos_key, "")
                        mod_storage:set_int("yaw_step_" .. pos_key, 0)
                        mod_storage:set_string("infotext_" .. pos_key, "Empty Pedestal")
                    end
                end
            else
                if stored == "" then
                    if def.on_place and not def.on_place(pos, clicker, itemstack) then 
                        return 
                    end

                    local item_name = itemstack:get_name()
                    mod_storage:set_string("item_" .. pos_key, item_name)
                    mod_storage:set_string("infotext_" .. pos_key, "Displaying: " .. item_name)
                    itemstack:take_item()

                    remove_display_entity_by_key(pos)
                    
                    local display_props_serialized = mod_storage:get_string("display_props_" .. pos_key)
                    local display_props = display_props_serialized ~= "" and 
                                          core.deserialize(display_props_serialized) or nil

                    local ent = spawn_display_entity(pos, item_name, node.param2, display_props)
                    if ent then
                        local entity_key = FUNCTIONS.gen_uuid()
                        FUNCTIONS.store_entity_reference(entity_key, ent)
                        mod_storage:set_string("entity_" .. pos_key, entity_key)
                    end
                    
                    return itemstack
                end
            end
        end,

        on_destruct = function(pos)
            local pos_key = get_pos_key(pos)
            local item = mod_storage:get_string("item_" .. pos_key)
            
            remove_display_entity_by_key(pos)
            
            cleanup_storage(pos)
            
            if item and item ~= "" then
                core.add_item(pos, item)
            end
        end,

        on_rotate = function(pos, node, user, _, axis)
            local pos_key = get_pos_key(pos)
            local stored = mod_storage:get_string("item_" .. pos_key)
            
            if stored ~= "" then
                local yaw_step = (mod_storage:get_int("yaw_step_" .. pos_key) + 1) % 4
                mod_storage:set_int("yaw_step_" .. pos_key, yaw_step)

                local entity_key = mod_storage:get_string("entity_" .. pos_key)
                if entity_key ~= "" then
                    local entity = FUNCTIONS.get_entity_reference(entity_key)
                    if entity and entity:get_luaentity() then
                        local dir = core.facedir_to_dir(node.param2)
                        local base_yaw = core.dir_to_yaw(dir)
                        entity:set_yaw(base_yaw + (yaw_step * (math.pi / 2)))
                    end
                end
            end
            return true
        end,
    }

    for k, v in pairs(def) do
        if node_def[k] == nil or type(v) ~= "function" then
            node_def[k] = v
        end
    end

    RADAPI.register(modname, name, node_def)
end

minetest.register_on_leaveplayer(function(player)
    radapi.detach_all(player)
end)

radapi = RADAPI
