local FUNCTIONS = dofile(core.get_modpath(core.get_current_modname()) .. "/functions.lua")

local ITEM_DATA          = {}
local ENTITIES           = {}
local CALLBACKS          = {}
local RADAPI             = {}
local ATTACHED_ENTITIES  = {}

local legacy_mod_storage = core.get_mod_storage()

function RADAPI.register_callback(event, func)
    CALLBACKS[event] = CALLBACKS[event] or {}
    table.insert(CALLBACKS[event], func)
end

function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name

    if ITEM_DATA[full_name] then
        core.log("warning", "[radapi] Duplicate registration attempt for " .. full_name)
        return false
    end

    if def.type == "tool" then
        core.register_tool(full_name, def)
    elseif def.type == "node" then
        core.register_node(full_name, def)
    elseif def.type == "craftitem" then
        core.register_craftitem(full_name, def)
    else
        return false
    end

    if def.craft then
        core.register_craft(def.craft)
    end

    ITEM_DATA[full_name] = {
        properties = def.properties,
        attach     = def.attach,
        on_attach  = def.on_attach,
        on_reload  = def.on_reload,
        wieldview  = def.wieldview,
    }

    if CALLBACKS.on_radapi_item_registered then
        for _, func in ipairs(CALLBACKS.on_radapi_item_registered) do
            func(full_name, def)
        end
    end

    return true
end

function RADAPI.get_extras(full_name)
    local extras = ITEM_DATA[full_name]
    if not extras then return nil end
    local is_flat = true
    for _, v in pairs(extras) do
        if type(v) == "table" then
            is_flat = false
            break
        end
    end

    return is_flat and table.copy(extras) or FUNCTIONS.deepcopy(extras)
end

function RADAPI.update_extras(full_name, fields)
    local extras = ITEM_DATA[full_name]
    if not extras then return false end
    for k, v in pairs(fields) do
        if type(v) == "table" and type(extras[k]) == "table" then
            extras[k] = FUNCTIONS.merge_properties(extras[k], v)
        else
            extras[k] = FUNCTIONS.deepcopy(v)
        end
    end
    return true
end

function RADAPI.get_registered_item_names()
    local names = {}
    for name, _ in pairs(ITEM_DATA) do
        table.insert(names, name)
    end
    return names
end

function RADAPI.get_registered_items()
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        table.insert(items, { name = name, def = extras })
    end
    return items
end

function RADAPI.get_registered_items_by_type(item_type)
    local items = {}
    for name, extras in pairs(ITEM_DATA) do
        local base = core.registered_items[name]
        if base and base.type == item_type then
            table.insert(items, { name = name, def = extras })
        end
    end
    return items
end

local function save_attachments(target)
    if not target:is_player() then return end
    local entries = RADAPI.get_attached_entries(target)
    local save_data = {}
    for _, entry in ipairs(entries) do
        table.insert(save_data, {
            item_string = entry.stack:to_string(),
            id = entry.id
        })
    end
    target:get_meta():set_string("radapi:attachments", core.serialize(save_data))
end

function RADAPI.attach_entity(target, itemstack, opts)
    opts = opts or {}
    if not target or not itemstack or itemstack:is_empty() then return false end

    local item_name = itemstack:get_name()
    local extras = ITEM_DATA[item_name]
    if not extras then return false end

    local ent
    local pos = target.get_pos and target:get_pos() or {x = 0, y = 0, z = 0}
    if extras.wieldview == "wielditem" then
        ent = core.add_entity(pos, "radapi:wield_entity_item")
    elseif extras.wieldview == "itemframe" then
        ent = core.add_entity(pos, "radapi:wield_entity_item")
    else
        ent = core.add_entity(pos, "radapi:wield_entity")
    end
    if not ent then return false end

    local current = ent:get_properties()
    local props = FUNCTIONS.merge_properties(current, extras.properties or {})

    if extras.wieldview == "wielditem" or extras.wieldview == "itemframe" then
        props.visual = "wielditem"
        props.wield_item = item_name
    else
        props.visual = "mesh"
        props.mesh = "blank.glb"
        props.textures = {"blank.png"}
    end

    ent:set_properties(props)

    local attach = extras.attach or {}
    ent:set_attach(target,
        attach.bone or "",
        attach.pos or {x = 0, y = 0, z = 0},
        attach.rot or {x = 0, y = 0, z = 0},
        attach.force_visible or false
    )

    local key = FUNCTIONS.get_target_key(target)
    ENTITIES[key] = ENTITIES[key] or {}

    if opts.id then
        for i, e in ipairs(ENTITIES[key]) do
            if e.id == opts.id then
                if e.entity and e.entity:get_luaentity() then
                    e.entity:remove()
                end
                table.remove(ENTITIES[key], i)
                break
            end
        end
    end
    local entry = {
        entity    = ent,
        item_name = item_name,
        stack     = ItemStack(itemstack),
        id        = opts.id,
    }
    table.insert(ENTITIES[key], entry)

    local player = target:is_player() and target or (target.get_attach and target:get_attach())
    if player and player:is_player() then
        local player_name = player:get_player_name()
        if not ATTACHED_ENTITIES[player_name] then
            ATTACHED_ENTITIES[player_name] = {}
        end
        table.insert(ATTACHED_ENTITIES[player_name], ent)
    end

    if extras.on_attach then extras.on_attach(target, ent) end

    if CALLBACKS.on_radapi_entity_attached then
        for _, func in ipairs(CALLBACKS.on_radapi_entity_attached) do
            func(target, ent, entry)
        end
    end

    save_attachments(target)
    return true
end

function RADAPI.detach_entity(target, id)
    local key = FUNCTIONS.get_target_key(target)
    local list = ENTITIES[key]
    if not list then return false end

    for i, e in ipairs(list) do
        if e.id == id then
            local player = target:is_player() and target or (target.get_attach and target:get_attach())
            if player and player:is_player() then
                local player_name = player:get_player_name()
                if ATTACHED_ENTITIES[player_name] then
                    for j, attached_ent in ipairs(ATTACHED_ENTITIES[player_name]) do
                        if attached_ent == e.entity then
                            table.remove(ATTACHED_ENTITIES[player_name], j)
                            break
                        end
                    end
                end
            end

            if e.entity and e.entity:get_luaentity() then
                e.entity:remove()
            end
            if CALLBACKS.on_radapi_entity_detached then
                for _, func in ipairs(CALLBACKS.on_radapi_entity_detached) do
                    func(target, e)
                end
            end
            table.remove(list, i)
            save_attachments(target)
            return true
        end
    end
    return false
end

function RADAPI.detach_all(target)
    local key = FUNCTIONS.get_target_key(target)
    local list = ENTITIES[key]
    if not list then return false end

    local player = target:is_player() and target or (target.get_attach and target:get_attach())
    if player and player:is_player() then
        local player_name = player:get_player_name()
        ATTACHED_ENTITIES[player_name] = nil
    end

    for _, e in ipairs(list) do
        if e.entity and e.entity:get_luaentity() then
            e.entity:remove()
        end
        if CALLBACKS.on_radapi_entity_detached then
            for _, func in ipairs(CALLBACKS.on_radapi_entity_detached) do
                func(target, e)
            end
        end
    end
    ENTITIES[key] = nil
    save_attachments(target)
    return true
end

function RADAPI.get_entities(player)
    local list = ENTITIES[FUNCTIONS.get_target_key(player)] or {}
    local copy = {}
    for i, e in ipairs(list) do
        copy[i] = {
            entity    = e.entity,
            item_name = e.item_name,
            stack     = ItemStack(e.stack),
            id        = e.id,
        }
    end
    return copy
end

function RADAPI.get_attached_items(player)
    local entries = RADAPI.get_entities(player)
    local list = {}
    for _, entry in ipairs(entries) do
        table.insert(list, entry.item_name)
    end
    return list
end

function RADAPI.get_attached_entries(player)
    local entries = RADAPI.get_entities(player)
    local out = {}
    for i, entry in ipairs(entries) do
        out[i] = {
            item_name = entry.item_name,
            id        = entry.id,
            stack     = ItemStack(entry.stack),
        }
    end
    return out
end

function RADAPI.reapply_attachment(player, entry)
    local extras = ITEM_DATA[entry.item_name]
    if not extras then return false end

    local attach = extras.attach or {}
    entry.entity:set_attach(player,
        attach.bone or "",
        attach.pos or {x = 0, y = 0, z = 0},
        attach.rot or {x = 0, y = 0, z = 0},
        attach.force_visible or false
    )
    return true
end

function RADAPI.reload_attached_items(player, item_list)
    if not player then return false end

    local player_name = player:get_player_name()
    local attached_list = ATTACHED_ENTITIES[player_name]

    if not item_list then
        if not attached_list or #attached_list == 0 then return true end

        for _, entity in ipairs(attached_list) do
            if entity and entity:get_luaentity() then
                local ent_obj = entity:get_luaentity()
                local player_key = FUNCTIONS.get_target_key(player)
                local player_entities = ENTITIES[player_key] or {}
                for _, entry in ipairs(player_entities) do
                    if entry.entity == entity and entry.entity:get_luaentity() then
                        local extras = ITEM_DATA[entry.item_name]
                        if extras then
                            RADAPI.reapply_attachment(player, entry)
                            if extras.on_reload then
                                extras.on_reload(player, entry.entity, entry)
                            end
                        end
                        break
                    end
                end
            else
                for i, ent in ipairs(attached_list) do
                    if ent == entity then
                        table.remove(attached_list, i)
                        break
                    end
                end
            end
        end
        return true
    end

    if item_list and #item_list > 0 then
        for _, entry in ipairs(item_list) do
            local extras = ITEM_DATA[entry.item_name]
            local entries = ENTITIES[FUNCTIONS.get_target_key(player)] or {}

            local attached
            for _, e in ipairs(entries) do
                if e.id == entry.id then
                    attached = e
                    break
                end
            end

            if attached then
                if extras then
                    RADAPI.reapply_attachment(player, attached)
                    if extras.on_reload then
                        extras.on_reload(player, attached.entity, attached)
                    end
                end
            else
                RADAPI.attach_entity(player, entry.stack, { id = entry.id })
            end
        end
        return true
    end

    return false
end

function RADAPI.has_item(name)
    return ITEM_DATA[name] ~= nil
end

function RADAPI.debug_dump(player)
    local entries = RADAPI.get_entities(player)
    for _, e in ipairs(entries) do
        core.log("action", "[radapi] Attached: " .. e.item_name .. " (id=" .. tostring(e.id) .. ")")
    end
    local player_name = player:get_player_name()
    local attached = ATTACHED_ENTITIES[player_name] or {}
    core.log("action", "[radapi] Player " .. player_name .. " has " .. #attached .. " attached entities tracked.")
end

core.register_entity("radapi:wield_entity", {
    initial_properties = {
        visual = "mesh",
        mesh   = "blank.glb",
        textures = {"blank.png"},
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical  = false,
        collide_with_objects = false,
        static_save = false, -- Critical: Prevents entities from getting stuck in the map on crash
    },
})

core.register_entity("radapi:wield_entity_item", {
    initial_properties = {
        visual     = "wielditem",
        wield_item = "",
        visual_size = {x = 1, y = 1},
        pointable = false,
        physical  = false,
        collide_with_objects = false,
        static_save = false,
    },
})

local function spawn_display_entity(pos, item_name, param2, display_props)
    local offset     = display_props and display_props.offset or {x = 0, y = 0.5, z = 0}
    local ent_pos    = vector.add(pos, offset)

    -- FIX: Scan for existing entities to prevent duplicates (Edge Case: Desync)
    for obj in core.objects_inside_radius(ent_pos, 0.5) do
        local ent = obj:get_luaentity()
        if ent and (ent.name == "radapi:wield_entity_item" or ent.name == "radapi:wield_entity") then
            obj:remove()
        end
    end

    local ent = core.add_entity(ent_pos, "radapi:wield_entity_item")
    if ent then
        local visual_size = display_props and display_props.visual_size or {x = 0.75, y = 0.75}

        ent:set_properties({
            visual      = "wielditem",
            wield_item  = item_name,
            visual_size = visual_size,
        })
        local dir  = core.facedir_to_dir(param2)
        local yaw  = core.dir_to_yaw(dir)
        ent:set_yaw(yaw)
    end
    return ent
end

local function remove_display_entities(pos)
    local node = core.get_node(pos)
    local def = core.registered_nodes[node.name]
    local props = def and def._radapi_display_props
    local offset = props and props.offset or {x = 0, y = 0.5, z = 0}
    local ent_pos = vector.add(pos, offset)

    -- Use tight radius around exact position
    for obj in core.objects_inside_radius(ent_pos, 0.5) do
        local ent = obj:get_luaentity()
        if ent and (ent.name == "radapi:wield_entity_item" or ent.name == "radapi:wield_entity") then
            obj:remove()
        end
    end
end

function RADAPI.register_item_frame(modname, name, def)
    local full_name = modname .. ":" .. name

    local groups = table.copy(def.groups or {choppy = 2, oddly_breakable_by_hand = 2})
    groups.radapi_display = 1

    local display_props = {
        offset       = def.display_offset or {x = 0, y = 0.5, z = 0},
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75},
    }

    local node_def = {
        description = def.description or "Item Frame",
        drawtype   = def.drawtype or "mesh",
        mesh       = def.mesh or "blank.glb",
        tiles      = def.tiles or {"blank.png"},
        paramtype2 = "facedir",
        groups     = groups,
        type       = "node",
        wieldview  = "itemframe",
        _radapi_display_props = display_props,

        on_construct = function(pos)
            -- Metadata is automatically clean on new placement
        end,

        on_rightclick = function(pos, node, clicker, itemstack)
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item")

            if itemstack:is_empty() then
                if clicker:get_player_control().sneak and stored ~= "" then
                    local yaw_step = (meta:get_int("radapi:yaw_step") + 1) % 4
                    meta:set_int("radapi:yaw_step", yaw_step)

                    -- Calculate exact position to avoid rotating neighbors
                    local offset = display_props.offset or {x = 0, y = 0.5, z = 0}
                    local ent_pos = vector.add(pos, offset)

                    for obj in core.objects_inside_radius(ent_pos, 0.5) do
                        local ent = obj:get_luaentity()
                        if ent and ent.name == "radapi:wield_entity_item" then
                            local dir = core.facedir_to_dir(node.param2)
                            local base_yaw = core.dir_to_yaw(dir)
                            obj:set_yaw(base_yaw + (yaw_step * (math.pi / 2)))
                        end
                    end
                elseif stored ~= "" then
                    local inv = clicker:get_inventory()
                    if not inv or not inv:room_for_item("main", stored) then return end

                    inv:add_item("main", stored)
                    
                    remove_display_entities(pos)
                    
                    meta:set_string("radapi:item", "")
                    meta:set_int("radapi:yaw_step", 0)
                end
            else
                if stored ~= "" then return end
                if def.on_place and not def.on_place(pos, clicker, itemstack) then return end

                meta:set_string("radapi:item", itemstack:get_name())
                itemstack:take_item()

                remove_display_entities(pos)

                spawn_display_entity(pos, meta:get_string("radapi:item"), node.param2, display_props)
                
                return itemstack
            end
        end,

        on_destruct = function(pos)
            local meta = core.get_meta(pos)
            local item = meta:get_string("radapi:item")
            
            remove_display_entities(pos)
            
            if item and item ~= "" then
                core.add_item(pos, item)
            end
        end,
    }

    for k, v in pairs(def) do
        if node_def[k] == nil or type(v) ~= "function" then
            node_def[k] = v
        end
    end

    RADAPI.register(modname, name, node_def)
end

function RADAPI.register_pedestal(modname, name, def)
    local full_name = modname .. ":" .. name

    local groups = table.copy(def.groups or {choppy = 2, oddly_breakable_by_hand = 2})
    groups.radapi_display = 1

    local display_props = {
        offset = def.display_offset or {x = 0, y = 0.5, z = 0},
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75},
    }

    local node_def = {
        description = def.description or "Pedestal",
        drawtype = def.drawtype or "mesh",
        mesh = def.mesh or "blank.glb",
        tiles = def.tiles or {"blank.png"},
        paramtype2 = "facedir",
        groups = groups,
        type = "node",
        wieldview = "itemframe",
        _radapi_display_props = display_props,

        on_construct = function(pos)
            local meta = core.get_meta(pos)
            meta:set_string("infotext", "Empty Pedestal")
        end,

        on_rightclick = function(pos, node, clicker, itemstack)
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item")

            if itemstack:is_empty() then
                if stored ~= "" then
                    local inv = clicker:get_inventory()
                    if inv and inv:room_for_item("main", stored) then
                        inv:add_item("main", stored)
                        
                        remove_display_entities(pos)
                        
                        meta:set_string("radapi:item", "")
                        meta:set_int("radapi:yaw_step", 0)
                        meta:set_string("infotext", "Empty Pedestal")
                    end
                end
            else
                if stored == "" then
                    if def.on_place and not def.on_place(pos, clicker, itemstack) then 
                        return 
                    end

                    local item_name = itemstack:get_name()
                    meta:set_string("radapi:item", item_name)
                    meta:set_string("infotext", "Displaying: " .. item_name)
                    itemstack:take_item()

                    remove_display_entities(pos)
                    
                    spawn_display_entity(pos, item_name, node.param2, display_props)
                    
                    return itemstack
                end
            end
        end,

        on_destruct = function(pos)
            local meta = core.get_meta(pos)
            local item = meta:get_string("radapi:item")
            
            remove_display_entities(pos)
            
            if item and item ~= "" then
                core.add_item(pos, item)
            end
        end,

        on_rotate = function(pos, node, user, _, axis)
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item")
            
            if stored ~= "" then
                local yaw_step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", yaw_step)

                -- Calculate exact position
                local offset = display_props.offset or {x = 0, y = 0.5, z = 0}
                local ent_pos = vector.add(pos, offset)

                for obj in core.objects_inside_radius(ent_pos, 0.5) do
                    local ent = obj:get_luaentity()
                    if ent and ent.name == "radapi:wield_entity_item" then
                        local dir = core.facedir_to_dir(node.param2)
                        local base_yaw = core.dir_to_yaw(dir)
                        obj:set_yaw(base_yaw + (yaw_step * (math.pi / 2)))
                    end
                end
            end
            return true
        end,
    }

    for k, v in pairs(def) do
        if node_def[k] == nil or type(v) ~= "function" then
            node_def[k] = v
        end
    end

    RADAPI.register(modname, name, node_def)
end

minetest.register_on_joinplayer(function(player)
    local meta = player:get_meta()
    local data = meta:get_string("radapi:attachments")
    if data == "" then return end
    
    local entries = core.deserialize(data)
    if not entries then return end
    
    for _, entry in ipairs(entries) do
        local stack = ItemStack(entry.item_string)
        if not stack:is_empty() then
            RADAPI.attach_entity(player, stack, {id = entry.id})
        end
    end
end)

minetest.register_on_leaveplayer(function(player)
    -- Clean up visuals only (do not wipe metadata)
    local key = FUNCTIONS.get_target_key(player)
    local list = ENTITIES[key]
    if list then
        for _, e in ipairs(list) do
            if e.entity and e.entity:get_luaentity() then e.entity:remove() end
        end
    end
    ENTITIES[key] = nil
    ATTACHED_ENTITIES[player:get_player_name()] = nil
end)

minetest.register_lbm({
    label = "Migrate RadAPI Storage",
    name = "radapi:migrate_storage",
    nodenames = {"group:radapi_display"},
    run_at_every_load = false, -- Run once per node
    action = function(pos, node)
        local meta = core.get_meta(pos)
        -- Only migrate if metadata is empty
        if meta:get_string("radapi:item") == "" then
            local pos_key = string.format("%d,%d,%d", pos.x, pos.y, pos.z)
            local old_item = legacy_mod_storage:get_string("item_" .. pos_key)
            
            if old_item and old_item ~= "" then
                core.log("action", "[radapi] Migrating data for node at " .. pos_key)
                meta:set_string("radapi:item", old_item)
                meta:set_int("radapi:yaw_step", legacy_mod_storage:get_int("yaw_step_" .. pos_key))
                meta:set_string("infotext", "Displaying: " .. old_item)
                
                -- Clean up old storage to free memory
                legacy_mod_storage:set_string("item_" .. pos_key, "")
                legacy_mod_storage:set_string("display_props_" .. pos_key, "")
                legacy_mod_storage:set_string("entity_" .. pos_key, "")
                legacy_mod_storage:set_string("infotext_" .. pos_key, "")
                legacy_mod_storage:set_int("yaw_step_" .. pos_key, 0)
            end
        end
    end,
})

minetest.register_lbm({
    label = "Restore RadAPI Display Entities",
    name = "radapi:restore_displays",
    nodenames = {"group:radapi_display"},
    run_at_every_load = true,
    action = function(pos, node)
        local meta = core.get_meta(pos)
        local item = meta:get_string("radapi:item")
        
        if item and item ~= "" then
            local def = core.registered_nodes[node.name]
            local props = def and def._radapi_display_props
            local offset = props and props.offset or {x = 0, y = 0.5, z = 0}
            local ent_pos = vector.add(pos, offset)
            
            local found = false
            for obj in core.objects_inside_radius(ent_pos, 0.2) do
                local ent = obj:get_luaentity()
                if ent and (ent.name == "radapi:wield_entity_item" or ent.name == "radapi:wield_entity") then
                    found = true
                    break
                end
            end
            
            if not found then
                spawn_display_entity(pos, item, node.param2, props)
            end
        end
    end,
})

radapi = RADAPI
