local Utils = dofile(core.get_modpath(core.get_current_modname()) .. "/functions.lua")

local ITEM_DATA         = {}
local ACTIVE_REGISTRY   = {} 
local RADAPI            = {}

-- API: Get item data with full Metadata preservation
function RADAPI.get_node_item(pos)
    local meta = core.get_meta(pos)
    local s = meta:get_string("radapi:item_full")
    if s == "" then return nil end
    -- Handle table format or legacy string format
    return s:sub(1,1) == "{" and ItemStack(core.deserialize(s)) or ItemStack(s)
end

-- API: Register 3D Item
function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name
    if ITEM_DATA[full_name] then return false end
    
    if def.type == "tool" then core.register_tool(full_name, def)
    elseif def.type == "node" then core.register_node(full_name, def)
    elseif def.type == "craftitem" then core.register_craftitem(full_name, def)
    else return false end
    
    if def.craft then core.register_craft(def.craft) end
    
    ITEM_DATA[full_name] = {
        properties = def.properties,
        attach     = def.attach,
        _on_attach = def._on_attach,
        _on_detach = def._on_detach,
        wieldview  = def.wieldview,
    }
    return true
end

-- API: Attach with Metadata persistence
function RADAPI.attach_entity(target, itemstack, opts)
    opts = opts or {}
    if not target or not itemstack or itemstack:is_empty() then return false end
    local extras = ITEM_DATA[itemstack:get_name()]
    if not extras then return false end
    
    local ref_id = Utils.get_object_id(target)
    ACTIVE_REGISTRY[ref_id] = ACTIVE_REGISTRY[ref_id] or {}
    
    if opts.id then RADAPI.detach_entity(target, opts.id) end
    
    local is_wield = (extras.wieldview == "wielditem" or extras.wieldview == "itemframe")
    local ent_type = is_wield and "radapi:wield_entity_item" or "radapi:wield_entity"
    local ent = core.add_entity(target:get_pos(), ent_type)
    
    if not ent then return false end
    
    local props = Utils.merge_properties(ent:get_properties(), extras.properties or {})
    if is_wield then props.wield_item = itemstack:get_name() end
    ent:set_properties(props)
    
    local attach = extras.attach or {}
    ent:set_attach(target, attach.bone or "", attach.pos or {x=0,y=0,z=0}, attach.rot or {x=0,y=0,z=0})
    
    table.insert(ACTIVE_REGISTRY[ref_id], {
        entity = ent,
        item_name = itemstack:get_name(),
        stack = ItemStack(itemstack), 
        id = opts.id or Utils.generate_id(),
    })
    
    if extras._on_attach then extras._on_attach(target, ent) end
    Utils.sync_storage(target, ACTIVE_REGISTRY)
    return true
end

-- API: Remove a specific attachment by ID
function RADAPI.detach_entity(target, id)
    local ref_id = Utils.get_object_id(target)
    local list = ACTIVE_REGISTRY[ref_id]
    if not list then return false end
    
    for i = #list, 1, -1 do
        local e = list[i]
        if e.id == id then
            local extras = ITEM_DATA[e.item_name]
            if extras and extras._on_detach then extras._on_detach(target, e.entity) end
            if e.entity then e.entity:remove() end
            local stack = e.stack
            table.remove(list, i)
            Utils.sync_storage(target, ACTIVE_REGISTRY)
            return stack
        end
    end
    return false
end

-- API: Detach all (Used for logout or death)
function RADAPI.detach_all(target)
    local ref_id = Utils.get_object_id(target)
    local list = ACTIVE_REGISTRY[ref_id]
    if not list then return end
    
    local dropped_stacks = {}
    for i = #list, 1, -1 do
        local e = list[i]
        local extras = ITEM_DATA[e.item_name]
        if extras and extras._on_detach then extras._on_detach(target, e.entity) end
        if e.entity then e.entity:remove() end
        table.insert(dropped_stacks, e.stack)
    end
    
    ACTIVE_REGISTRY[ref_id] = nil
    if target:is_player() then Utils.clear_storage(target:get_player_name()) end
    return dropped_stacks
end

-- API: Drops all attachments on ground
function RADAPI.drop_all_attachments(target)
    local pos = target:get_pos()
    local stacks = RADAPI.detach_all(target)
    if stacks and pos then
        for _, stack in ipairs(stacks) do core.add_item(pos, stack) end
    end
end

-- Registration Helper: 3D Item Frame
function RADAPI.register_item_frame(modname, name, def)
    local display_props = { 
        offset_value = def.display_offset_value or {x=0, y=0, z=0}, 
        visual_size = def.display_visual_size or {x = 0.4, y = 0.4} 
    }
    RADAPI.register(modname, name, {
        type = "node", 
        description = def.description, 
        drawtype = "mesh", mesh = def.mesh, tiles = def.tiles,
        paramtype = "light", paramtype2 = "wallmounted", 
        groups = table.copy(def.groups or {choppy=2, radapi_display=1}),
        _radapi_display_props = display_props,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then
                core.record_protection_violation(pos, pname)
                return itemstack
            end

            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item_full")
            
            -- Rotate Item
            if clicker:get_player_control().sneak and stored ~= "" then
                local step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", step)
                Utils.spawn_display_entity(pos, stored, node.param2, display_props, "wallmounted", step, ITEM_DATA)
                return
            end
            
            -- Take Item (Meta Preserved)
            if itemstack:is_empty() and stored ~= "" then
                local out = RADAPI.get_node_item(pos)
                if clicker:get_inventory():room_for_item("main", out) then
                    clicker:get_inventory():add_item("main", out)
                    Utils.remove_display_entities(pos)
                    meta:set_string("radapi:item_full", "")
                    meta:set_int("radapi:yaw_step", 0)
                    return out
                end
            -- Place Item (Meta Saved)
            elseif not itemstack:is_empty() and stored == "" then
                local stack = itemstack:take_item()
                local s = core.serialize(stack:to_table())
                meta:set_string("radapi:item_full", s)
                Utils.spawn_display_entity(pos, s, node.param2, display_props, "wallmounted", 0, ITEM_DATA)
                return itemstack
            end
        end,
        on_destruct = function(pos)
            local out = RADAPI.get_node_item(pos)
            Utils.remove_display_entities(pos)
            if out then core.add_item(pos, out) end
        end,
    })
end

-- Registration Helper: 3D Pedestal
function RADAPI.register_pedestal(modname, name, def)
    local display_props = { 
        offset_value = def.display_offset_value or {x=0, y=0.5, z=0}, 
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75} 
    }
    RADAPI.register(modname, name, {
        type = "node", description = def.description, 
        drawtype = "mesh", mesh = def.mesh, tiles = def.tiles,
        paramtype2 = "facedir", 
        groups = table.copy(def.groups or {choppy=2, radapi_display=1}),
        _radapi_display_props = display_props,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then
                core.record_protection_violation(pos, pname)
                return itemstack
            end

            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item_full")
            
            if clicker:get_player_control().sneak and stored ~= "" then
                local step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", step)
                Utils.spawn_display_entity(pos, stored, node.param2, display_props, "facedir", step, ITEM_DATA)
                return
            end
            
            if itemstack:is_empty() and stored ~= "" then
                local out = RADAPI.get_node_item(pos)
                if clicker:get_inventory():room_for_item("main", out) then
                    clicker:get_inventory():add_item("main", out)
                    Utils.remove_display_entities(pos)
                    meta:set_string("radapi:item_full", "")
                    return out
                end
            elseif not itemstack:is_empty() and stored == "" then
                local stack = itemstack:take_item()
                local s = core.serialize(stack:to_table())
                meta:set_string("radapi:item_full", s)
                Utils.spawn_display_entity(pos, s, node.param2, display_props, "facedir", 0, ITEM_DATA)
                return itemstack
            end
        end,
        on_destruct = function(pos)
            local out = RADAPI.get_node_item(pos)
            Utils.remove_display_entities(pos)
            if out then core.add_item(pos, out) end
        end,
    })
end

-- persistence Event Listeners
minetest.register_on_joinplayer(function(player)
    local name = player:get_player_name()
    local entries = Utils.load_storage(name)
    if entries then
        for _, e in ipairs(entries) do 
            local stack = e.item_data and ItemStack(e.item_data) or ItemStack(e.item_string)
            RADAPI.attach_entity(player, stack, {id = e.id}) 
        end
    end
end)

minetest.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    local list = ACTIVE_REGISTRY[name]
    if list then for _, e in ipairs(list) do if e.entity then e.entity:remove() end end end
    ACTIVE_REGISTRY[name] = nil
end)

minetest.register_on_dieplayer(function(player) 
    RADAPI.drop_all_attachments(player) 
end)

-- LBM: Instant restoration of world visuals
minetest.register_lbm({
    label = "Restore RadAPI Displays", 
    name = "radapi:restore_displays", 
    nodenames = {"group:radapi_display"},
    run_at_every_load = true,
    action = function(pos, node)
        local meta = core.get_meta(pos)
        local s = meta:get_string("radapi:item_full")
        if s ~= "" then
            local def = core.registered_nodes[node.name]
            local ptype2 = (def.paramtype2 == "wallmounted") and "wallmounted" or "facedir"
            Utils.spawn_display_entity(pos, s, node.param2, def._radapi_display_props, ptype2, meta:get_int("radapi:yaw_step"), ITEM_DATA)
        end
    end,
})

-- Visual Entities (Invisible base)
core.register_entity("radapi:wield_entity", { 
    initial_properties = { visual = "mesh", mesh = "blank.glb", static_save = false } 
})
core.register_entity("radapi:wield_entity_item", { 
    initial_properties = { visual = "wielditem", static_save = false } 
})

radapi = RADAPI