# RADAPI - Register, Attach and Detach API (v2.0.0)

**RADAPI** is a high-performance utility for Minetest that bridges the gap between static items and dynamic visual entities. It allows modders to attach 3D meshes or 2D sprites to players and entities, and provides customizable display nodes like item frames and pedestals.

Version **2.0.0** is a major architectural overhaul that moves from lossy string-based storage to full **ItemStack Metadata Serialization**. This ensures that custom item names, enchantments, and wear levels are perfectly preserved across world reloads and player sessions.

---

## Breaking Changes in v2.0.0

* **Metadata Key Shift**: Switched from `radapi:item` (string) to `radapi:item_full` (serialized table) for all node storage.
* **Property Naming**: The registration key `offset` has been renamed to `display_offset_value` to prevent conflicts with standard node properties.
* **Internal Callbacks**: Renamed `on_attach` and `on_detach` to `_on_attach` and `_on_detach` to avoid shadowing engine-reserved names.
* **Table Structure**: The `ITEM_DATA` and `ACTIVE_REGISTRY` tables have been refactored; direct manual manipulation by external mods is no longer supported—use the API.
* **Storage Migration**: Player attachment data is now synchronized via `ModStorage` instead of `PlayerMeta` strings.

---

## Core Features

* **Metadata Persistence**: Items with custom descriptions, wear, or metadata are fully supported.
* **Smart Rotation**: Built-in support for item rotation via sneak-right-click in frames and pedestals.
* **Protection Awareness**: Automatically respects `minetest.is_protected` for all world-placed display nodes.
* **LBM Restoration**: Uses a Loading Block Modifier to instantly restore visual entities when a MapBlock loads.
* **Automatic Cleanup**: Entities are automatically removed and items dropped upon node destruction or player death.

---

## Index

1. [Registration API](https://www.google.com/search?q=%23registration-api)
2. [Attachment & Detach API](https://www.google.com/search?q=%23attachment--detach-api)
3. [Display Nodes (Frames & Pedestals)](https://www.google.com/search?q=%23display-nodes)
4. [Utility Functions](https://www.google.com/search?q=%23utility-functions)
5. [API Reference Table](https://www.google.com/search?q=%23api-reference-table)
6. [Extensive Examples](https://www.google.com/search?q=%23extensive-examples)

---

## Registration API

### radapi.register(modname, item_name, def)

Registers an item and its associated RADAPI properties. This handles the standard Minetest registration internally.

**Definition Fields (`def`):**

* `type`: The item type (`"tool"`, `"node"`, or `"craftitem"`).
* `wieldview`: Visual mode (`"mesh"`, `"wielditem"`, or `"itemframe"`).
* `properties`: Entity properties table (mesh, textures, visual_size).
* `attach`: Positioning table: `{bone, pos, rot}`.
* `_on_attach`: `function(target, ent)` — Triggered when the entity is spawned.
* `_on_detach`: `function(target, ent)` — Triggered when the entity is removed.
* `craft`: Optional standard crafting recipe table.

---

## Attachment & Detach API

### radapi.attach_entity(target, itemstack, opts)

Attaches the visual representation of an item to an `ObjectRef`.

* **Metadata Support**: Captures the exact state of the `itemstack`.
* **Replacement**: If `opts.id` is provided and already exists on the target, the old attachment is automatically detached first.

### radapi.detach_entity(target, id)

Removes a specific attachment from the target by its unique ID.

* **Returns**: The `ItemStack` object that was attached, allowing you to return the item to an inventory.

### radapi.detach_all(target)

Removes every RADAPI attachment from the target.

* **Returns**: A table of all `ItemStack` objects removed.

---

## Display Nodes

### radapi.register_item_frame(modname, name, def)

Registers a node that uses `wallmounted` orientation to display items.

* **Logic**: Handles 4-step rotation and metadata-safe placement/retrieval.

### radapi.register_pedestal(modname, name, def)

Registers a node that uses `facedir` orientation to display items above the node.

* **Logic**: Optimized for floor-standing displays with customizable vertical offsets.

---

## API Reference Table

| Function | Parameters | Return |
| --- | --- | --- |
| `register` | `mod, name, def` | `boolean` (Success) |
| `attach_entity` | `target, stack, opts` | `boolean` (Success) |
| `detach_entity` | `target, id` | `ItemStack` or `false` |
| `detach_all` | `target` | `Table` (List of stacks) |
| `drop_all_attachments` | `target` | `nil` |
| `get_node_item` | `pos` | `ItemStack` or `nil` |
| `register_item_frame` | `mod, name, def` | `nil` |
| `register_pedestal` | `mod, name, def` | `nil` |

---

## Extensive Examples

### 1. Registering a Tool with a 3D Back-Attachment

```lua
radapi.register("my_mod", "battle_axe", {
    type = "tool",
    description = "Battle Axe",
    inventory_image = "axe_inv.png",
    wieldview = "mesh",
    properties = {
        mesh = "axe_model.obj",
        textures = {"axe_uv.png"},
        visual_size = {x = 1, y = 1},
    },
    attach = {
        bone = "Back",
        pos = {x = 0, y = 3, z = 1.5},
        rot = {x = 0, y = 0, z = 180},
    },
    _on_attach = function(player, ent)
        minetest.chat_send_player(player:get_player_name(), "Axe equipped to back!")
    end
})

```

### 2. Manual Attachment with Preservation

```lua
local function give_unique_reward(player)
    local stack = ItemStack("my_mod:battle_axe")
    local meta = stack:get_meta()
    
    meta:set_string("description", "Slayer of Dragons\nOwner: " .. player:get_player_name())
    meta:set_int("dragon_kills", 5)
    
    -- v2.0.0 captures all the metadata set above
    radapi.attach_entity(player, stack, {id = "back_slot"})
end

```

### 3. Creating a Custom Marble Pedestal

```lua
radapi.register_pedestal("decor", "marble_pedestal", {
    description = "Marble Pedestal",
    mesh = "pedestal.obj",
    tiles = {"marble.png"},
    groups = {cracky = 1},
    display_offset_value = {x = 0, y = 0.9, z = 0}, -- Item floats higher
    display_visual_size = {x = 0.6, y = 0.6},
})

```

### 4. Customizing Attachment Removal

```lua
local function strip_visuals(player)
    -- Detach specific id and drop on floor
    local stack = radapi.detach_entity(player, "back_slot")
    if stack then
        minetest.add_item(player:get_pos(), stack)
    end
end

```

### 5. Interaction with World Items

```lua
-- Retrieve an item from a pedestal via code
local pos = {x = 100, y = 10, z = 100}
local stack = radapi.get_node_item(pos)

if stack and stack:get_name() == "default:diamond" then
    print("Found a diamond on the pedestal!")
end

```

---

## Technical Persistence Details

* **On Join**: The mod reads serialized `ModStorage` data. If legacy strings are found, it converts them to `ItemStack` objects automatically.
* **LBM (Loading Block Modifier)**: Every time a map block loads, the LBM checks for nodes in the `radapi_display` group. If `radapi:item_full` contains data, it spawns a fresh display entity and applies the stored `yaw_step` rotation.
* **On Death**: RADAPI calls `drop_all_attachments`, which clears the target's `ACTIVE_REGISTRY` entry and spawns all attached item stacks as physical dropped items at the death location.