# RADAPI - Register, Attach and Detach API (v2.1.0)

**RADAPI** is a high-performance utility for Minetest that bridges the gap between static items and dynamic visual entities. It allows modders to attach 3D meshes or 2D sprites to players and entities, provides customizable display nodes like item frames and pedestals, and features a **built-in dynamic lighting engine**.

---

## Index

1. [Registration API](https://www.google.com/search?q=%23registration-api)
2. [Attachment & Detach API](https://www.google.com/search?q=%23attachment--detach-api)
3. [World Display API](https://www.google.com/search?q=%23world-display-api)
4. [Utility & Data API](https://www.google.com/search?q=%23utility--data-api)
5. [API Reference Table](https://www.google.com/search?q=%23api-reference-table)

---

## 1. Registration API

### `radapi.register(modname, item_name, def)`

Registers an item and its RADAPI properties. This handles the standard Minetest registration internally.

**Example: A 3D Glowing Torch**

```lua
radapi.register("my_mod", "magic_torch", {
    type = "craftitem",
    description = "Magic Torch",
    inventory_image = "torch_inv.png",
    wield_glow = 14, -- Emits maximum light when held or attached
    wieldview = "mesh",
    properties = {
        mesh = "torch.obj",
        textures = {"torch_uv.png"},
        visual_size = {x = 1, y = 1},
    },
    attach = {
        bone = "Arm_Right",
        pos = {x = 0, y = 5, z = 0},
        rot = {x = 0, y = 0, z = 0},
    }
})

```

---

## 2. Attachment & Detach API

### `radapi.attach_entity(target, itemstack, opts)`

Attaches an item's visual entity to a player or entity.

**Example: Equipping a Backpack**

```lua
local stack = ItemStack("my_mod:backpack")
-- 'id' allows you to track and remove this specific attachment later
radapi.attach_entity(player, stack, {id = "back_slot"})

```

### `radapi.detach_entity(target, id)`

Removes a specific attachment by ID and returns the `ItemStack` (with all metadata).

**Example: Taking off a Backpack**

```lua
local stack = radapi.detach_entity(player, "back_slot")
if stack then
    player:get_inventory():add_item("main", stack)
end

```

### `radapi.detach_all(target)`

Removes every RADAPI attachment. Returns a table of ItemStacks.

**Example: Strip-searching a player**

```lua
local stacks = radapi.detach_all(player)
for _, stack in ipairs(stacks) do
    minetest.add_item(player:get_pos(), stack)
end

```

### `radapi.drop_all_attachments(target)`

A shortcut to detach everything and spawn the items on the ground (used automatically on death).

**Example: Manual Drop**

```lua
radapi.drop_all_attachments(player)

```

---

## 3. World Display API

### `radapi.register_item_frame(modname, name, def)`

Registers a wall-mounted display node.

**Example: Oak Item Frame**

```lua
radapi.register_item_frame("my_mod", "oak_frame", {
    description = "Oak Item Frame",
    mesh = "frame.obj",
    tiles = {"oak_wood.png"},
    display_offset_value = {x=0, y=0, z=0.05},
    display_visual_size = {x=0.4, y=0.4},
})

```

### `radapi.register_pedestal(modname, name, def)`

Registers a floor-based display node.

**Example: Stone Pedestal**

```lua
radapi.register_pedestal("my_mod", "stone_pedestal", {
    description = "Stone Pedestal",
    mesh = "pedestal.obj",
    tiles = {"stone.png"},
    display_offset_value = {x=0, y=0.8, z=0},
    display_visual_size = {x=0.6, y=0.6},
})

```

---

## 4. Utility & Data API

### `radapi.get_node_item(pos)`

Retrieves the `ItemStack` currently sitting inside a Frame or Pedestal at the given position.

**Example: Checking a display**

```lua
local stack = radapi.get_node_item(pos)
if stack and not stack:is_empty() then
    print("This pedestal contains: " .. stack:get_description())
end

```

---

## 5. API Reference Table

| Function | Parameters | Return | Description |
| --- | --- | --- | --- |
| `register` | `mod, name, def` | `bool` | Registers item and RADAPI data. |
| `attach_entity` | `target, stack, opts` | `bool` | Spawns visual entity + light. |
| `detach_entity` | `target, id` | `ItemStack/false` | Removes specific attachment. |
| `detach_all` | `target` | `table` | Removes and returns all stacks. |
| `drop_all_attachments` | `target` | `nil` | Drops all attachments on ground. |
| `get_node_item` | `pos` | `ItemStack/nil` | Gets item stored in a display node. |
| `register_item_frame` | `mod, name, def` | `nil` | Registers a 3D wall frame. |
| `register_pedestal` | `mod, name, def` | `nil` | Registers a 3D floor pedestal. |

---

## Integrated Lighting Technicals

The mod handles a global light-map (`active_lights`) and a loop that tracks player positions.

1. **Air Swapping**: It swaps `air` for `radapi:light_N` at the player's rounded coordinates.
2. **Metadata Safe**: Light levels are updated without deleting item metadata.
3. **Multi-Source**: If a player has a glowing hand item AND a glowing backpack, the system automatically picks the highest light level for that position.
