# RADAPI - Register, Attach and Detach API (v2.2.0)

**RADAPI** is a high-performance utility for Minetest that bridges the gap between static items and dynamic visual entities. It allows modders to attach 3D meshes or 2D sprites to players and entities, provides customizable display nodes like item frames and pedestals, and features a **built-in dynamic lighting engine**.

---

## 1. Registration API

### `radapi.register(modname, item_name, def)`

Registers an item and its RADAPI properties. Handles handheld light and visual data.

**Example A: A Basic Glowing Tool**

```lua
radapi.register("emerald_mod", "glow_sword", {
    type = "tool",
    description = "Emerald Greatsword",
    inventory_image = "emerald_sword.png",
    wield_glow = 12, -- Glows moderately when held
    tool_capabilities = {
        full_punch_interval = 1.0,
        max_drop_level=1,
        groupcaps={ fleshy={times={2.0, 1.4, 1.0}, uses=20, maxlevel=3} }
    }
})

```

**Example B: 3D Item (Wieldview Style)**

```lua
radapi.register("cosmetics", "top_hat", {
    type = "craftitem",
    description = "Fancy Top Hat",
    inventory_image = "hat_icon.png",
    wieldview = "mesh",
    properties = {
        mesh = "top_hat.obj",
        textures = {"wool_black.png"},
        visual_size = {x=1.2, y=1.2, z=1.2},
    },
    attach = {
        bone = "Head",
        pos = {x=0, y=6.5, z=0},
        rot = {x=0, y=0, z=0},
    }
})

```

---

## 2. Torch Engine API

### `radapi.register_torch(modname, basename, def)`

Creates a 12-node torch system with burnout stages and wall/ceiling/floor logic.

**Example A: HD Mesh Torch**

```lua
radapi.register_torch("survival", "torch", {
    description = "Dungeon Torch",
    drawtype = "mesh",
    mesh = "torch_floor.obj",
    mesh_wall = "torch_wall.obj",
    mesh_ceiling = "torch_ceiling.obj",
    tiles = {"torch_texture.png"},
    selection_box = { type = "fixed", fixed = {-0.1, -0.5, -0.1, 0.1, 0.4, 0.1} }
})

```

**Example B: Classic Nodebox Torch**

```lua
radapi.register_torch("retro", "wood_torch", {
    description = "Old School Torch",
    drawtype = "nodebox",
    nodebox = { type = "fixed", fixed = {-0.05, -0.5, -0.05, 0.05, 0.2, 0.05} },
    nodebox_wall = { type = "fixed", fixed = {-0.05, -0.2, 0.3, 0.05, 0.5, 0.5} },
    tiles = {"wood_grain.png"}
})

```

---

## 3. Attachment & Detach API

### `radapi.attach_entity(target, itemstack, opts)`

Attaches an item's visual entity to a player/entity.

**Example A: Attaching a Quiver to a Player's back**

```lua
local stack = ItemStack("archery:quiver")
radapi.attach_entity(player, stack, { id = "back_accessory" })

```

**Example B: Attaching a lantern to a NPC (using bone attachment)**

```lua
-- Assuming 'self.object' is an NPC entity
local lantern = ItemStack("lighting:lantern")
radapi.attach_entity(self.object, lantern, { id = "hand_lamp" })

```

### `radapi.detach_entity(target, id)`

Removes a specific attachment and returns the item.

**Example: Swapping a backpack**

```lua
local old_bag = radapi.detach_entity(player, "back_slot")
if old_bag then
    player:get_inventory():add_item("main", old_bag)
end

```

---

## 4. World Display API

### `radapi.register_item_frame(modname, name, def)`

**Example: Golden Armor Frame**

```lua
radapi.register_item_frame("museum", "gold_frame", {
    description = "Golden Display Frame",
    mesh = "gold_frame.obj",
    tiles = {"gold_block.png"},
    display_offset_value = {x=0, y=0, z=0.1},
    display_visual_size = {x=0.5, y=0.5}
})

```

### `radapi.register_pedestal(modname, name, def)`

**Example: Marble Pedestal**

```lua
radapi.register_pedestal("museum", "marble_pedestal", {
    description = "Marble Pedestal",
    mesh = "pedestal_classic.obj",
    tiles = {"marble.png"},
    display_offset_value = {x=0, y=0.9, z=0},
    display_visual_size = {x=0.8, y=0.8}
})

```

---

## 5. Utility & Data API

### `radapi.get_node_item(pos)`

**Example: Checking if a pedestal has a specific key**

```lua
local item = radapi.get_node_item(pos)
if item and item:get_name() == "quest:boss_key" then
    minetest.chat_send_all("The boss door has opened!")
end

```

### `radapi.detach_all(target)`

**Example: Teleporting a player (Clear visuals before move)**

```lua
local visuals = radapi.detach_all(player)
-- Visuals is a table of ItemStacks. You can re-attach them after teleport.

```

---

## Technical Details

### The Lighting Logic

The engine utilizes a `globalstep` to track the "wielded" item of every player. If the item has a `wield_glow` value in its `ITEM_DATA`, the API swaps the `air` block at the player's position for a non-collidable light node (`radapi:light_1` through `radapi:light_14`).

### Storage & Persistence

RADAPI uses `player_meta` to store attachment IDs. When a player rejoins, the `register_on_joinplayer` hook automatically scans the metadata and re-attaches all 3D visuals, ensuring your backpacks and hats stay equipped between sessions.
