local Utils = dofile(core.get_modpath(core.get_current_modname()) .. "/functions.lua")

local ITEM_DATA         = {}
local ACTIVE_REGISTRY   = {} 
local RADAPI            = {}

-------------------------------------------------------------------------------
-- INTEGRATED LIGHTING ENGINE
-------------------------------------------------------------------------------
local active_lights = {}    
local update_interval = 0.2
local mod_name = core.get_current_modname()
local glow_logic_initialized = false 

-- Register the 14 light nodes (always registered to prevent unknown nodes)
for i = 1, 14 do
    core.register_node(":" .. mod_name .. ":light_" .. i, {
        drawtype = "airlike",
        paramtype = "light",
        light_source = i,
        sunlight_propagates = true,
        walkable = false,
        pointable = false,
        diggable = false,
        buildable_to = true,
        drop = "",
        groups = {not_in_creative_inventory = 1, radapi_light = 1},
    })
end

local function update_light_at_pos(pos_str)
    local lights = active_lights[pos_str]
    local pos = core.string_to_pos(pos_str)
    if not pos then return end

    local max_level = 0
    local has_any = false
    
    if lights then
        for _, level in pairs(lights) do
            has_any = true
            if level > max_level then max_level = level end
        end
    end

    local current_node = core.get_node(pos).name
    if not has_any or max_level == 0 then
        if core.get_item_group(current_node, "radapi_light") > 0 then
            core.swap_node(pos, {name = "air"})
        end
        active_lights[pos_str] = nil
        return
    end

    local target_node = mod_name .. ":light_" .. math.floor(max_level)
    if current_node == "air" or core.get_item_group(current_node, "radapi_light") > 0 then
        if current_node ~= target_node then core.swap_node(pos, {name = target_node}) end
    end
end

local function init_glow_logic()
    if glow_logic_initialized then return end
    glow_logic_initialized = true

    local timer = 0
    core.register_globalstep(function(dtime)
        timer = timer + dtime
        if timer < update_interval then return end
        timer = 0

        for _, player in ipairs(core.get_connected_players()) do
            local pname = player:get_player_name()
            local pos = vector.round(player:get_pos())
            pos.y = pos.y + 1 
            local pos_str = core.pos_to_string(pos)
            
            local wielded = player:get_wielded_item():get_name()
            local info = ITEM_DATA[wielded]
            local hand_id = pname .. "_hand"
            local pmeta = player:get_meta()
            local last_hand_pos = pmeta:get_string("radapi:last_light_pos")

            if info and info.wield_glow then
                local level = type(info.wield_glow) == "number" and info.wield_glow or 10
                if last_hand_pos ~= "" and last_hand_pos ~= pos_str then
                    if active_lights[last_hand_pos] then
                        active_lights[last_hand_pos][hand_id] = nil
                        update_light_at_pos(last_hand_pos)
                    end
                end
                active_lights[pos_str] = active_lights[pos_str] or {}
                active_lights[pos_str][hand_id] = level
                update_light_at_pos(pos_str)
                pmeta:set_string("radapi:last_light_pos", pos_str)
            else
                if last_hand_pos ~= "" then
                    if active_lights[last_hand_pos] then
                        active_lights[last_hand_pos][hand_id] = nil
                        update_light_at_pos(last_hand_pos)
                    end
                    pmeta:set_string("radapi:last_light_pos", "")
                end
            end

            if ACTIVE_REGISTRY[pname] then
                for _, e in ipairs(ACTIVE_REGISTRY[pname]) do
                    if e.glow then
                        local level = type(e.glow) == "number" and e.glow or 10
                        if e.last_pos and e.last_pos ~= pos_str then
                            if active_lights[e.last_pos] then
                                active_lights[e.last_pos][e.id] = nil
                                update_light_at_pos(e.last_pos)
                            end
                        end
                        active_lights[pos_str] = active_lights[pos_str] or {}
                        active_lights[pos_str][e.id] = level
                        e.last_pos = pos_str
                        update_light_at_pos(pos_str)
                    end
                end
            end
        end
    end)
end

-------------------------------------------------------------------------------
-- RADAPI CORE FUNCTIONS
-------------------------------------------------------------------------------

function RADAPI.get_node_item(pos)
    local meta = core.get_meta(pos)
    local s = meta:get_string("radapi:item_full")
    if s == "" then return nil end
    return s:sub(1,1) == "{" and ItemStack(core.deserialize(s)) or ItemStack(s)
end

function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name
    if ITEM_DATA[full_name] then return false end
    
    if def.type == "tool" then core.register_tool(full_name, def)
    elseif def.type == "node" then core.register_node(full_name, def)
    elseif def.type == "craftitem" then core.register_craftitem(full_name, def)
    else core.register_node(full_name, def) end -- Default to node for specific engine needs
    
    if def.craft then core.register_craft(def.craft) end
    if def.wield_glow then init_glow_logic() end
    
    ITEM_DATA[full_name] = {
        properties = def.properties,
        attach     = def.attach,
        _on_attach = def._on_attach,
        _on_detach = def._on_detach,
        wieldview  = def.wieldview,
        wield_glow = def.wield_glow,
        _radapi_display_props = def._radapi_display_props
    }
    return true
end

function RADAPI.attach_entity(target, itemstack, opts)
    opts = opts or {}
    if not target or not itemstack or itemstack:is_empty() then return false end
    local item_name = itemstack:get_name()
    local extras = ITEM_DATA[item_name]
    if not extras then return false end
    
    local ref_id = Utils.get_object_id(target)
    ACTIVE_REGISTRY[ref_id] = ACTIVE_REGISTRY[ref_id] or {}
    if opts.id then RADAPI.detach_entity(target, opts.id) end
    
    local is_wield = (extras.wieldview == "wielditem" or extras.wieldview == "itemframe")
    local ent_type = is_wield and "radapi:wield_entity_item" or "radapi:wield_entity"
    local ent = core.add_entity(target:get_pos(), ent_type)
    if not ent then return false end
    
    local props = Utils.merge_properties(ent:get_properties(), extras.properties or {})
    if is_wield then props.wield_item = item_name end
    ent:set_properties(props)
    
    local attach = extras.attach or {}
    ent:set_attach(target, attach.bone or "", attach.pos or {x=0,y=0,z=0}, attach.rot or {x=0,y=0,z=0})
    
    table.insert(ACTIVE_REGISTRY[ref_id], {
        entity = ent,
        item_name = item_name,
        stack = ItemStack(itemstack), 
        id = opts.id or Utils.generate_id(),
        glow = extras.wield_glow,
        last_pos = nil
    })
    
    if extras._on_attach then extras._on_attach(target, ent) end
    Utils.sync_storage(target, ACTIVE_REGISTRY)
    return true
end

function RADAPI.detach_entity(target, id)
    local ref_id = Utils.get_object_id(target)
    local list = ACTIVE_REGISTRY[ref_id]
    if not list then return false end
    for i = #list, 1, -1 do
        local e = list[i]
        if e.id == id then
            if e.last_pos and active_lights[e.last_pos] then
                active_lights[e.last_pos][e.id] = nil
                update_light_at_pos(e.last_pos)
            end
            local extras = ITEM_DATA[e.item_name]
            if extras and extras._on_detach then extras._on_detach(target, e.entity) end
            if e.entity then e.entity:remove() end
            local stack = e.stack
            table.remove(list, i)
            Utils.sync_storage(target, ACTIVE_REGISTRY)
            return stack
        end
    end
    return false
end

function RADAPI.detach_all(target)
    local ref_id = Utils.get_object_id(target)
    local list = ACTIVE_REGISTRY[ref_id]
    if not list then return end
    local dropped_stacks = {}
    for i = #list, 1, -1 do
        local e = list[i]
        if e.last_pos and active_lights[e.last_pos] then
            active_lights[e.last_pos][e.id] = nil
            update_light_at_pos(e.last_pos)
        end
        local extras = ITEM_DATA[e.item_name]
        if extras and extras._on_detach then extras._on_detach(target, e.entity) end
        if e.entity then e.entity:remove() end
        table.insert(dropped_stacks, e.stack)
    end
    ACTIVE_REGISTRY[ref_id] = nil
    if target:is_player() then Utils.clear_storage(target:get_player_name()) end
    return dropped_stacks
end

function RADAPI.drop_all_attachments(target)
    local pos = target:get_pos()
    local stacks = RADAPI.detach_all(target)
    if stacks and pos then
        for _, stack in ipairs(stacks) do core.add_item(pos, stack) end
    end
end

-------------------------------------------------------------------------------
-- DISPLAY NODES (Frames & Pedestals)
-------------------------------------------------------------------------------

function RADAPI.register_item_frame(modname, name, def)
    local display_props = { 
        offset_value = def.display_offset_value or {x=0, y=0, z=0}, 
        visual_size = def.display_visual_size or {x = 0.4, y = 0.4} 
    }
    RADAPI.register(modname, name, {
        type = "node", description = def.description, 
        drawtype = "mesh", mesh = def.mesh, tiles = def.tiles or {""},
        paramtype = "light", paramtype2 = "wallmounted", 
        groups = table.copy(def.groups or {choppy=2, radapi_display=1}),
        _radapi_display_props = display_props,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then return itemstack end
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item_full")
            if clicker:get_player_control().sneak and stored ~= "" then
                local step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", step)
                Utils.spawn_display_entity(pos, stored, node.param2, display_props, "wallmounted", step, ITEM_DATA)
                return
            end
            if itemstack:is_empty() and stored ~= "" then
                local out = RADAPI.get_node_item(pos)
                clicker:get_inventory():add_item("main", out)
                Utils.remove_display_entities(pos)
                meta:set_string("radapi:item_full", "")
                return out
            elseif not itemstack:is_empty() and stored == "" then
                local stack = itemstack:take_item()
                local s = core.serialize(stack:to_table())
                meta:set_string("radapi:item_full", s)
                Utils.spawn_display_entity(pos, s, node.param2, display_props, "wallmounted", 0, ITEM_DATA)
                return itemstack
            end
        end,
        on_destruct = function(pos)
            local out = RADAPI.get_node_item(pos)
            Utils.remove_display_entities(pos)
            if out then core.add_item(pos, out) end
        end,
    })
end

function RADAPI.register_pedestal(modname, name, def)
    local display_props = { 
        offset_value = def.display_offset_value or {x=0, y=0.5, z=0}, 
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75} 
    }
    RADAPI.register(modname, name, {
        type = "node", description = def.description, 
        drawtype = "mesh", mesh = def.mesh, tiles = def.tiles or {""},
        paramtype2 = "facedir", 
        groups = table.copy(def.groups or {choppy=2, radapi_display=1}),
        _radapi_display_props = display_props,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then return itemstack end
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item_full")
            if clicker:get_player_control().sneak and stored ~= "" then
                local step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", step)
                Utils.spawn_display_entity(pos, stored, node.param2, display_props, "facedir", step, ITEM_DATA)
                return
            end
            if itemstack:is_empty() and stored ~= "" then
                local out = RADAPI.get_node_item(pos)
                clicker:get_inventory():add_item("main", out)
                Utils.remove_display_entities(pos)
                meta:set_string("radapi:item_full", "")
                return out
            elseif not itemstack:is_empty() and stored == "" then
                local stack = itemstack:take_item()
                local s = core.serialize(stack:to_table())
                meta:set_string("radapi:item_full", s)
                Utils.spawn_display_entity(pos, s, node.param2, display_props, "facedir", 0, ITEM_DATA)
                return itemstack
            end
        end,
        on_destruct = function(pos)
            local out = RADAPI.get_node_item(pos)
            Utils.remove_display_entities(pos)
            if out then core.add_item(pos, out) end
        end,
    })
end

-------------------------------------------------------------------------------
-- TORCH ENGINE (API INTEGRATED)
-------------------------------------------------------------------------------

function RADAPI.register_torch(modname, basename, def)
    local torch_timer = tonumber(core.settings:get("mytorches.torch_time")) or 3600
    
    local stages = {
        {suffix = "",      light = 14, next = "_med", desc = ""},
        {suffix = "_med",  light = 11, next = "_dim", desc = " (Dying)"},
        {suffix = "_dim",  light = 8,  next = "_out", desc = " (Flickering)"},
        {suffix = "_out",  light = 0,  next = "air",  desc = " (Burnt Out)"}
    }

    local types = {
        {name = "",         m_key = "mesh",         nb_key = "nodebox"},
        {name = "_wall",    m_key = "mesh_wall",    nb_key = "nodebox_wall"},
        {name = "_ceiling", m_key = "mesh_ceiling", nb_key = "nodebox_ceiling"}
    }

    for _, stage in ipairs(stages) do
        for _, t in ipairs(types) do
            local item_node = basename .. t.name .. stage.suffix
            local full_name = modname .. ":" .. item_node
            
            -- Override core default if it's the base torch
            if basename == "torch" and stage.suffix == "" and t.name == "" then
                full_name = "default:torch"
            end

            local node_def = {
                description = (def.description or "Torch") .. stage.desc,
                drawtype = def.drawtype or "mesh",
                mesh = def[t.m_key],
                node_box = def[t.nb_key],
                tiles = def.tiles or {""},
                paramtype = "light",
                paramtype2 = "facedir",
                walkable = false,
                light_source = stage.light,
                wield_glow = stage.light,
                selection_box = def.selection_box or {type="fixed", fixed={-0.15, -0.5, -0.15, 0.15, 0.4, 0.15}},
                groups = table.copy(def.groups or {dig_immediate=3, radapi_torch=1}),
                
                on_timer = function(pos)
                    if stage.next == "air" then
                        core.remove_node(pos)
                    else
                        local n = core.get_node(pos)
                        local next_node = modname .. ":" .. basename .. t.name .. stage.next
                        core.set_node(pos, {name = next_node, param2 = n.param2})
                        core.get_node_timer(pos):start(torch_timer)
                    end
                end,

                on_place = (stage.suffix == "" and t.name == "") and function(itemstack, placer, pointed_thing)
                    if pointed_thing.type ~= "node" then return itemstack end
                    local p0, p1 = pointed_thing.under, pointed_thing.above
                    local target = full_name
                    if p0.y > p1.y then target = modname..":"..basename.."_ceiling"
                    elseif p0.y < p1.y then target = full_name
                    else target = modname..":"..basename.."_wall" end
                    
                    local res = core.item_place(ItemStack(target), placer, pointed_thing)
                    if res then 
                        core.get_node_timer(p1):start(torch_timer)
                        itemstack:take_item() 
                    end
                    return itemstack
                end or nil,
            }

            if stage.suffix ~= "" or t.name ~= "" then
                node_def.groups.not_in_creative_inventory = 1
            end

            RADAPI.register(modname, item_node, node_def)
        end
    end
end

-- Gravity ABM
core.register_abm({
    nodenames = {"group:radapi_torch"},
    interval = 1.0,
    chance = 1,
    action = function(pos, node)
        if core.get_node({x=pos.x, y=pos.y-1, z=pos.z}).name == "air" then
            core.remove_node(pos)
            if not node.name:find("out") then core.add_item(pos, "default:torch") end
        end
    end
})

-------------------------------------------------------------------------------
-- ENGINE HOOKS & LBM
-------------------------------------------------------------------------------

core.register_lbm({
    name = mod_name .. ":restore_displays",
    nodenames = {"group:radapi_display"},
    run_at_every_load = true,
    action = function(pos, node)
        local meta = core.get_meta(pos)
        local stored = meta:get_string("radapi:item_full")
        if stored ~= "" then
            Utils.remove_display_entities(pos)
            local def = ITEM_DATA[node.name]
            local props = def and def._radapi_display_props or {}
            local ptype = core.get_node_def(node.name).paramtype2 or "facedir"
            local step = meta:get_int("radapi:yaw_step")
            Utils.spawn_display_entity(pos, stored, node.param2, props, ptype, step, ITEM_DATA)
        end
    end,
})

minetest.register_on_joinplayer(function(player)
    local entries = Utils.load_storage(player:get_player_name())
    if entries then
        for _, e in ipairs(entries) do 
            RADAPI.attach_entity(player, ItemStack(e.item_data), {id = e.id}) 
        end
    end
end)

minetest.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    if ACTIVE_REGISTRY[name] then
        for _, e in ipairs(ACTIVE_REGISTRY[name]) do
            if e.last_pos and active_lights[e.last_pos] then
                active_lights[e.last_pos][e.id] = nil
                update_light_at_pos(e.last_pos)
            end
            if e.entity then e.entity:remove() end
        end
    end
    ACTIVE_REGISTRY[name] = nil
end)

minetest.register_on_dieplayer(function(player) 
    RADAPI.drop_all_attachments(player) 
end)

core.register_entity("radapi:wield_entity", { initial_properties = { visual = "mesh", mesh = "blank.glb", static_save = false } })
core.register_entity("radapi:wield_entity_item", { initial_properties = { visual = "wielditem", static_save = false } })

radapi = RADAPI