local API = {}
local CURRENT_MODNAME = core.get_current_modname()
local REGISTERED_DEFS = {}

API.Entities = {}

function API.get_entity_info(object)
    local guid = object:get_guid()
    return guid and API.Entities[guid]
end

local function deep_merge(target, source)
    for k, v in pairs(source) do
        if type(v) == "table" and type(target[k]) == "table" then
            deep_merge(target[k], v)
        else
            target[k] = v
        end
    end
end


function API.register_entity(modname, name, def)
    assert(type(def) == "table", "Entity definition must be a table")
    assert(modname and name, "Missing modname or entity name")
    
    local def_id = modname .. ":" .. name
    assert(not REGISTERED_DEFS[def_id], "Entity already registered: " .. def_id)
    
    REGISTERED_DEFS[def_id] = def
end

function API.morph_entity(object, def_id, initial_state)
    local guid = object:get_guid()
    local blueprint = REGISTERED_DEFS[def_id]

    if not guid then
        core.log("error", "[API] Entity has no GUID when morphing")
        return
    end

    if not blueprint then
        core.log("error", "[API] Unknown def_id: " .. tostring(def_id))
        return
    end

    local current_info = API.get_entity_info(object)
    
    if current_info and current_info.name == def_id then
        if initial_state then
            deep_merge(current_info.state, initial_state)
            if blueprint._on_morph then
                blueprint._on_morph(object, current_info.state)
            end
        end
        return
    end

    local state = {}
    if initial_state then
        deep_merge(state, initial_state)
    end

    API.Entities[guid] = {
        name = def_id,
        state = state,
        def = blueprint
    }

    if blueprint.initial_properties then
        object:set_properties(blueprint.initial_properties)
    end

    if blueprint._on_morph then
        blueprint._on_morph(object, state)
    end
end

core.register_entity(CURRENT_MODNAME .. ":blank_entity", {
    initial_properties = {
        visual = "sprite",
        textures = {"blank.png"},
        static_save = true,  -- Critical for entity persistence
    },

    on_activate = function(self, staticdata, dtime_s)
        local guid = self.object:get_guid()
        if not guid then return end
        
        local saved_data = staticdata ~= "" and core.deserialize(staticdata) or {}
        
        if saved_data.name and not API.get_entity_info(self.object) then
            local blueprint = REGISTERED_DEFS[saved_data.name]
            if blueprint then
                API.Entities[guid] = {
                    name = saved_data.name,
                    state = saved_data.state or {},
                    def = blueprint
                }

                self.object:set_properties(blueprint.initial_properties)
            end
        end

        local info = API.get_entity_info(self.object)
        if info and info.def and info.def.on_activate then
            info.def.on_activate(self, staticdata, dtime_s, info.state)
        end
    end,

    on_step = function(self, dtime)
        local info = API.get_entity_info(self.object)
        if info and info.def and info.def.on_step then
            info.def.on_step(self, dtime, info.state)
        end
    end,

    on_deactivate = function(self, removal)
        local info = API.get_entity_info(self.object)
        if info and info.def and info.def.on_deactivate then
            info.def.on_deactivate(self, removal, info.state)
        end

        if removal then
            local guid = self.object:get_guid()
            if guid then API.Entities[guid] = nil end
        end
    end,

    get_staticdata = function(self)
        local info = API.get_entity_info(self.object)
        if not info then return "" end

        local data = {
            name = info.name,
            state = info.state
        }
        
        if info.def and info.def.get_staticdata then
            local extra = info.def.get_staticdata(info.state)
            if type(extra) == "table" then
                for k, v in pairs(extra) do
                    data[k] = v
                end
            end
        end

        return core.serialize(data)
    end
})

radapi.v3.entities = API