local Utils = {}
local storage = core.get_mod_storage()

function Utils.get_object_id(entity)
    if entity:is_player() then
        return true, entity:get_player_name()
    else
        local luaentity = get_luaentity(entity)
        if luaentity and luaentity.name then
            return true, luaentity.name
        else
            return false, "no name"
        end
    end
end

-- Generates a unique ID for attachment tracking
function Utils.generate_id()
  local timestamp = os.time()
  local random_number = math.random(1000, 9999)
  return "id_" .. timestamp .. "_" .. random_number
end

function Utils.merge_properties(base, overlay)
    local new = table.copy(base)
    for k, v in pairs(overlay) do new[k] = v end
    return new
end

-- Saves attachment data (Full Table Metadata) to ModStorage
function Utils.sync_storage(target, active_registry)
    if not target or not target:is_player() then return end
    local name = target:get_player_name()
    local list = active_registry[name] or {}
    
    local save_data = {}
    for _, entry in ipairs(list) do
        table.insert(save_data, {
            -- Save as table to preserve metadata (names, wear, enchantments)
            item_data = entry.stack:to_table(),
            id = entry.id
        })
    end
    
    storage:set_string("attachments:" .. name, core.serialize(save_data))
end

-- Loads data from ModStorage
function Utils.load_storage(player_name)
    local data = storage:get_string("attachments:" .. player_name)
    return (data ~= "") and core.deserialize(data) or nil
end

-- Clears ModStorage for a specific player
function Utils.clear_storage(player_name)
    storage:set_string("attachments:" .. player_name, "")
end

-- Cleans up any existing display entities at a specific node position
function Utils.remove_display_entities(pos)
    for _, obj in ipairs(core.get_objects_inside_radius(pos, 0.75)) do
        local ent = obj:get_luaentity()
        if ent and (ent.name == "radapi:wield_entity_item" or ent.name == "radapi:wield_entity") then
            obj:remove()
        end
    end
end

-- Spawns the visual representation with Full Metadata support
function Utils.spawn_display_entity(pos, item_raw, param2, display_props, ptype2, rotation_step, registry_data)
    -- Detect if input is a serialized table or a standard string
    local stack
    if type(item_raw) == "string" and item_raw:sub(1,1) == "{" then
        stack = ItemStack(core.deserialize(item_raw))
    else
        stack = ItemStack(item_raw)
    end
    
    if stack:is_empty() then return end
    
    local item_name = stack:get_name()
    local extras = registry_data[item_name]
    
    local dir = (ptype2 == "wallmounted") and core.wallmounted_to_dir(param2) or core.facedir_to_dir(param2)
    
    -- Vector Offset logic
    local offset_vec = display_props and display_props.offset_value or {x=0, y=0, z=0}
    if type(offset_vec) == "number" then
        offset_vec = vector.multiply(dir, offset_vec)
    end
    
    local ent_pos = vector.add(pos, offset_vec)
    Utils.remove_display_entities(pos)
    
    local is_wield = (not extras) or (extras.wieldview == "wielditem") or (extras.wieldview == "itemframe")
    local ent_type = is_wield and "radapi:wield_entity_item" or "radapi:wield_entity"
    local ent = core.add_entity(ent_pos, ent_type)
    
    if ent then
        if is_wield then
            ent:set_properties({
                visual = "wielditem",
                wield_item = item_name,
                visual_size = display_props and display_props.visual_size or {x = 0.4, y = 0.4},
            })
        else
            ent:set_properties(Utils.merge_properties(ent:get_properties(), extras.properties or {}))
        end
        
        local yaw = core.dir_to_yaw(dir) + ((rotation_step or 0) * (math.pi / 2))
        local pitch = (dir.y ~= 0) and (dir.y * (math.pi / 2)) or 0
        if ent.set_rotation then 
            ent:set_rotation({x = pitch, y = yaw, z = 0})
        else 
            ent:set_yaw(yaw) 
        end
    end
end

return Utils