# RADAPI - Register, Attach and Detach API (v2.4.0)

**RADAPI** is a high-performance utility for Minetest that bridges the gap between static items and dynamic visual entities. It allows modders to attach 3D meshes or 2D sprites to players and entities, provides customizable display nodes, features a **built-in dynamic lighting engine**, a **modular automation system**, and a **Newtonian Physics engine** for projectiles and archery.

---

## 1. Registration API

### `radapi.register(modname, item_name, def)`

Registers an item and its RADAPI properties. Handles handheld light, physical weight, and visual data.

**Example A: A Basic Glowing Tool**

```lua
radapi.register("emerald_mod", "glow_sword", {
    type = "tool",
    description = "Emerald Greatsword",
    inventory_image = "emerald_sword.png",
    wield_glow = 12, 
    weight = 5.5, -- Weight in kg (affects throw physics/damage)
    tool_capabilities = {
        full_punch_interval = 1.0,
        max_drop_level=1,
        groupcaps={ fleshy={times={2.0, 1.4, 1.0}, uses=20, maxlevel=3} }
    }
})

```

**Example B: 3D Wearable Item**

```lua
radapi.register("cosmetics", "top_hat", {
    type = "craftitem",
    description = "Fancy Top Hat",
    inventory_image = "hat_icon.png",
    wieldview = "mesh",
    properties = {
        mesh = "top_hat.obj",
        textures = {"wool_black.png"},
        visual_size = {x=1.2, y=1.2, z=1.2},
    },
    attach = {
        bone = "Head",
        pos = {x=0, y=6.5, z=0},
        rot = {x=0, y=0, z=0},
    }
})

```

---

## 2. Archery & Physics API (v2.4.0)

### `radapi.register_bow(modname, name, def)`

Creates a functional bow with built-in charging, FOV zoom, and projectile logic.

**Example: Heavy Crossbow**

```lua
radapi.register_bow("siege", "crossbow_heavy", {
    description = "Heavy Steel Crossbow",
    inventory_image = "crossbow_inv.png",
    ammo_type = "default:arrow",
    max_force = 60,       -- High velocity
    fov_multiplier = 0.5,  -- High zoom (Sniper feel)
    zoom_speed = 5,       -- Slow, heavy zoom transition
    fire_sound = "crossbow_release"
})

```

### Customizable Arrow Trails

Arrows can now have fully customizable particle trails:

```lua
radapi.register("magic", "fire_arrow", {
    type = "craftitem",
    description = "Fire Arrow",
    inventory_image = "fire_arrow.png",
    weight = 0.8,
    trail_def = {
        image = "fire_trail.png^[opacity:200",
        glow = 10,
        amount = 20,
        minvel = {x=-0.3, y=-0.3, z=-0.3},
        maxvel = {x=0.3, y=0.3, z=0.3},
        minexptime = 0.2,
        maxexptime = 0.5,
        minsize = 0.4,
        maxsize = 1.0,
    }
})
```

**Trail Parameters:**
- `image` - Texture for particles (supports colorization/opacity)
- `glow` - Light emission level (0-14)
- `amount` - Number of particles per second
- `minvel/maxvel` - Velocity range for particle spread
- `minexptime/maxexptime` - Particle lifetime range
- `minsize/maxsize` - Size variation of particles

### `radapi.throw_item(player, itemstack, force)`

Launches an item using Newtonian gravity.

* **Sticky Projectiles:** Items automatically "pin" to entities or stick into walls.
* **Kinetic Damage:** Damage is calculated based on `velocity * weight`.
* **Custom Trails:** Particle effects are automatically generated based on item definition.

---

## 3. Hopper & Automation API

### `radapi.register_hopper(modname, name, def)`

Creates an automated item-transfer node. Hoppers pull items from a source and push them to a target.

**Example A: Standard Industrial Hopper**

```lua
radapi.register_hopper("factory", "steel_hopper", {
    description = "Steel Item Hopper",
    mesh = "factory_hopper.obj",
    tiles = {"steel_block.png"},
    source_offset = {x=0, y=1, z=0}, -- Above
    target_offset = {x=0, y=-1, z=0} -- Below
})

```

**Example B: Side-Ejecting Sorting Arm**

```lua
radapi.register_hopper("factory", "sorter_right", {
    description = "Side-Ejecting Sorter",
    mesh = "sorter_mesh.obj",
    tiles = {"brass_texture.png"},
    source_offset = {x=0, y=1, z=0},  -- Pull from top
    target_offset = {x=1, y=0, z=0}   -- Push to the RIGHT
})

```

---

## 4. Torch & Industrial API

### `radapi.register_torch(modname, basename, def)`

Creates a 12-node torch system with burnout stages (Normal -> Med -> Dim -> Out).

```lua
radapi.register_torch("survival", "torch", {
    description = "Dungeon Torch",
    drawtype = "mesh",
    mesh = "torch_floor.obj",
    mesh_wall = "torch_wall.obj",
    mesh_ceiling = "torch_ceiling.obj",
    tiles = {"torch_texture.png"}
})

```

### `radapi.register_furnace(modname, name, def)`

Registers a high-performance furnace with custom UI progress bars.

```lua
radapi.register_furnace("industry", "blast_furnace", {
    tiles = {"iron_side.png", "iron_side.png", "iron_side.png", "iron_side.png", "iron_side.png", "iron_front.png"},
    tiles_active = {"iron_side.png", "iron_side.png", "iron_side.png", "iron_side.png", "iron_side.png", "iron_front_active.png"},
    ui_fire_fg = "fire_anim.png",
    ui_arrow_fg = "arrow_anim.png"
})

```

---

## 5. Attachment & Detach API

### `radapi.attach_entity(target, itemstack, opts)`

**Example A: Backpack Accessory**

```lua
local stack = ItemStack("travel:backpack")
radapi.attach_entity(player, stack, { id = "back_slot" })

```

**Example B: NPC Equipment (Glowing Lantern)**

```lua
local lantern = ItemStack("lighting:lantern")
radapi.attach_entity(npc_object, lantern, { id = "left_hand_light" })

```

### `radapi.detach_entity(target, id)`

Removes an attachment by ID and returns the ItemStack.

---

## 6. World Display API

### `radapi.register_item_frame(modname, name, def)`

**Example: Secure Wall Display**

```lua
radapi.register_item_frame("vault", "security_frame", {
    description = "High-Security Item Frame",
    mesh = "frame_iron.obj",
    tiles = {"iron_block.png"},
    display_offset_value = {x=0, y=0, z=0.05},
    display_visual_size = {x=0.4, y=0.4}
})

```

### `radapi.register_pedestal(modname, name, def)`

**Example: Floating Artifact Pedestal**

```lua
radapi.register_pedestal("relics", "altar", {
    description = "Ancient Altar",
    mesh = "altar_stone.obj",
    tiles = {"stone_carved.png"},
    display_offset_value = {x=0, y=1.2, z=0}, -- Item floats high above
    display_visual_size = {x=1.0, y=1.0}
})

```

---

## 7. Utility & Data API

### `radapi.get_node_item(pos)`

Retrieves the item currently stored in a display node (frame/pedestal/hopper).

### `radapi.detach_all(target)`

Removes all attachments at once. Essential for cleaning up on player death.

---

## Technical Details

### Lighting & Zoom Engine

RADAPI uses a consolidated high-frequency `globalstep`. If a player holds a bow, it smoothly interpolates the Field of View (FOV). Simultaneously, it checks for `wield_glow` properties, managing invisible light nodes (`radapi:light_1` to `14`) to provide real-time illumination without server-side lag.

### Physics Collision

The engine utilizes `core.raycast` to prevent projectiles from passing through walls. When a collision is detected, the API calculates kinetic damage and automatically converts the flying entity into a "Stuck" entity or an attachment.

### Storage & Persistence

Visual attachments are persistent. Attachment IDs and item data are stored in `player_meta`. Upon rejoining, the API reconstructs all 3D visuals automatically.

---

### What's New in v2.4.0

- **Customizable Arrow Trails**: Full control over particle effects, textures, glow, and behavior
- **Enhanced Physics**: Improved collision detection and damage calculation
- **Performance Optimizations**: More efficient globalstep handling
- **API Improvements**: Better error handling and attachment management

---