local Utils = dofile(core.get_modpath(core.get_current_modname()) .. "/functions.lua")

local ITEM_DATA         = {}
local ACTIVE_REGISTRY   = {} 
local RADAPI            = {}
RADAPI.registered_hoppers = {}
 
local active_lights = {}
local GLOWING_TARGETS   = {}
local update_interval = 0.2
local mod_name = core.get_current_modname()
local glow_logic_initialized = false 

-- ==========================================
-- 1. PHYSICS & ARCHERY MODULE (NEW)
-- ==========================================

-- Entity: The Arrow (Flying)
core.register_entity(mod_name .. ":arrow_entity", {
    initial_properties = {
        visual = "wielditem",
        wield_item = "default:arrow",
        collisionbox = {0,0,0,0,0,0},
        physical = true,
    },
    on_activate = function(self)
        -- Visual Particle Trail (now customizable)
        if self.trail_def then
            local trail = self.trail_def
            core.add_particlespawner({
                amount = trail.amount or 15,
                time = 0,
                minpos = {x=0, y=0, z=0},
                maxpos = {x=0, y=0, z=0},
                minvel = trail.minvel or {x=-0.1, y=-0.1, z=-0.1},
                maxvel = trail.maxvel or {x=0.1, y=0.1, z=0.1},
                minexptime = trail.minexptime or 0.5,
                maxexptime = trail.maxexptime or 0.8,
                minsize = trail.minsize or 0.5,
                maxsize = trail.maxsize or 1.2,
                texture = trail.image or "radapi_trail.png^[opacity:120",
                attached = self.object,
                glow = trail.glow or 5,
            })
        end
    end,
    on_step = function(self, dtime)
        local pos = self.object:get_pos()
        local vel = self.object:get_velocity()
        
        -- Newtonian Rotation (Face direction of travel)
        if vector.length(vel) > 1 then
            self.object:set_yaw(math.atan2(vel.z, vel.x) - math.pi / 2)
            self.object:set_look_vertical(math.atan2(vel.y, vector.length({x=vel.x, y=0, z=vel.z})))
        end

        if self.last_pos then
            local ray = core.raycast(self.last_pos, pos, true, false)
            local hit = ray:next()
            if hit and hit.type ~= "nothing" then
                -- Kinetic Damage calculation
                if hit.type == "object" then
                    local damage = (vector.length(vel) * (self.mass or 1)) / 5
                    hit.ref:punch(self.object, 1.0, {
                        full_punch_interval = 1.0,
                        damage_groups = {fleshy = damage}
                    })
                    -- Stick to the Entity
                    RADAPI.attach_entity(hit.ref, ItemStack(self.itemstring), {id = "stuck_" .. core.get_gametime()})
                else
                    -- Stick into the Node
                    local stuck = core.add_entity(hit.intersection_point, mod_name .. ":stuck_arrow")
                    stuck:set_yaw(self.object:get_yaw())
                    stuck:get_luaentity().itemstring = self.itemstring
                end
                self.object:remove()
                return
            end
        end
        self.last_pos = pos
    end,
})

-- Entity: The Stuck Arrow (Persistent visual)
core.register_entity(mod_name .. ":stuck_arrow", {
    initial_properties = { visual = "wielditem", physical = false, pointable = true },
    on_activate = function(self) self.timer = 0 end,
    on_step = function(self, dtime)
        self.timer = self.timer + dtime
        if self.timer > 60 then self.object:remove() end -- Auto-clean after 60s
    end,
    on_punch = function(self, puncher)
        local inv = puncher:get_inventory()
        if inv:room_for_item("main", self.itemstring) then
            inv:add_item("main", self.itemstring)
            self.object:remove()
        end
    end
})

-- Launcher logic
function RADAPI.throw_item(player, itemstack, force)
    local p_pos = player:get_pos()
    local p_dir = player:get_look_dir()
    p_pos.y = p_pos.y + 1.5
    
    local ent = core.add_entity(p_pos, mod_name .. ":arrow_entity")
    if ent then
        local lua = ent:get_luaentity()
        local name = itemstack:get_name()
        local mass = (ITEM_DATA[name] and ITEM_DATA[name].weight) or 1
        local trail_def = ITEM_DATA[name] and ITEM_DATA[name].trail_def

        lua.mass = mass
        lua.itemstring = name
        lua.trail_def = trail_def -- Pass trail settings to entity
        
        ent:set_velocity(vector.multiply(p_dir, force))
        ent:set_acceleration({x=0, y=-9.81, z=0}) -- Gravity
        itemstack:take_item()
    end
    return itemstack
end

-- ==========================================
-- 2. LIGHTING SYSTEM (YOUR ORIGINAL)
-- ==========================================

for i = 1, 14 do
    core.register_node(":" .. mod_name .. ":light_" .. i, {
        drawtype = "airlike",
        paramtype = "light",
        light_source = i,
        sunlight_propagates = true,
        walkable = false,
        pointable = false,
        diggable = false,
        buildable_to = true,
        drop = "",
        groups = {not_in_creative_inventory = 1, radapi_light = 1},
    })
end

local function update_light_at_pos(pos_str)
    local lights = active_lights[pos_str]
    local pos = core.string_to_pos(pos_str)
    if not pos then return end

    local max_level = 0
    local has_any = false
    
    if lights then
        for _, level in pairs(lights) do
            has_any = true
            if level > max_level then max_level = level end
        end
    end

    local current_node = core.get_node(pos).name
    if not has_any or max_level == 0 then
        if core.get_item_group(current_node, "radapi_light") > 0 then
            core.swap_node(pos, {name = "air"})
        end
        active_lights[pos_str] = nil
        return
    end

    local target_node = mod_name .. ":light_" .. math.floor(max_level)
    if current_node == "air" or core.get_item_group(current_node, "radapi_light") > 0 then
        if current_node ~= target_node then core.swap_node(pos, {name = target_node}) end
    end
end

local function init_glow_logic()
    if glow_logic_initialized then return end
    glow_logic_initialized = true

    local timer = 0
    core.register_globalstep(function(dtime)
        timer = timer + dtime
        
        -- COMBINED GLOBALSTEP: Zoom Logic + Lighting Logic
        for _, player in ipairs(core.get_connected_players()) do
            local pname = player:get_player_name()
            local stack = player:get_wielded_item()
            local item_name = stack:get_name()

            -- [Zoom Handler]
            if core.get_item_group(item_name, "radapi_bow") > 0 then
                local start = stack:get_meta():get_float("charge_start")
                if start > 0 then
                    local held = core.get_gametime() - start
                    local z = ITEM_DATA[item_name]._radapi_zoom or {fov_mult = 0.7, zoom_speed = 10}
                    local zoom_factor = math.min(held / 2.0, 1.0)
                    player:set_fov(1 - (zoom_factor * (1 - z.fov_mult)), true, z.zoom_speed)
                end
            else
                player:set_fov(0, true, 20)
            end

            -- [Lighting Handler]
            if timer >= update_interval then
                local pos = vector.round(player:get_pos())
                pos.y = pos.y + 1 
                local pos_str = core.pos_to_string(pos)
                
                local info = ITEM_DATA[item_name]
                local hand_id = pname .. "_hand"
                local pmeta = player:get_meta()
                local last_hand_pos = pmeta:get_string("radapi:last_light_pos")

                if info and info.wield_glow then
                    local level = type(info.wield_glow) == "number" and info.wield_glow or 10
                    if last_hand_pos ~= "" and last_hand_pos ~= pos_str then
                        if active_lights[last_hand_pos] then
                            active_lights[last_hand_pos][hand_id] = nil
                            update_light_at_pos(last_hand_pos)
                        end
                    end
                    active_lights[pos_str] = active_lights[pos_str] or {}
                    active_lights[pos_str][hand_id] = level
                    update_light_at_pos(pos_str)
                    pmeta:set_string("radapi:last_light_pos", pos_str)
                elseif last_hand_pos ~= "" then
                    if active_lights[last_hand_pos] then
                        active_lights[last_hand_pos][hand_id] = nil
                        update_light_at_pos(last_hand_pos)
                    end
                    pmeta:set_string("radapi:last_light_pos", "")
                end
            end
        end

        if timer >= update_interval then
            -- Handle all glowing attachments
            for ref_id, target in pairs(GLOWING_TARGETS) do
                local p = target:get_pos()
                if p then
                    local pos = vector.round(p)
                    pos.y = pos.y + 1 
                    local pos_str = core.pos_to_string(pos)

                    if ACTIVE_REGISTRY[ref_id] then
                        for _, e in ipairs(ACTIVE_REGISTRY[ref_id]) do
                            if e.glow then
                                local level = type(e.glow) == "number" and e.glow or 10
                                if e.last_pos and e.last_pos ~= pos_str then
                                    if active_lights[e.last_pos] then
                                        active_lights[e.last_pos][e.id] = nil
                                        update_light_at_pos(e.last_pos)
                                    end
                                end
                                active_lights[pos_str] = active_lights[pos_str] or {}
                                active_lights[pos_str][e.id] = level
                                e.last_pos = pos_str
                                update_light_at_pos(pos_str)
                            end
                        end
                    end
                else
                    GLOWING_TARGETS[ref_id] = nil
                end
            end
            timer = 0
        end
    end)
end

-- ==========================================
-- 3. CORE API FUNCTIONS (ATTACHMENTS & REGISTRY)
-- ==========================================

function RADAPI.get_node_item(pos)
    local meta = core.get_meta(pos)
    local s = meta:get_string("radapi:item_full")
    if s == "" then return nil end
    return s:sub(1,1) == "{" and ItemStack(core.deserialize(s)) or ItemStack(s)
end

function RADAPI.register(modname, item_name, def)
    local full_name = modname .. ":" .. item_name
    if ITEM_DATA[full_name] then return false end
    
    if def.type == "tool" then core.register_tool(full_name, def)
    elseif def.type == "node" then core.register_node(full_name, def)
    elseif def.type == "craftitem" then core.register_craftitem(full_name, def)
    else core.register_node(full_name, def) end 
    
    if def.craft then core.register_craft(def.craft) end
    if def.wield_glow then init_glow_logic() end
    
    ITEM_DATA[full_name] = {
        properties = def.properties,
        attach     = def.attach,
        weight     = def.weight or 1,
        _on_attach = def._on_attach,
        _on_detach = def._on_detach,
        wieldview  = def.wieldview,
        wield_glow = def.wield_glow,
        _radapi_zoom = def._radapi_zoom,
        _radapi_display_props = def._radapi_display_props,
        trail_def  = def.trail_def, -- Store trail definition
    }
    return true
end

-- Special Registration for Bows
function RADAPI.register_bow(modname, name, def)
    RADAPI.register(modname, name, {
        type = "tool",
        description = def.description,
        inventory_image = def.inventory_image,
        groups = {radapi_bow = 1},
        _radapi_zoom = { fov_mult = def.fov_multiplier or 0.7, zoom_speed = def.zoom_speed or 10 },
        on_use = function(itemstack, user)
            itemstack:get_meta():set_float("charge_start", core.get_gametime())
            return itemstack
        end,
        on_secondary_use = function(itemstack, user)
            local meta = itemstack:get_meta()
            local start = meta:get_float("charge_start")
            if start == 0 then return itemstack end
            
            user:set_fov(0, true) -- Kill zoom
            local charge = math.min(core.get_gametime() - start, 2.0)
            local force = charge * (def.max_force or 40)
            
            local inv = user:get_inventory()
            if inv:contains_item("main", def.ammo_type) then
                inv:remove_item("main", def.ammo_type)
                RADAPI.throw_item(user, ItemStack(def.ammo_type), force)
                core.sound_play(def.fire_sound or "bow_fire", {pos=user:get_pos()})
            end
            meta:set_float("charge_start", 0)
            return itemstack
        end
    })
end

function RADAPI.attach_entity(target, itemstack, opts)
    opts = opts or {}
    if not target or not itemstack or itemstack:is_empty() then return false end
    local item_name = itemstack:get_name()
    local extras = ITEM_DATA[item_name]
    if not extras then return false end
    
    local ref_id = Utils.get_object_id(target)
    ACTIVE_REGISTRY[ref_id] = ACTIVE_REGISTRY[ref_id] or {}
    if opts.id then RADAPI.detach_entity(target, opts.id) end
    
    local is_wield = (extras.wieldview == "wielditem" or extras.wieldview == "itemframe")
    local ent_type = is_wield and "radapi:wield_entity_item" or "radapi:wield_entity"
    local ent = core.add_entity(target:get_pos(), ent_type)
    if not ent then return false end
    
    local props = Utils.merge_properties(ent:get_properties(), extras.properties or {})
    if is_wield then props.wield_item = item_name end
    ent:set_properties(props)
    
    local attach = extras.attach or {}
    ent:set_attach(target, attach.bone or "", attach.pos or {x=0,y=0,z=0}, attach.rot or {x=0,y=0,z=0})
    
    table.insert(ACTIVE_REGISTRY[ref_id], {
        entity = ent,
        item_name = item_name,
        stack = ItemStack(itemstack), 
        id = opts.id or Utils.generate_id(),
        glow = extras.wield_glow,
        last_pos = nil
    })
    
    if extras.wield_glow then GLOWING_TARGETS[ref_id] = target end
    if extras._on_attach then extras._on_attach(target, ent) end
    Utils.sync_storage(target, ACTIVE_REGISTRY)
    return true
end

function RADAPI.detach_entity(target, id)
    local ref_id = Utils.get_object_id(target)
    local list = ACTIVE_REGISTRY[ref_id]
    if not list then return false end
    for i = #list, 1, -1 do
        local e = list[i]
        if e.id == id then
            if e.glow then
                if e.last_pos and active_lights[e.last_pos] then
                    active_lights[e.last_pos][e.id] = nil
                    update_light_at_pos(e.last_pos)
                end
                local has_other_glow = false
                for _, other_e in ipairs(list) do
                    if other_e.id ~= id and other_e.glow then
                        has_other_glow = true
                        break
                    end
                end
                if not has_other_glow then GLOWING_TARGETS[ref_id] = nil end
            end
            local extras = ITEM_DATA[e.item_name]
            if extras and extras._on_detach then extras._on_detach(target, e.entity) end
            if e.entity then e.entity:remove() end
            local stack = e.stack
            table.remove(list, i)
            Utils.sync_storage(target, ACTIVE_REGISTRY)
            return stack
        end
    end
    return false
end

function RADAPI.detach_all(target)
    local ref_id = Utils.get_object_id(target)
    local list = ACTIVE_REGISTRY[ref_id]
    if not list then return end
    local dropped_stacks = {}
    for i = #list, 1, -1 do
        local e = list[i]
        if e.last_pos and active_lights[e.last_pos] then
            active_lights[e.last_pos][e.id] = nil
            update_light_at_pos(e.last_pos)
        end
        local extras = ITEM_DATA[e.item_name]
        if extras and extras._on_detach then extras._on_detach(target, e.entity) end
        if e.entity then e.entity:remove() end
        table.insert(dropped_stacks, e.stack)
    end
    ACTIVE_REGISTRY[ref_id] = nil
    GLOWING_TARGETS[ref_id] = nil
    if target:is_player() then Utils.clear_storage(target:get_player_name()) end
    return dropped_stacks
end

function RADAPI.drop_all_attachments(target)
    local pos = target:get_pos()
    local stacks = RADAPI.detach_all(target)
    if stacks and pos then
        for _, stack in ipairs(stacks) do core.add_item(pos, stack) end
    end
end

-- ==========================================
-- 4. HOPPER & INDUSTRIAL SYSTEMS
-- ==========================================

function RADAPI.register_hopper(modname, name, def)
    local full_name = modname .. ":" .. name
    RADAPI.register(modname, name, {
        type = "node",
        description = def.description or "RADAPI Hopper",
        drawtype = "mesh",
        mesh = def.mesh or "hopper.obj",
        tiles = def.tiles or {""},
        paramtype = "light",
        paramtype2 = "facedir",
        groups = table.copy(def.groups or {choppy=2, radapi_hopper=1}),
        on_construct = function(pos)
            local meta = core.get_meta(pos)
            meta:get_inventory():set_size("main", 4)
        end,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then return itemstack end
            local spos = pos.x .. "," .. pos.y .. "," .. pos.z
            local formspec = "size[8,6.5]" ..
                "list[nodemeta:" .. spos .. ";main;3,0.5;2,2;]" ..
                "list[current_player;main;0,2.8;8,1;]" ..
                "list[current_player;main;0,4.0;8,3;8]" ..
                "listring[nodemeta:" .. spos .. ";main]" ..
                "listring[current_player;main]"
            core.show_formspec(pname, full_name, formspec)
        end,
        can_dig = function(pos)
            return core.get_meta(pos):get_inventory():is_empty("main")
        end,
    })
    RADAPI.registered_hoppers[full_name] = {
        pull_offset = def.source_offset or {x=0, y=1, z=0},
        push_offset = def.target_offset or {x=0, y=-1, z=0},
    }
end

core.register_abm({
    label = "RADAPI_Hopper_System",
    nodenames = {"group:radapi_hopper"},
    interval = 2.0, chance = 1,
    action = function(pos, node)
        local h_data = RADAPI.registered_hoppers[node.name]
        if not h_data then return end
        local inv = core.get_meta(pos):get_inventory()
        local p_src = vector.add(pos, h_data.pull_offset)
        local p_dst = vector.add(pos, h_data.push_offset)
        local i_src = core.get_meta(p_src):get_inventory()
        if i_src and i_src:get_size("main") > 0 then
            for i = 1, i_src:get_size("main") do
                local stack = i_src:get_stack("main", i)
                if not stack:is_empty() and inv:room_for_item("main", stack:get_name()) then
                    inv:add_item("main", stack:take_item(1))
                    i_src:set_stack("main", i, stack)
                    break
                end
            end
        end
        if not inv:is_empty("main") then
            local i_dst = core.get_meta(p_dst):get_inventory()
            if i_dst then
                local t_list = i_dst:get_size("main") > 0 and "main" or (i_dst:get_size("src") > 0 and "src" or "main")
                for i = 1, inv:get_size("main") do
                    local stack = inv:get_stack("main", i)
                    if not stack:is_empty() and i_dst:room_for_item(t_list, stack:get_name()) then
                        i_dst:add_item(t_list, stack:take_item(1))
                        inv:set_stack("main", i, stack)
                        break
                    end
                end
            end
        end
    end
})

-- ==========================================
-- 5. FURNACE SYSTEM
-- ==========================================

function RADAPI.register_furnace(modname, name, def)
    local active_name = modname .. ":" .. name .. "_active"
    local inactive_name = modname .. ":" .. name
    local ui_fire_bg, ui_fire_fg = def.ui_fire_bg or "blank.png", def.ui_fire_fg or "blank.png"
    local ui_arrow_bg, ui_arrow_fg = def.ui_arrow_bg or "blank.png", def.ui_arrow_fg or "blank.png"

    local function get_fs(pos, lit, smelt)
        local spos = pos.x .. "," .. pos.y .. "," .. pos.z
        return "size[8,8.5]list[nodemeta:"..spos..";src;2.5,0.5;1,1;]list[nodemeta:"..spos..";fuel;2.5,2.5;1,1;]list[nodemeta:"..spos..";dst;5.5,1.5;1,1;]"..
               "image[2.5,1.5;1,1;"..ui_fire_bg.."^[lowpart:"..lit..":"..ui_fire_fg.."]"..
               "image[3.7,1.5;1.5,1;"..ui_arrow_bg.."^[lowpart:"..smelt..":"..ui_arrow_fg.."^[transformR270]"..
               "list[current_player;main;0,4.25;8,1;]list[current_player;main;0,5.5;8,3;8]"
    end

    RADAPI.register(modname, name, {
        tiles = def.tiles, paramtype2 = "facedir", groups = table.copy(def.groups or {cracky=2, radapi_furnace=1}),
        on_construct = function(pos)
            local inv = core.get_meta(pos):get_inventory()
            inv:set_size("src", 1) inv:set_size("fuel", 1) inv:set_size("dst", 1)
        end,
        on_timer = function(pos)
            local meta, inv = core.get_meta(pos), core.get_meta(pos):get_inventory()
            local f_time, f_total = meta:get_float("fuel_time"), meta:get_float("fuel_totaltime")
            local s_time = meta:get_float("src_time")
            local cookable = core.get_craft_result({method="cooking", width=1, items=inv:get_list("src")})
            local fuelable = core.get_craft_result({method="fuel", width=1, items=inv:get_list("fuel")})

            if f_time <= 0 then
                if cookable.time > 0 and fuelable.time > 0 then
                    f_time = fuelable.time f_total = f_time
                    local fs = inv:get_stack("fuel", 1) fs:take_item(1) inv:set_stack("fuel", 1, fs)
                    meta:set_float("fuel_totaltime", f_total)
                    core.swap_node(pos, {name = active_name, param2 = core.get_node(pos).param2})
                else
                    core.swap_node(pos, {name = inactive_name, param2 = core.get_node(pos).param2})
                    meta:set_string("formspec", get_fs(pos, 0, 0)) return false
                end
            end
            if f_time > 0 then
                f_time = f_time - 1
                if cookable.time > 0 then
                    s_time = s_time + 1
                    if s_time >= cookable.time and inv:room_for_item("dst", cookable.item) then
                        inv:add_item("dst", cookable.item)
                        local ss = inv:get_stack("src", 1) ss:take_item(1) inv:set_stack("src", 1, ss)
                        s_time = 0
                    end
                end
            end
            meta:set_float("fuel_time", f_time) meta:set_float("src_time", s_time)
            meta:set_string("formspec", get_fs(pos, (f_time/f_total)*100, (s_time/cookable.time)*100))
            return true
        end,
        on_metadata_inventory_put = function(pos) core.get_node_timer(pos):start(1.0) end,
    })
    local adef = table.copy(core.registered_nodes[inactive_name])
    adef.light_source = 13 adef.tiles = def.tiles_active adef.groups.not_in_creative_inventory = 1
    RADAPI.register(modname, name .. "_active", adef)
end

-- ==========================================
-- 6. DISPLAY NODES (ITEM FRAMES & PEDESTALS)
-- ==========================================

function RADAPI.register_item_frame(modname, name, def)
    local display_props = { 
        offset_value = def.display_offset_value or {x=0, y=0, z=0}, 
        visual_size = def.display_visual_size or {x = 0.4, y = 0.4} 
    }
    RADAPI.register(modname, name, {
        type = "node", description = def.description, 
        drawtype = "mesh", mesh = def.mesh, tiles = def.tiles or {""},
        paramtype = "light", paramtype2 = "wallmounted", 
        groups = table.copy(def.groups or {choppy=2, radapi_display=1}),
        _radapi_display_props = display_props,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then return itemstack end
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item_full")
            if clicker:get_player_control().sneak and stored ~= "" then
                local step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", step)
                Utils.spawn_display_entity(pos, stored, node.param2, display_props, "wallmounted", step, ITEM_DATA)
                return
            end
            if itemstack:is_empty() and stored ~= "" then
                local out = RADAPI.get_node_item(pos)
                clicker:get_inventory():add_item("main", out)
                Utils.remove_display_entities(pos)
                meta:set_string("radapi:item_full", "")
                return out
            elseif not itemstack:is_empty() and stored == "" then
                local stack = itemstack:take_item()
                local s = core.serialize(stack:to_table())
                meta:set_string("radapi:item_full", s)
                Utils.spawn_display_entity(pos, s, node.param2, display_props, "wallmounted", 0, ITEM_DATA)
                return itemstack
            end
        end,
        on_destruct = function(pos)
            local out = RADAPI.get_node_item(pos)
            Utils.remove_display_entities(pos)
            if out then core.add_item(pos, out) end
        end,
    })
end

function RADAPI.register_pedestal(modname, name, def)
    local display_props = { 
        offset_value = def.display_offset_value or {x=0, y=0.5, z=0}, 
        visual_size = def.display_visual_size or {x = 0.75, y = 0.75} 
    }
    RADAPI.register(modname, name, {
        type = "node", description = def.description, 
        drawtype = "mesh", mesh = def.mesh, tiles = def.tiles or {""},
        paramtype2 = "facedir", 
        groups = table.copy(def.groups or {choppy=2, radapi_display=1}),
        _radapi_display_props = display_props,
        on_rightclick = function(pos, node, clicker, itemstack)
            local pname = clicker:get_player_name()
            if core.is_protected(pos, pname) then return itemstack end
            local meta = core.get_meta(pos)
            local stored = meta:get_string("radapi:item_full")
            if clicker:get_player_control().sneak and stored ~= "" then
                local step = (meta:get_int("radapi:yaw_step") + 1) % 4
                meta:set_int("radapi:yaw_step", step)
                Utils.spawn_display_entity(pos, stored, node.param2, display_props, "facedir", step, ITEM_DATA)
                return
            end
            if itemstack:is_empty() and stored ~= "" then
                local out = RADAPI.get_node_item(pos)
                clicker:get_inventory():add_item("main", out)
                Utils.remove_display_entities(pos)
                meta:set_string("radapi:item_full", "")
                return out
            elseif not itemstack:is_empty() and stored == "" then
                local stack = itemstack:take_item()
                local s = core.serialize(stack:to_table())
                meta:set_string("radapi:item_full", s)
                Utils.spawn_display_entity(pos, s, node.param2, display_props, "facedir", 0, ITEM_DATA)
                return itemstack
            end
        end,
        on_destruct = function(pos)
            local out = RADAPI.get_node_item(pos)
            Utils.remove_display_entities(pos)
            if out then core.add_item(pos, out) end
        end,
    })
end

-- ==========================================
-- 7. TIMED TORCHES SYSTEM
-- ==========================================

function RADAPI.register_torch(modname, basename, def)
    local torch_timer = tonumber(core.settings:get("mytorches.torch_time")) or 3600
    local stages = {
        {suffix = "",    light = 14, next = "_med", desc = ""},
        {suffix = "_med",  light = 11, next = "_dim", desc = " (Dying)"},
        {suffix = "_dim",  light = 8,  next = "_out", desc = " (Flickering)"},
        {suffix = "_out",  light = 0,  next = "air",  desc = " (Burnt Out)"}
    }
    local types = {
        {name = "",          m_key = "mesh",         nb_key = "nodebox"},
        {name = "_wall",      m_key = "mesh_wall",     nb_key = "nodebox_wall"},
        {name = "_ceiling", m_key = "mesh_ceiling", nb_key = "nodebox_ceiling"}
    }
    for _, stage in ipairs(stages) do
        for _, t in ipairs(types) do
            local item_node = basename .. t.name .. stage.suffix
            local full_name = modname .. ":" .. item_node
            if basename == "torch" and stage.suffix == "" and t.name == "" then full_name = "default:torch" end
            local node_def = {
                description = (def.description or "Torch") .. stage.desc,
                drawtype = def.drawtype or "mesh",
                mesh = def[t.m_key], node_box = def[t.nb_key], tiles = def.tiles or {""},
                paramtype = "light", paramtype2 = "facedir", walkable = false,
                light_source = stage.light, wield_glow = stage.light,
                groups = table.copy(def.groups or {dig_immediate=3, radapi_torch=1}),
                on_timer = function(pos)
                    if stage.next == "air" then core.remove_node(pos)
                    else
                        local n = core.get_node(pos)
                        local next_node = modname .. ":" .. basename .. t.name .. stage.next
                        core.set_node(pos, {name = next_node, param2 = n.param2})
                        core.get_node_timer(pos):start(torch_timer)
                    end
                end,
                on_place = (stage.suffix == "" and t.name == "") and function(itemstack, placer, pointed_thing)
                    if pointed_thing.type ~= "node" then return itemstack end
                    local p1 = pointed_thing.above
                    local target = full_name
                    if pointed_thing.under.y > p1.y then target = modname..":"..basename.."_ceiling"
                    elseif pointed_thing.under.y < p1.y then target = full_name
                    else target = modname..":"..basename.."_wall" end
                    local res = core.item_place(ItemStack(target), placer, pointed_thing)
                    if res then core.get_node_timer(p1):start(torch_timer) itemstack:take_item() end
                    return itemstack
                end or nil,
            }
            if stage.suffix ~= "" or t.name ~= "" then node_def.groups.not_in_creative_inventory = 1 end
            RADAPI.register(modname, item_node, node_def)
        end
    end
end

core.register_abm({
    nodenames = {"group:radapi_torch"},
    interval = 1.0, chance = 1,
    action = function(pos, node)
        if core.get_node({x=pos.x, y=pos.y-1, z=pos.z}).name == "air" then
            core.remove_node(pos)
            if not node.name:find("out") then core.add_item(pos, "default:torch") end
        end
    end
})

-- ==========================================
-- 8. LOAD / SAVE & UTILITY LBMS
-- ==========================================

core.register_lbm({
    name = mod_name .. ":restore_displays",
    nodenames = {"group:radapi_display"},
    run_at_every_load = true,
    action = function(pos, node)
        local meta = core.get_meta(pos)
        local stored = meta:get_string("radapi:item_full")
        if stored ~= "" then
            Utils.remove_display_entities(pos)
            local def = ITEM_DATA[node.name]
            local props = def and def._radapi_display_props or {}
            local ptype = core.get_node_def(node.name).paramtype2 or "facedir"
            local step = meta:get_int("radapi:yaw_step")
            Utils.spawn_display_entity(pos, stored, node.param2, props, ptype, step, ITEM_DATA)
        end
    end,
})

minetest.register_on_joinplayer(function(player)
    local entries = Utils.load_storage(player:get_player_name())
    if entries then
        for _, e in ipairs(entries) do RADAPI.attach_entity(player, ItemStack(e.item_data), {id = e.id}) end
    end
end)

minetest.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    if ACTIVE_REGISTRY[name] then
        for _, e in ipairs(ACTIVE_REGISTRY[name]) do
            if e.last_pos and active_lights[e.last_pos] then
                active_lights[e.last_pos][e.id] = nil
                update_light_at_pos(e.last_pos)
            end
            if e.entity then e.entity:remove() end
        end
    end
    GLOWING_TARGETS[name] = nil
    ACTIVE_REGISTRY[name] = nil
end)

minetest.register_on_dieplayer(function(player) RADAPI.drop_all_attachments(player) end)

core.register_entity("radapi:wield_entity", { initial_properties = { visual = "mesh", mesh = "blank.glb", static_save = false } })
core.register_entity("radapi:wield_entity_item", { initial_properties = { visual = "wielditem", static_save = false } })

radapi = RADAPI
radapi.v3 = {}
dofile(core.get_modpath(core.get_current_modname()) .. "/new_api.lua")
