## Table of Contents
1. [Introduction](#introduction)
2. [Installation](#installation)
3. [Core Concepts](#core-concepts)
4. [API Reference](#api-reference)
   - [ENTITY.register()](#entityregister)
   - [ENTITY.spawn()](#entityspawn)
   - [ENTITY.morph()](#entitymorph)
   - [ENTITY.unregister()](#entityunregister)
5. [Entity Lifecycle Hooks](#entity-lifecycle-hooks)
6. [Advanced Examples](#advanced-examples)
   - [State Management](#state-management)
   - [Dynamic Morphing](#dynamic-morphing)
   - [Persistent State](#persistent-state)
   - [Entity Composition](#entity-composition)
7. [Best Practices](#best-practices)
8. [Troubleshooting](#troubleshooting)

---

## Introduction

The **RADAPI v3 Entity API** provides a robust, stateful system for managing entities in Minetest. Unlike vanilla entities, this API enables:
- **State persistence** across saves
- **Dynamic morphing** between entity types
- **Modular entity definitions** with lifecycle hooks
- **Efficient state management** with deep merging
- **Safe entity unregistration** with active checks

This API is designed for mod developers who need complex entity behavior without reinventing state management.

---

## Installation

1. Add `radapi` as a dependency in your `mod.conf`:
   ```ini
   depends = radapi
   ```
2. The API is automatically available as `radapi.v3.entity` after loading `radapi`.
---

## Core Concepts

| Concept | Description |
|---------|-------------|
| **Def ID** | `modname:entity_name` (e.g., `my_mod:fireball`) |
| **State** | Persistent data stored per entity (e.g., `health`, `cooldown`) |
| **Morph** | Changing an entity's behavior without despawning |
| **Active Entity** | An entity currently in-world with its state |
| **Blueprint** | The registered entity definition table |

---

## API Reference

### ENTITY.register()
Register a new entity definition with lifecycle hooks.

```lua
ENTITY.register(modname, entity_name, {
    initial_properties = {visual = "mesh", mesh = "my_entity.b3d"},
    on_activate = function(self, staticdata, dtime, state)
        -- Called when entity spawns
    end,
    on_step = function(self, dtime, state)
        -- Called every game tick
    end,
    on_deactivate = function(self, removal, state)
        -- Called when entity is removed
    end,
    _on_morph = function(self, state)
        -- Called when morphing (internal)
    end,
    get_staticdata = function(state)
        -- Return extra data for saving
    end
})
```

**Parameters**:
- `modname`: String (your mod's name)
- `entity_name`: String (unique identifier)
- `def`: Table (entity definition)

**Example**:
```lua
-- Register a basic entity
radapi.v3.entity.register("my_mod", "fireball", {
    initial_properties = {
        visual = "mesh",
        mesh = "fireball.b3d",
        collision_box = {0.1, 0.1, 0.1}
    },
    on_activate = function(self, _, _, state)
        self.object:set_velocity({x=0, y=0, z=0})
    end
})
```

---

### ENTITY.spawn()
Spawn a new entity at a position.

```lua
local entity = ENTITY.spawn({x=0, y=0, z=0}, "my_mod:fireball", {
    speed = 10,
    damage = 5
})
```

**Parameters**:
- `pos`: Vector (position to spawn)
- `def_id`: String (registered entity ID)
- `initial_state`: Table (initial state)

**Returns**: Entity object or `nil` on failure.

---

### ENTITY.morph()
Change an entity's behavior *without* despawning.

```lua
-- Morph existing entity to a new type
ENTITY.morph(entity, "my_mod:explosion", {
    radius = 3,
    damage = 20
})
```

**Parameters**:
- `object`: Entity object
- `def_id`: String (new entity ID)
- `initial_state`: Table (state to merge)

> **Note**: Automatically merges new state with existing state.

---

### ENTITY.unregister()
Remove a registered entity definition.

```lua
-- Safe unregister (fails if active)
if ENTITY.unregister("my_mod:fireball") then
    minetest.log("success", "Entity unregistered safely")
end
```

**Parameters**:
- `def_id`: String (entity ID to remove)

**Returns**: `true` on success, `false` if active.

---

## Entity Lifecycle Hooks

| Hook | Called When | Parameters |
|------|-------------|------------|
| `on_activate` | Entity spawns | `self`, `staticdata`, `dtime`, `state` |
| `on_step` | Every game tick | `self`, `dtime`, `state` |
| `on_deactivate` | Entity removed | `self`, `removal`, `state` |
| `get_staticdata` | Saving entity | `state` → Returns extra data |

**Example with State**:
```lua
radapi.v3.entity.register("my_mod", "health_potion", {
    on_activate = function(self, _, _, state)
        self.object:set_properties({visual = "sprite", textures = {"potion.png"}})
        self.object:set_velocity({x=0, y=0.5, z=0})
    end,
    on_step = function(self, dtime, state)
        if state.health <= 0 then
            self.object:remove()
        end
    end,
    get_staticdata = function(state)
        return {health = state.health}
    end
})
```

---

## Advanced Examples

### 🔹 State Management (Simple)
```lua
-- Register entity with state
radapi.v3.entity.register("my_mod", "health_potion", {
    initial_properties = {visual = "sprite", textures = {"potion.png"}},
    on_activate = function(self, _, _, state)
        self.object:set_properties({visual = "sprite", textures = {"potion.png"}})
    end,
    on_step = function(self, dtime, state)
        state.health = (state.health or 100) - 5 * dtime
        if state.health <= 0 then
            self.object:remove()
        end
    end
})

-- Spawn with initial state
local potion = radapi.v3.entity.spawn({x=0, y=5, z=0}, "my_mod:health_potion", {
    health = 80
})
```

---

### 🔹 Dynamic Morphing (Advanced)
```lua
-- Register two entity types
radapi.v3.entity.register("my_mod", "fireball", {
    initial_properties = {visual = "mesh", mesh = "fireball.b3d"},
    on_activate = function(self, _, _, state)
        self.object:set_velocity({x=0, y=0.5, z=0})
    end
})

radapi.v3.entity.register("my_mod", "explosion", {
    initial_properties = {visual = "sprite", textures = {"explosion.png"}},
    on_activate = function(self, _, _, state)
        self.object:set_properties({visual = "sprite", textures = {"explosion.png"}})
    end
})

-- Spawn fireball
local fireball = radapi.v3.entity.spawn({x=0, y=5, z=0}, "my_mod:fireball")

-- Morph to explosion after 2 seconds
minetest.after(2, function()
    radapi.v3.entity.morph(fireball, "my_mod:explosion", {
        radius = 3,
        damage = 10
    })
end)
```

---

### 🔹 Persistent State (Save/Load)
```lua
-- Register entity with persistent state
radapi.v3.entity.register("my_mod", "magic_crate", {
    initial_properties = {visual = "mesh", mesh = "crate.b3d"},
    get_staticdata = function(state)
        return {locked = state.locked} -- Save locked state
    end,
    on_activate = function(self, staticdata, _, state)
        local data = core.deserialize(staticdata)
        if data.locked then
            state.locked = true
        end
    end
})

-- Spawn and lock
local crate = radapi.v3.entity.spawn({x=0, y=5, z=0}, "my_mod:magic_crate", {
    locked = true
})

-- When reloading world:
-- Entity automatically loads with `locked=true`
```

---

### 🔹 Entity Composition (Modular)
```lua
-- Register base entity
radapi.v3.entity.register("my_mod", "base_entity", {
    on_step = function(self, dtime, state)
        if state.speed then
            self.object:set_velocity({x=state.speed, y=0, z=0})
        end
    end
})

-- Register enhanced entity
radapi.v3.entity.register("my_mod", "flying_entity", {
    initial_properties = {visual = "mesh", mesh = "flying.b3d"},
    on_activate = function(self, _, _, state)
        state.speed = 5
    end,
    _on_morph = function(self, state)
        -- Called when morphing to this type
        state.speed = 10
    end
})

-- Spawn base entity
local entity = radapi.v3.entity.spawn({x=0, y=5, z=0}, "my_mod:base_entity")

-- Morph to flying entity (keeps speed=5, then sets new speed=10)
radapi.v3.entity.morph(entity, "my_mod:flying_entity")
```

---

## Best Practices

1. **Always unregister entities** when your mod unloads:
   ```lua
   minetest.register_on_shutdown(function()
       radapi.v3.entity.unregister("my_mod:fireball")
   end)
   ```

2. **Use `initial_state` for defaults** instead of hardcoding:
   ```lua
   -- Good
   ENTITY.spawn(pos, "my_mod:fireball", {damage = 5})
   
   -- Avoid
   ENTITY.spawn(pos, "my_mod:fireball") -- Hardcoded damage
   ```

3. **Merge state safely** with `deep_merge` (handled internally):
   ```lua
   -- Inside your on_step hook:
   state.cooldown = (state.cooldown or 0) - dtime
   if state.cooldown <= 0 then
       -- Attack logic
   end
   ```

4. **Never modify `state` directly** - use `state` as read-only in hooks.

---

## Troubleshooting

| Symptom | Solution |
|---------|----------|
| `Entity has no GUID` | Ensure entity is spawned via `ENTITY.spawn()` (not `core.add_entity()`) |
| `Unknown def_id` | Verify `ENTITY.register()` was called before spawning |
| `Entity not persisting` | Check `get_staticdata` returns table with all state keys |
| `Morph fails silently` | Confirm entity is active (`is_active(def_id) == false`) |
| `State not saved` | Ensure `get_staticdata` returns table, not string |

> **Debug Tip**: Use `radapi.v3.entity.is_active("my_mod:entity")` to check if entities are active.


