local ENTITY = {}
local REGISTERED_ENTITIES = {}

local ACTIVE_ENTITIES = {}

local function get_entity_info(object)
    local guid = object:get_guid()
    return guid and ACTIVE_ENTITIES[guid]
end

local function deep_merge(target, source)
    for k, v in pairs(source) do
        if type(v) == "table" and type(target[k]) == "table" then
            deep_merge(target[k], v)
        else
            target[k] = v
        end
    end
end

local function is_active(def_id)
    for _, info in pairs(ACTIVE_ENTITIES) do
        if info.name == def_id then
            return true
        end
    end
    return false
end

function ENTITY.register(modname, name, def)
    assert(type(def) == "table", "Entity definition must be a table")
    assert(modname and name, "Missing modname or entity name")

    local def_id = modname .. ":" .. name
    assert(not REGISTERED_ENTITIES[def_id], "Entity already registered: " .. def_id)

    REGISTERED_ENTITIES[def_id] = def
end

local function morph(object, def_id, initial_state)
    local guid = object:get_guid()
    local blueprint = REGISTERED_ENTITIES[def_id]

    if not guid then
        core.log("error", "[API] Entity has no GUID when morphing")
        return
    end

    if not blueprint then
        core.log("error", "[API] Unknown def_id: " .. tostring(def_id))
        return
    end

    local current_info = get_entity_info(object)

    if current_info and current_info.name == def_id then
        if initial_state then
            deep_merge(current_info.state, initial_state)
            if blueprint._on_morph then
                blueprint._on_morph(object, current_info.state)
            end
        end
        return
    end

    local state = {}
    if initial_state then
        deep_merge(state, initial_state)
    end

    ACTIVE_ENTITIES[guid] = {
        name = def_id,
        state = state,
        def = blueprint
    }

    if blueprint.initial_properties then
        object:set_properties(blueprint.initial_properties)
    end

    if blueprint._on_morph then
        blueprint._on_morph(object, state)
    end
end

core.register_entity("radapi:blank_entity", {
    initial_properties = {
        visual = "sprite",
        textures = {"blank.png"},
        static_save = true,
    },

    on_activate = function(self, staticdata, dtime_s)
        local guid = self.object:get_guid()
        if not guid then return end

        local saved_data = staticdata ~= "" and core.deserialize(staticdata) or {}

        if saved_data.name and not get_entity_info(self.object) then
            local blueprint = REGISTERED_ENTITIES[saved_data.name]
            if blueprint then
                ACTIVE_ENTITIES[guid] = {
                    name = saved_data.name,
                    state = saved_data.state or {},
                    def = blueprint
                }

                self.object:set_properties(blueprint.initial_properties)
            end
        end

        local info = get_entity_info(self.object)
        if info and info.def and info.def.on_activate then
            info.def.on_activate(self, staticdata, dtime_s, info.state)
        end
    end,

    on_step = function(self, dtime)
        local info = get_entity_info(self.object)
        if info and info.def and info.def.on_step then
            info.def.on_step(self, dtime, info.state)
        end
    end,

    on_deactivate = function(self, removal)
        local info = get_entity_info(self.object)
        if info and info.def and info.def.on_deactivate then
            info.def.on_deactivate(self, removal, info.state)
        end

        if removal then
            local guid = self.object:get_guid()
            if guid then
                ACTIVE_ENTITIES[guid] = nil
            end
        end
    end,

    get_staticdata = function(self)
        local info = get_entity_info(self.object)
        if not info then return "" end

        local data = {
            name = info.name,
            state = info.state
        }

        if info.def and info.def.get_staticdata then
            local extra = info.def.get_staticdata(info.state)
            if type(extra) == "table" then
                for k, v in pairs(extra) do
                    data[k] = v
                end
            end
        end

        return core.serialize(data)
    end
})

function ENTITY.spawn(pos, def_id, initial_state)
    assert(type(pos) == "table", "Position must be a table (vector)")
    assert(REGISTERED_ENTITIES[def_id], "Entity definition not registered: " .. def_id)
    
    local entity_name = "radapi:blank_entity"
    local obj = core.add_entity(pos, entity_name)

    if not obj then
        core.log("error", "[API] Failed to spawn entity at " .. minetest.pos_to_string(pos))
        return
    end

    morph(obj, def_id, initial_state)
    return obj
end

function ENTITY.unregister(def_id)
    assert(type(def_id) == "string", "def_id must be a string")
    
    if not REGISTERED_ENTITIES[def_id] then
        core.log("error", "[API] Entity definition not registered: " .. def_id)
        return false
    end

    if is_active(def_id) then
        core.log("error", "[API] Cannot unregister entity definition '" .. def_id .. "' because it's currently in use")
        return false
    end

    REGISTERED_ENTITIES[def_id] = nil
    core.log("info", "[API] Unregistered entity definition: " .. def_id)
    return true
end

radapi.v3.entity = ENTITY
