## Table of Contents
1. [Introduction](#introduction)
2. [Installation](#installation)
3. [Core Concepts](#core-concepts)
4. [API Reference](#api-reference)
   - [PEDESTAL.register()](#pedestalregister)
   - [PEDESTAL.unregister()](#pedestalunregister)
5. [Pedestal Node Behavior](#pedestal-node-behavior)
6. [Advanced Examples](#advanced-examples)
   - [Basic Item Display](#basic-item-display)
   - [Custom Rotation Logic](#custom-rotation-logic)
   - [Item Removal with Custom Logic](#item-removal-with-custom-logic)
   - [Multi-Item Pedestals](#multi-item-pedestals)
7. [Best Practices](#best-practices)
8. [Troubleshooting](#troubleshooting)

---

## Introduction

The **RADAPI v3 Pedestal API** provides a complete system for creating interactive pedestal nodes that display items as entities. Unlike vanilla pedestals, this API enables:

- **Persistent item display** with proper entity management
- **Customizable interaction logic** (rotation, removal, display)
- **Automatic entity synchronization** with node metadata
- **Safe node unregistration** with active checks
- **Modular extension points** for all interaction hooks

This API is designed for mod developers who need to create item display pedestals with rich interaction capabilities.

---

## Installation

1. Add `radapi` as a dependency in your `mod.conf`:
   ```ini
   depends = radapi
   ```
2. The API is automatically available as `radapi.v3.pedestal` after loading `radapi`.

> **Note**: Requires RADAPI v3 or higher. Install via [Minetest Game](https://github.com/minetest/minetest_game) or [Mod Browser](https://mod.minetest.net).

---

## Core Concepts

| Concept | Description |
|---------|-------------|
| **Pedestal Node** | A node that serves as a base for item display (e.g., `my_mod:wood_pedestal`) |
| **Entity** | `radapi:pedestal_item` entity that renders the item (automatically spawned) |
| **Metadata** | Node metadata storing `has_item` and `entity_guid` |
| **Def ID** | `modname:pedestal_name` (e.g., `my_mod:wood_pedestal`) |
| **State** | Persistent data stored in node metadata (not directly used by API) |

---

## API Reference

### PEDESTAL.register()
Register a new pedestal node with custom behavior.

```lua
PEDESTAL.register(modname, pedestal_name, {
    entity_name = "radapi:custom_pedestal_item", -- Optional (default: "radapi:pedestal_item")
    on_display = function(pos, itemstack, placer, node, entity)
        -- Called when item is placed on pedestal
    end,
    on_remove = function(pos, itemstack, clicker, node)
        -- Called when item is removed (right-click with empty hand)
        -- Return modified itemstack to replace original
    end,
    on_rotate = function(pos, itemstack, clicker, node, entity)
        -- Called when rotating (right-click with item)
    end,
    on_construct = function(pos)
        -- Called when pedestal is placed
    end,
    on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
        -- Custom right-click handler (optional)
    end,
    after_destruct = function(pos, oldnode)
        -- Called when pedestal is destroyed
    end
})
```

**Parameters**:
- `modname`: String (your mod's name)
- `pedestal_name`: String (unique identifier for node)
- `def`: Table (pedestal definition with optional hooks)

> **Note**: All hooks are optional. The API handles entity creation and metadata automatically.

---

### PEDESTAL.unregister()
Remove a registered pedestal definition.

```lua
if PEDESTAL.unregister("my_mod:wood_pedestal") then
    minetest.log("success", "Pedestal unregistered safely")
end
```

**Parameters**:
- `def_id`: String (pedestal ID to remove)

**Returns**: `true` on success, `false` if active pedestals exist.

---

## Pedestal Node Behavior

When a player places an item on a pedestal:
1. The pedestal node checks for existing items
2. If empty, spawns `radapi:pedestal_item` entity at the pedestal position
3. Sets node metadata:
   - `has_item = "true"`
   - `entity_guid = [entity guid]`
4. Calls `on_display` hook

When player right-clicks:
- **Empty hand**: Removes item (calls `on_remove`)
- **Item in hand**: Rotates item (calls `on_rotate`)

---

## Advanced Examples

### 🔹 Basic Item Display (Simple)
```lua
-- Register a basic pedestal
radapi.v3.pedestal.register("my_mod", "wood_pedestal", {
    on_display = function(pos, itemstack, placer)
        minetest.chat_send_player(placer:get_player_name(), "Item displayed on pedestal!")
    end
})

-- Place item on pedestal
-- (Player places item on top of pedestal node)
-- Entity automatically appears with item texture
```

---

### 🔹 Custom Rotation Logic (Advanced)
```lua
radapi.v3.pedestal.register("my_mod", "rotating_pedestal", {
    on_rotate = function(pos, itemstack, clicker, node, entity)
        -- Rotate item 45 degrees on each right-click
        local rotation = entity:get_rotation()
        rotation.y = rotation.y + math.rad(45)
        entity:set_rotation(rotation)
        
        minetest.chat_send_player(clicker:get_player_name(), "Item rotated!")
    end
})

-- Usage:
-- Player right-clicks pedestal with item in hand
-- Item rotates 45 degrees each click
```

---

### 🔹 Item Removal with Custom Logic
```lua
radapi.v3.pedestal.register("my_mod", "rechargeable_pedestal", {
    on_remove = function(pos, itemstack, clicker, node)
        -- If item is a battery, return 50% charged
        if itemstack:get_name() == "my_mod:battery" then
            local battery = ItemStack("my_mod:battery:50")
            minetest.chat_send_player(clicker:get_player_name(), "Battery recharged!")
            return battery
        end
        return itemstack -- Default behavior
    end
})

-- Usage:
-- Player right-clicks with empty hand
-- If item is battery, returns 50% charged item
-- Otherwise returns original item
```

---

### 🔹 Multi-Item Pedestals (Composite)
```lua
radapi.v3.pedestal.register("my_mod", "multi_pedestal", {
    entity_name = "radapi:multi_pedestal_item", -- Custom entity
    
    on_display = function(pos, itemstack, placer, node, entity)
        -- Track multiple items
        local meta = core.get_meta(pos)
        local count = tonumber(meta:get_string("item_count") or "0") + 1
        meta:set_string("item_count", tostring(count))
        
        minetest.chat_send_player(placer:get_player_name(), "Added item #" .. count)
    end,
    
    on_remove = function(pos, itemstack, clicker, node)
        local meta = core.get_meta(pos)
        local count = tonumber(meta:get_string("item_count") or "0") - 1
        meta:set_string("item_count", tostring(count))
        
        if count <= 0 then
            meta:set_string("has_item", "")
        end
        
        return itemstack
    end
})

-- Register custom entity (radapi:multi_pedestal_item)
radapi.v3.entity.register("my_mod", "multi_pedestal_item", {
    on_activate = function(self, _, _, state)
        self.object:set_properties({
            visual = "wielditem",
            wield_item = state.itemstack.name,
            textures = {state.itemstack.name}
        })
    end,
    -- ... other hooks
})
```

---

## Best Practices

1. **Always unregister pedestals** on mod unload:
   ```lua
   minetest.register_on_shutdown(function()
       radapi.v3.pedestal.unregister("my_mod:wood_pedestal")
   end)
   ```

2. **Use `on_remove` for item transformation**:
   ```lua
   -- Never just remove item - transform it
   on_remove = function(pos, itemstack, clicker)
       if itemstack:get_name() == "default:stone" then
           return ItemStack("default:cobble")
       end
       return itemstack
   end
   ```

3. **Avoid direct entity manipulation** in hooks:
   ```lua
   -- Good
   entity:set_rotation(rotation)
   
   -- Bad (direct entity access)
   core.get_object_by_guid(guid):set_rotation(rotation)
   ```

4. **Use metadata for persistent state**:
   ```lua
   -- In on_construct
   meta:set_string("item_count", "0")
   
   -- In on_display
   local count = tonumber(meta:get_string("item_count")) + 1
   meta:set_string("item_count", tostring(count))
   ```

---

## Troubleshooting

| Symptom | Solution |
|---------|----------|
| Item doesn't display | Verify pedestal node is registered with `PEDESTAL.register()` |
| Entity disappears after save | Check `on_deactivate` in entity definition for cleanup |
| Rotation doesn't work | Ensure `on_rotate` hook is properly implemented |
| Item not removable | Verify `on_remove` returns a valid itemstack |
| "Entity has no GUID" error | Confirm pedestal node has `is_pedestal = true` in node definition |

> **Debug Tip**: Use `core.get_meta(pos):to_table()` to inspect pedestal metadata.

---

## Conclusion

The RADAPI v3 Pedestal API provides a **production-ready foundation** for creating interactive item display systems. By leveraging:
- Automatic entity management
- Modular hook system
- Safe metadata synchronization
- Customizable interaction logic

You can create:
- Item display pedestals with rotation
- Rechargeable item stands
- Multi-item display systems
- Custom item transformation stations

**Start using it today** to build pedestals that feel alive and interactive!

---

> **Note**: The `radapi:pedestal_item` entity is automatically managed by the API. You don't need to handle entity creation - just register your pedestal and define the hooks. The API handles all entity synchronization with node metadata.