-- Définir la table gameripvault si elle n'existe pas
-- tout le texte sur le jeux est en anglais
gameripvault = {}

-- Fonction pour lire le contenu d'un fichier JSON
local function read_json_file(path)
    local file = io.open(path, "r")
    if not file then
        return nil, "the JSON file could not be opened for reading."
    end
    local content = file:read("*a")
    file:close()
    return minetest.parse_json(content) or {}, nil
end

-- Fonction pour écrire des données dans un fichier JSON
local function write_json_file(path, data)
    local file = io.open(path, "w")
    if not file then
        return false, "Imposible to open the JSON file for writing."
    end
    local json = minetest.write_json(data, true) -- Le `true` formate le JSON pour plus de lisibilité
    file:write(json)
    file:close()
    return true, nil
end

-- Fonction pour vérifier si une table contient une valeur
function gameripvault.table_contains(tbl, value)
    for _, v in ipairs(tbl) do
        if v == value then
            return true
        end
    end
    return false
end

-- Déclaration de la fonction
function gameripvault.get_player_ip_hash(player_name)
    -- Récupérer l'IP du joueur
    local player_ip = minetest.get_player_ip(player_name)

    -- Vérifier que player_ip n'est pas nil
    if not player_ip then
        return nil, "IP not found for player '" .. player_name .. "'"
    end

    -- Hasher l'IP du joueur
    local player_ip_hash = minetest.sha1(player_ip)

    -- Renvoyer le hash de l'IP du joueur
    return player_ip_hash
end

--fonction qui recupere un speudo et renvoie une liste de joueur ayant des ip en commun
-- il va deja recupere la liste des ip hashé du joueur
-- puis il va chercher dans le fichier json les joeurs ayant les meme ip hashé
-- et les renvoie
function gameripvault.get_players_with_same_ip(player_name)
    -- Récupérer le chemin du fichier JSON
    local path = minetest.get_worldpath() .. "/gameripvault.json"

    -- Lire le contenu du fichier JSON
    local data, err = read_json_file(path)
    if err then
        return nil, err
    end

    -- Récupérer l'IP hashée du joueur
    local player_ip_hash, err = gameripvault.get_player_ip_hash(player_name)
    if not player_ip_hash then
        return nil, err
    end

    -- Récupérer les joueurs ayant la même IP hashée
    local players = {}
    for ip_hash, player_list in pairs(data) do
        if ip_hash == player_ip_hash then
            for _, player in ipairs(player_list) do
                table.insert(players, player)
            end
        end
    end

    return players
end




minetest.register_on_joinplayer(function(player)
    -- Récupérer le nom du joueur
    local player_name = player:get_player_name()

    -- Récupérer l'IP hashée du joueur
    local player_ip_hash, err = gameripvault.get_player_ip_hash(player_name)
    if not player_ip_hash then
        minetest.log("error", err)
        return
    end

    -- Consigner l'événement dans les logs
    minetest.log("action", "[GamerIPVault] " .. player_name .. " joined the game with IP hash: " .. player_ip_hash)

    -- Sauvegarder les IP hashées dans le fichier JSON
    gameripvault.save_ip_hashed(player_name, player_ip_hash)
end)

function gameripvault.save_ip_hashed(player_name, player_ip_hash)
    -- Vérifier si le mod est actif
    if not minetest.get_modpath("gameripvault") then
        minetest.log("error", "The GamerIPVault mod is not active.")
        return
    end

    -- Récupérer le chemin du fichier JSON
    local path = minetest.get_worldpath() .. "/gameripvault.json"

    -- Lire le contenu du fichier JSON
    local data, err = read_json_file(path)
    if err then
        -- Créer un nouveau fichier JSON avec une table vide si la lecture échoue
        local success, write_err = write_json_file(path, {})
        if not success then
            minetest.log("error", write_err)
            return
        end
        data = {}
    end

    -- Ajouter ou mettre à jour les données
    if not data[player_ip_hash] then
        data[player_ip_hash] = {}
    end

    if not gameripvault.table_contains(data[player_ip_hash], player_name) then
        table.insert(data[player_ip_hash], player_name)
    end

    -- Écrire les données mises à jour dans le fichier JSON
    local success, write_err = write_json_file(path, data)
    if not success then
        minetest.log("error", write_err)
    end
end

-- Fonction pour créer un message pour les commandes de chat
local function create_message_for_chat(name, param, data, key, is_players)
    local results = {}

    for hash, players in pairs(data) do
        if (key == hash and is_players) or (key == players and not is_players) then
            for _, player in ipairs(players) do
                table.insert(results, player)
            end
        end
    end

    if #results == 0 then
        minetest.chat_send_player(name, "no results found for '" .. param .. "'")
        return
    end

    local message = "Results found for '" .. param .. "':"
    for _, result in ipairs(results) do
        message = message .. "\n" .. result
    end

    minetest.chat_send_player(name, message)
    
    local formspec = "size[8,3]textarea[0.5,0.5;7.5,2.5;message;;" .. message .. "]button_exit[3,2.5;2,1;exit;exit]"
    minetest.show_formspec(name, "gameripvault:result", formspec)
end

-- Fonction qui récupère un pseudo et renvoie la liste des IP où le joueur s'est connecté
function gameripvault.get_ip(player_name)
    -- Récupérer le chemin du fichier JSON
    local path = minetest.get_worldpath() .. "/gameripvault.json"

    -- Lire le contenu du fichier JSON
    local data, err = read_json_file(path)
    if err then
        return nil, "Error reading JSON file : " .. err
    end

    -- Valider les données lues
    if type(data) ~= "table" then
        return nil, "Invalid JSON data"
    end

    -- Récupérer les IP où le joueur s'est connecté
    local ips = {}
    for ip_hash, player_list in pairs(data) do
        if type(player_list) == "table" then
            for _, player in ipairs(player_list) do
                if player == player_name then
                    table.insert(ips, ip_hash)
                end
            end
        end
    end

    return ips
end

-- Fonction pour récupérer les joueurs ayant la même IP hashée
function gameripvault.get_players_with_same_ip_hash(ip_hash)
    -- Récupérer le chemin du fichier JSON
    local path = minetest.get_worldpath() .. "/gameripvault.json"

    -- Lire le contenu du fichier JSON
    local data, err = read_json_file(path)
    if not data then
        return nil, err
    end

    -- Récupérer les joueurs ayant la même IP hashée
    local players = {}
    for hash, player_list in pairs(data) do -- Parcourir les données
        if hash == ip_hash then -- Vérifier si l'IP hashée correspond
            for _, player in ipairs(player_list) do -- Parcourir la liste des joueurs
                table.insert(players, player) -- Ajouter le joueur à la liste
            end
        end
    end

    return players
end


-- Fonction pour trouver les joueurs ayant utilisé les mêmes IP que le joueur donné
function gameripvault.find_players_with_same_ips(player_name)
    -- Récupérer le chemin du fichier JSON
    local path = minetest.get_worldpath() .. "/gameripvault.json"

    -- Lire le contenu du fichier JSON
    local data, err = read_json_file(path)
    if err then
        return nil, "Error reading JSON file : " .. err
    end

    -- Obtenir les IP hashées associées au joueur
    local ips, err = gameripvault.get_ip(player_name)
    if err then
        return nil, err
    end

    -- Trouver tous les joueurs ayant utilisé ces IP
    local players_with_same_ips = {}
    local found_players = {}

    for _, ip_hash in ipairs(ips) do
        local players, err = gameripvault.get_players_with_same_ip_hash(ip_hash)
        if err then
            return nil, err
        end
        
        for _, player in ipairs(players) do
            if not found_players[player] then
                found_players[player] = true
                table.insert(players_with_same_ips, player)
            end
        end
    end

    --suprime le speudo player_name de la liste
    for i, player in ipairs(players_with_same_ips) do
        if player == player_name then
            table.remove(players_with_same_ips, i)
            break
        end
    end

    return players_with_same_ips
end







--declare la permission gameripvault pour les commandes
minetest.register_privilege("gameripvault", {
    description = "permit to use the gameripvault commands",
    give_to_singleplayer = false
})


--commande qui utilise get_ip pour afficher les ip ou le joueur s'est connecté
minetest.register_chatcommand("getip", {
    params = "<pseudo>",
    description = "find the IP where the player has connected",
    privs = {gameripvault = true},
    func = function(name, param)
        local players, err = gameripvault.get_ip(param)
        if err then
            minetest.chat_send_player(name, err)
            return
        end

        if #players == 0 then
            minetest.chat_send_player(name, "No IP found for '" .. param .. "'")
            return
        end

        local message = "IP found for '" .. param .. "':"
        for _, player in ipairs(players) do
            message = message .. "\n" .. player
        end

        minetest.chat_send_player(name, message)
        
        local formspec = "size[8,3]textarea[0.5,0.5;7.5,2.5;message;;" .. message .. "]button_exit[3,2.5;2,1;exit;exit]"
        minetest.show_formspec(name, "gameripvault:ip", formspec)
    end
})

-- commande qui utilise get_players_with_same_ip_hash pour afficher les joueurs ayant la meme ip
minetest.register_chatcommand("getplayers", {
    params = "<hash>",
    description = "Find players with the same IP hash",
    privs = {gameripvault = true},
    func = function(name, param)
        -- Vérifier que le paramètre est non vide
        if param == "" then
            minetest.chat_send_player(name, "Please provide an IP hash.")
            return
        end

        -- Récupérer les joueurs ayant la même IP hashée
        local players, err = gameripvault.get_players_with_same_ip_hash(param)
        if err then
            minetest.chat_send_player(name, "Error: " .. err)
            return
        end

        -- convrtir la liste en chaine de caractere
        local message = "Players with the same IP hash as '" .. param .. "':"
        for _, player in ipairs(players) do
            message = message .. "\n" .. player
        end

        -- Envoyer le message au joueur
        minetest.chat_send_player(name, message)

        -- utiliser la fonction show_formspec pour afficher le message
        local formspec = "size[8,3]textarea[0.5,0.5;7.5,2.5;message;;" .. message .. "]button_exit[3,2.5;2,1;exit;exit]"
        minetest.show_formspec(name, "gameripvault:players", formspec)
    end
})

-- Commande pour trouver les joueurs ayant utilisé les mêmes IP qu'un joueur donné
minetest.register_chatcommand("findsameips", {
    params = "<pseudo>",
    description = "Find players who have used the same IPs as the specified player",
    privs = {gameripvault = true},
    func = function(name, param)
        local players, err = gameripvault.find_players_with_same_ips(param)
        if err then
            minetest.chat_send_player(name, err)
            return
        end

        if #players == 0 then
            minetest.chat_send_player(name, "No players found with the same IPs as '" .. param .. "'")
            return
        end

        local message = "Players who have used the same IPs as '" .. param .. "':"
        for _, player in ipairs(players) do
            message = message .. "\n" .. player
        end

        minetest.chat_send_player(name, message)
        
        local formspec = "size[8,3]textarea[0.5,0.5;7.5,2.5;message;;" .. message .. "]button_exit[3,2.5;2,1;exit;exit]"
        minetest.show_formspec(name, "gameripvault:same_ips", formspec)
    end
})





