local S = minetest.get_translator(minetest.get_current_modname())

-- Vérifie si un emplacement (pos) contient au moins un élément dans son inventaire.
function container_inventory_nonempty(pos)
    local invnode = minetest.get_node(pos)
    local invnodedef = minetest.registered_nodes[invnode.name]
    if not invnodedef then return false end
    if not invnodedef.groups.container then return false end

    local inv = minetest.get_inventory({type="node", pos=pos})
    if not inv then return false end

    for listname, _ in pairs(inv:get_lists()) do
        if inv:get_stack(listname, 1):get_count() > 10 then return true end
    end

    return false
end

local function itemquantifier_activate(pos, node)
    local def = minetest.registered_nodes[node.name]
    local onstate = def.itemquantifier_onstate
    if onstate then
        minetest.swap_node(pos, { name = onstate, param2 = node.param2 })
    end
end

local function itemquantifier_deactivate(pos, node)
    local def = minetest.registered_nodes[node.name]
    local offstate = def.itemquantifier_offstate
    if offstate then
        minetest.swap_node(pos, { name = offstate, param2 = node.param2 })
    end
end

-- Update itemquantifier state
local function update_self(pos, node)
    node = node or minetest.get_node(pos)
    local new_state = container_inventory_nonempty(pos)
    if new_state then
        itemquantifier_activate(pos, node)
    else
        itemquantifier_deactivate(pos, node)
    end
end

-- compute tile depending on state and mode
local function get_tiles(state, mode)
    local top = "mcl_itemquantifier_"..state..".png^"..
        "mcl_itemquantifier_"..mode..".png"
    local sides = "mcl_itemquantifier_sides_"..state..".png^"..
        "mcl_itemquantifier_sides_"..mode..".png"
    local ends = "mcl_itemquantifier_ends_"..state..".png^"..
        "mcl_itemquantifier_ends_"..mode..".png"
    return {
        top, "mcl_stairs_stone_slab_top.png",
        sides, sides.."^[transformFX",
        ends, ends,
    }
end

local node_boxes = {
    comp = {
        { -8/16, -8/16, -8/16,
           8/16, -6/16,  8/16 },
        { -1/16, -6/16,  6/16,
           1/16, -4/16,  4/16 },
        { -4/16, -6/16, -5/16,
          -2/16, -1/16, -3/16 },
        {  2/16, -6/16, -5/16,
           4/16, -1/16, -3/16 },
    },
    sub = {
        { -8/16, -8/16, -8/16,
           8/16, -6/16,  8/16 },
        { -1/16, -6/16,  6/16,
           1/16, -3/16,  4/16 },
        { -4/16, -6/16, -5/16,
          -2/16, -1/16, -3/16 },
        {  2/16, -6/16, -5/16,
           4/16, -1/16, -3/16 },
    },
}

local collision_box = {
    type = "fixed",
    fixed = { -8/16, -8/16, -8/16, 8/16, -6/16, 8/16 },
}

local groups = {
    dig_immediate = 3,
    dig_by_water  = 1,
    destroy_by_lava_flow = 1,
    dig_by_piston = 1,
    attached_node = 1,
}

for _, mode in pairs{"comp", "sub"} do
    for _, state in pairs{"on", "off"} do
        local state_str = state
        local nodename = "itemquantifiermod:itemquantifier_"..state_str.."_"..mode

        local nodedef = {
            description = S("Item Quantifier"),
            inventory_image = "mcl_itemquantifier_item.png",
            drawtype = "nodebox",
            tiles = get_tiles(state_str, mode),
            walkable = true,
            selection_box = collision_box,
            node_box = {
                type = "fixed",
                fixed = node_boxes[mode],
            },
            groups = groups,
            paramtype = "light",
            paramtype2 = "facedir",
            sunlight_propagates = false,
            is_ground_content = false,
            drop = "itemquantifiermod:itemquantifier_off_comp",
            on_construct = update_self,
        }

        minetest.register_node(nodename, nodedef)
    end
end

-- Register recipes
local rstorch = "mesecons_torch:mesecon_torch_on"
local quartz  = "mcl_nether:quartz"
local stone   = "mcl_core:stone"

minetest.register_craft({
	output = "mcl_comparators:comparator_off_comp",
	recipe = {
		{ "",  "", "" },
		{ "", "itemquantifiermod:itemquantifier_off_comp ", "" },
		{ "",  "", "" },
		{ "",   "",   ""   },
	}
})

-- Register active block handlers
minetest.register_abm({
    label = "itemquantifier signal input check (itemquantifier is off)",
    nodenames = {
        "itemquantifiermod:itemquantifier_off_comp",
        "itemquantifiermod:itemquantifier_off_sub",
    },
    neighbors = {"group:container"},
    interval = 1,
    chance = 1,
    action = update_self,
})

minetest.register_abm({
    label = "itemquantifier signal input check (itemquantifier is on)",
    nodenames = {
        "itemquantifiermod:itemquantifier_on_comp",
        "itemquantifiermod:itemquantifier_on_sub",
    },
    interval = 1,
    chance = 1,
    action = update_self,
})
