# Internal Data Representation

`warn_system` stores warnings in a key-value pair `warn_system.warns` where they key is the player name
and the value is list of `warn_data` tables

`warn_data` is a table with the following format:

```lua
{
    read = true,
    canceled = false,
    reason = "reason",
    date = "2024-07-31 19:31:28",
    handled = true 
}

```

Here is what each field means:
- `read` is a boolean value that indicates whether the user read the warning
- `canceled` is a boolean value that indicates whether the warning is cancelled, 
cancelled warnings don't count towards player's warnings but can still be viewed by moderators
- `reason` is a string explaining the reason for the warning
- `date` is a string that shows the date when the string occurred
- `handled` is a boolean value that indicates whether callback functions were
called on the warn_data

Generally you shouldn't modify `warn_system.warns` directly, instead you should use
the provided helper functions (see below) that handle edge cases and saving to mod storage

Due to deleting warnings, the list of `warn_data` tables isn't guaranteed to be an
array like table since deleting warnings can create gaps

# API


## `warn_system.give_warn(player_name, reason)`

gives a warning to a player

### Parameters

- `player_name` - string - name of player
- `reason` - string - warn reason

### Return value

`nil`


## `warns_system.get_num_warns(player_name)`

returns how much warning a player has,
dosent count cancelled warnings

### Parameters

- `player_name` - string - name of player

### Return value

`warning_amount` - number - number of warnings


## `warn_system.register_on_warning(function)`

registers a callback function to be called whenever
any player gets a warning

### Parameters

- `func` - function - callback function, takes arguments: \
         - `player_name` - name of player who got the warning \
         - `warn_data` - info about the warning, see `Internal Data Representation` \
         - `warn_num` - number of the warning \

### Return value

`nil`


## `warn_system.cancel_warn(player_name, warn_num)`

Cancels a warn, cancelling means that the warnings dosent count 
but can still be seen by moderators and easily reactivated

See `warn_system.reactivate_warn`

### Parameters

- `player_name` - string - name of player
- `warn_num` - warning number

### Return value

`nil`


## `warn_system.reactivate_warn(player_name, warn_num)`

Reactivates a warning that was previously cancelled

See `warn_system.cancel_warn`

### Parameters

- `player_name` - string - name of player
- `warn_num` - warning number

### Return value

`nil`


## `warn_system.read_warn(player_name, warn_num)`

Marks the warning as "read" 

### Parameters

- `player_name` - string - name of player
- `warn_num` - warning number

### Return value

`nil`


## `warn_system.unread_warn(player_name, warn_num)`

Removes the `read` flag from the warning

### Parameters

- `player_name` - string - name of player
- `warn_num` - warning number

### Return value

`nil`


## `warn_system.delete_warn(player_name, warn_num)`

Removes a warning completely, this means it wont show up
for moderators and cant be reactivated

See `warn_system.cancel_warn`

### Parameters

- `player_name` - string - name of player
- `warn_num` - warning number

### Return value

`nil`


## `warns_system.show_warn_formspec(player_name, warn_num)`

Shows a warning formspec to the player, you should generally you shouldn't use this

### Parameters

- `player_name` - string - name of player
- `warn_num` - warning number

### Return value

`nil`


## `warn_system.run_callbacks(player_name, warn_data, warn_num)`

calls registered callbacks, `warn_system.give_warn` calls this already so generally 
don't need to use this

see `warn_system.register_on_warning`

### Parameters

- `player_name` - string - name who got the warning
- `warn_data` - info about the warning, see `Internal Data Representation`
- `warn_num` - warning number

### Return value

`nil`


## `warn_system.load_warn_database()`

loads warnings from mod storage, generally you shouldn't have to touch this

### Return value

`nil`


## `warn_system.save_warns()`

Saves warnings to mod storage, generally you shouldn't have to use this
as long as you use `warn_system` methods instead of modifying `warn_system.warns` directly.
However when changing a lot of warnings at once, its faster if you directly make all changes in the table
and save warnings afterwards

### Return value

`nil`
