--- Process functions for orgs

function modpol.orgs:call_module(module_slug, initiator, config, result) 
    if not modpol.modules[module_slug] then
        modpol.ocutil.log('Error in ' .. self.name .. ':call_module -> module "' .. module_slug .. '" not found')
        return
    end

    local empty_index = nil
    -- linear search for empty process slots (lazy deletion)
    for k, v in ipairs(self.processes) do
        if v == 'deleted' then
            empty_index = k
            break
        end
    end

    local index
    -- attempts to fill empty spots in list, otherwise appends to end
    if empty_index then
        index = empty_index
    else
        index = #self.processes + 1
    end

    local module = modpol.modules[module_slug]

    -- sets default values for undeclared config variables
    if modpol.util.num_pairs(module.config) > 0 and config then
        for k, v in pairs(module.config) do
            if config[k] == nil then
                config[k] = v
            end
        end
    end


    -- setting default params
    local new_process = {
        metatable = {__index = module},
        initiator = initiator,
        org = self,
        id = index,
        config = config,
        data = module.data,
        slug = module_slug
    }

    setmetatable(new_process, new_process.metatable)

    self.processes[index] = new_process
    self.processes[index]:initiate(result)
    local msg = "Initiating "..module_slug..
      " process id "..index.." in org "..self.name

    return index
end

function modpol.orgs:delete_process(id)
    local msg = "Deleting "..self.processes[id].slug..
      " process id "..id.." in org "..self.name
    self:record(msg, self.processes[id].slug)
    self:wipe_pending_actions(id)
    self.processes[id] = 'deleted'
end

function modpol.orgs:add_pending_action(process_id, user, callback) 
    self.pending[user] = self.pending[user] or {}
    self.pending[user][process_id] = callback
    modpol.interactions.message(
       user, "New pending action in org "..self.name)
end

function modpol.orgs:remove_pending_action(process_id, user) 
    if self.pending[user] then
        self.pending[user][process_id] = nil
    end
end

function modpol.orgs:wipe_pending_actions(process_id)
    for user in pairs(self.pending) do
        self.pending[user][process_id] = nil
    end
end

function modpol.orgs:has_pending_actions(user)
    -- next() will return the next pair in a table
    -- if next() returns nil, the table is empty
    if not self.pending[user] then
        return false
    else
        if not next(self.pending[user]) then
            return false
        else 
            return true
        end
    end
end

function modpol.orgs:interact(process_id, user)
   local process = self.processes[process_id]
   if self.pending[user] then
      local callback = self.pending[user][process_id]
      if callback then
         -- get data in case callback ends process
         local slug = self.processes[process_id].slug
         -- run callback
         process[callback](process, user)
         -- record org data
         local msg = "Updating "..slug..
            " process id "..process_id.." in org "..self.name
         self:record(msg, slug)
      end
   end
end
