--- Process functions for orgs

function modpol.orgs:call_module(module_slug, initiator, config, result, parent_id) 
    if not modpol.modules[module_slug] then
        modpol.ocutil.log('Error in ' .. self.name .. ':call_module -> module "' .. module_slug .. '" not found')
        return
    end

    local index = #self.processes + 1

    local module = modpol.modules[module_slug]

    -- sets default values for undeclared config variables
    if modpol.util.num_pairs(module.config) > 0 and config then
        for k, v in pairs(module.config) do
            if config[k] == nil then
                config[k] = v
            end
        end
    end

    -- setting default params
    local new_process = {
        metatable = {__index = module},
        initiator = initiator,
        org = self,
        id = index,
        parent_id = parent_id,
        children = {},
        config = config,
        data = modpol.util.copy_table(module.data),
        slug = module_slug
    }

    -- call module wrapper for modules, passes its id to child process when called
    function new_process:call_module(module_slug, initiator, config, result)
        local child_id = self.org:call_module(module_slug, initiator, config, result, self.id)
        table.insert(self.children, child_id)
    end

    setmetatable(new_process, new_process.metatable)

    self.processes[index] = new_process
    self.processes[index]:initiate(result)
    local msg = "Initiating "..module_slug..
      " process id "..index.." in org "..self.name

    return index
end

function modpol.orgs:delete_process(id)
    local process = self.processes[id]
    if process and process ~= "deleted" then
        -- recursively deletes any children
        if #process.children > 0 then
            for i, child_id in pairs(process.children) do
                self:delete_process(child_id)
            end
        end
        local msg = "Deleting " .. self.processes[id].slug .. " process id "..id.." in org "..self.name
        modpol.ocutil.log(msg)
        self:record(msg, self.processes[id].slug)
        self:wipe_pending_actions(id)
        -- sets process to 'deleted' in process table
        self.processes[id] = 'deleted'
    end
end

function modpol.orgs:add_pending_action(process_id, user, callback) 
    self.pending[user] = self.pending[user] or {}
    self.pending[user][process_id] = callback
    modpol.interactions.message(
       user, "New pending action in org "..self.name)
end

function modpol.orgs:remove_pending_action(process_id, user) 
    if self.pending[user] then
        self.pending[user][process_id] = nil
    end
end

function modpol.orgs:wipe_pending_actions(process_id)
    for user in pairs(self.pending) do
        self.pending[user][process_id] = nil
    end
end

function modpol.orgs:has_pending_actions(user)
    -- next() will return the next pair in a table
    -- if next() returns nil, the table is empty
    if not self.pending[user] then
        return false
    else
        if not next(self.pending[user]) then
            return false
        else 
            return true
        end
    end
end

function modpol.orgs:interact(process_id, user)
   local process = self.processes[process_id]
   if self.pending[user] then
      local callback = self.pending[user][process_id]
      if callback and process ~= "deleted" then
         -- get data in case callback ends process
         local slug = self.processes[process_id].slug
         -- run callback
         process[callback](process, user)
         -- record org data
         local msg = "Updating "..slug..
            " process id "..process_id.." in org "..self.name
         self:record(msg, slug)
      end
   end
end
