----------------------------
-- ALIASES -----------------
----------------------------
-- VECTOR ALIASES ----------
local vadd = vector.add
local vcopy = vector.copy
local vnew = vector.new
local voffset = vector.offset
local vrandom_direction = vector.random_direction
local vround = vector.round
-- CORE ALIASES ------------
local add_entity = core.add_entity
local get_node = core.get_node
local get_objects_in_area = core.get_objects_in_area
local registered_entities = core.registered_entities
local registered_nodes = core.registered_nodes
-- MATH ALIASES ------------
local ceil, floor = math.ceil, math.floor
local max, min = math.max, math.min
local random = math.random
-- TABLE ALIASES -----------
local tinsert = table.insert
local tsort = table.sort
-- LIBRARY ALIASES ---------
local ENTITIES_MAXIMUM = natural_entities.ENTITIES_MAXIMUM
local SPAWN_RATE = natural_entities.SPAWN_RATE
----------------------------
local storage = core.get_mod_storage()
----------------------------
-- FUNCTIONS ---------------
----------------------------
-- IS_PAUSED ---------------
---@return boolean paused Whether or not entity spawning is paused.
local function is_paused()
	return storage:get_int("paused") ~= 0
end
natural_entities.is_paused = is_paused
-- PAUSE -------------------
--- Pauses entity spawning.
local function pause()
	storage:set_int("paused", 1)
end
natural_entities.pause = pause
-- UNPAUSE -----------------
--- Unpauses entity spawning.
local function unpause()
	storage:set_int("paused", 0)
end
natural_entities.unpause = unpause
-- TOGGLE ------------------
--- Toggles entity spawning.
local function toggle()
	if is_paused() then
		unpause()
	else
		pause()
	end
end
natural_entities.toggle = toggle
-- GENERATE ENTITY ---------
local function generate_entity(entities)
	local temp = {}

	for name,chance in pairs(entities) do
		tinsert(temp, {name, chance * random()})
	end

	tsort(temp, function(a,b)
		return a[2] > b[2]
	end)
	
	return temp[1][1]
end
natural_entities.generate_entity = generate_entity
-- IS SPAWNABLE ------------
local function is_spawnable(pos)
	local node_name = get_node(pos).name
	local node_def = registered_nodes[node_name]
	return node_name ~= "ignore" and not node_def.walkable
end
natural_entities.is_spawnable = is_spawnable
-- ADJUST UNTIL SPAWNABLE --
local ADJUSTMENT_MAX_DISTANCE = 16
--- Tries to find a node an entity can spawn in near the provided position.
---@param origin vector The position to start from.
---@return boolean can_spawn Whether or not a spawnable node was found.
---@return vector pos The position the function left off at, whether or not a spawnable node was found.
local function adjust_until_spawnable(origin)
	local pos = vcopy(origin)
	local direction = vrandom_direction()
	local distance = 0
	
	repeat
		pos = vround(origin + direction * distance)
		if is_spawnable(pos) then return true, pos end
		distance = distance + 1
	until distance > ADJUSTMENT_MAX_DISTANCE
	
	return is_spawnable(pos), pos
end
natural_entities.adjust_until_spawnable = adjust_until_spawnable
-- DO SPAWN ----------------
--- Checks whether or not a spawn group should apply to the given position.
--- Does not check if the node at the position is spawnable.
---@param pos vector The position being checked.
---@param def table The definition table of the spawn group being checked.
---@return boolean applies `true` if spawn group applies, otherwise `false`.
local function spawn_group_applies(pos, def)
	local minp, maxp =
		voffset(pos, -256, def.min_y, -256), 
		voffset(pos, 256, def.max_y, 256)
	return
		-- spawn rate check
		random() * (def.spawn_rate or 1.0) > random()
		-- spawn group bounds check
		and pos.y > def.min_y - 32 and pos.y < def.max_y + 32
		-- entity count check
		and #get_objects_in_area(minp, maxp) >= ENTITIES_MAXIMUM
end
natural_entities.spawn_group_applies = spawn_group_applies
-- DO SPAWN ----------------
local MAXIMUM_ADJUSTMENTS = 16
local function do_spawn(pos, name, def)
	def = def or natural_entities.registered_spawns[name]

	if not spawn_group_applies(pos, def) then return end

	local min_y, max_y = def.min_y, def.max_y

	min_y = max(min_y, ceil(pos.y - 32))
	max_y = min(max_y, floor(pos.y + 32))
	pos = voffset(pos,
		random(4, 256) * random(-1, 1),
		random(min_y, max_y),
		random(4, 256) * random(-1, 1)
	)

	-- get entity
	local entities = def.entities or {}
	local ent_name = generate_entity(entities)
	local ent_def = ent_name and registered_entities[ent_name]
	
	if not ent_def then return end

	-- find a spawn position for the entity
	local can_spawn, pos2 = nil, pos
	local adjustments = 0

	repeat
		can_spawn, pos2 = adjust_until_spawnable(pos2)
		adjustments = adjustments + 1
		can_spawn = can_spawn and def.check(pos2, ent_name) ~= nil
	until can_spawn or adjustments > MAXIMUM_ADJUSTMENTS

	if not can_spawn then return end

	-- adjust for collisionbox
	local collisionbox = ent_def and ent_def.collisionbox
	pos2.y = pos2.y - (collisionbox and collisionbox[2] or 0)

	add_entity(pos2, ent_name)
end
natural_entities.do_spawn = do_spawn
-- DO SPAWNS FOR PLAYER ----
local function do_spawns_for_player(player)
	local spawns = natural_entities.registered_spawns
	for name, def in pairs(spawns) do
		do_spawn(vround(player:get_pos()), def)
	end
end
natural_entities.do_spawns_for_player = do_spawns_for_player
-- SPAWNSTEP ---------------
local function spawnstep(player)
	natural_entities.do_spawns_for_player(player)
end
natural_entities.spawnstep = spawnstep
----------------------------